"""Tests for Related Context Service (Repo Map).

This module tests the RelatedContextService for correctness
according to Property 11: Related Code Context Correctness.
"""

import tempfile
from pathlib import Path

import pytest
from hypothesis import given, settings, strategies as st

from core.services.related_context import (
    RelatedContextService,
    SignatureInfo,
    FileContext,
    RelatedCodeContext,
)
from core.storage.sqlite import SQLiteStorage
from core.scanner import CodeScanner
from core.graph import GraphBuilder


def scan_and_save_project(project_dir: Path, storage: SQLiteStorage):
    """Helper function to scan a project and save to storage."""
    scanner = CodeScanner(project_dir)
    files = scanner.scan()
    graph_builder = GraphBuilder(project_dir)
    graph_data = graph_builder.build(files)
    graph = graph_data.to_dict()
    storage.save_project(project_dir, files, graph)
    return files, graph


class TestRelatedContextServiceUnit:
    """Unit tests for RelatedContextService."""

    @pytest.fixture
    def temp_db(self, tmp_path):
        """Create a temporary database."""
        db_path = str(tmp_path / "test.db")
        storage = SQLiteStorage(db_path)
        yield storage
        storage.close()

    @pytest.fixture
    def sample_project(self, tmp_path):
        """Create a sample project with dependencies."""
        project_dir = tmp_path / "sample_project"
        project_dir.mkdir()

        # Create main.py that imports utils
        (project_dir / "main.py").write_text("""
from utils import helper_func

def main():
    result = helper_func("test")
    return result

if __name__ == "__main__":
    main()
""")

        # Create utils.py that imports config
        (project_dir / "utils.py").write_text("""
from config import DEFAULT_VALUE

def helper_func(input_str: str) -> str:
    \"\"\"Process input string.\"\"\"
    return input_str.upper() + str(DEFAULT_VALUE)

def another_helper(x: int, y: int) -> int:
    \"\"\"Add two numbers.\"\"\"
    return x + y
""")

        # Create config.py (no imports)
        (project_dir / "config.py").write_text("""
DEFAULT_VALUE = 42
MAX_SIZE = 100

def get_config() -> dict:
    \"\"\"Get configuration dictionary.\"\"\"
    return {
        "default": DEFAULT_VALUE,
        "max_size": MAX_SIZE
    }
""")

        # Create views.py that imports main
        (project_dir / "views.py").write_text("""
from main import main

def render_view():
    \"\"\"Render the main view.\"\"\"
    result = main()
    return f"<div>{result}</div>"

class ViewHandler:
    \"\"\"Handler for views.\"\"\"

    def handle(self, request):
        \"\"\"Handle a request.\"\"\"
        return render_view()
""")

        return project_dir

    def test_target_file_always_included(self, temp_db, sample_project):
        """Test that target file is always included in response.

        Property 11: Target file should always be included in response.
        """
        scan_and_save_project(sample_project, temp_db)

        project_path = str(sample_project.resolve())
        service = RelatedContextService(temp_db, sample_project)
        result = service.get_related_context(project_path, "main.py", hops=1)

        assert result.target_file is not None
        assert result.target_file.file_path == "main.py"
        assert result.target_file.relation == "target"
        assert result.target_file.hop_distance == 0

    def test_related_files_within_hop_distance(self, temp_db, sample_project):
        """Test that all related files are within specified hop distance.

        Property 11: All related files should be within the specified hop distance.
        """
        scan_and_save_project(sample_project, temp_db)

        project_path = str(sample_project.resolve())
        service = RelatedContextService(temp_db, sample_project)

        for hops in [1, 2, 3]:
            result = service.get_related_context(project_path, "main.py", hops=hops)

            for related_file in result.related_files:
                assert related_file.hop_distance <= hops, (
                    f"File {related_file.file_path} has hop distance "
                    f"{related_file.hop_distance} but max is {hops}"
                )

    def test_no_duplicate_files(self, temp_db, sample_project):
        """Test that no duplicate files appear in response.

        Property 11: No duplicate files should appear in the response.
        """
        scan_and_save_project(sample_project, temp_db)

        project_path = str(sample_project.resolve())
        service = RelatedContextService(temp_db, sample_project)
        result = service.get_related_context(project_path, "main.py", hops=3)

        all_files = [result.target_file.file_path]
        all_files.extend([f.file_path for f in result.related_files])

        assert len(all_files) == len(set(all_files)), "Duplicate files found"

    def test_result_count_respects_max_files(self, temp_db, tmp_path):
        """Test that result count respects MAX_FILES limit.

        Property 11: Result count should be <= MAX_FILES (50).
        """
        # Create many files with dependencies
        project_dir = tmp_path / "large_project"
        project_dir.mkdir()

        # Create hub file that many files import
        (project_dir / "hub.py").write_text("""
def hub_function():
    return "hub"
""")

        # Create many files that import the hub
        for i in range(60):
            (project_dir / f"module_{i}.py").write_text(f"""
from hub import hub_function

def func_{i}():
    return hub_function()
""")

        scan_and_save_project(project_dir, temp_db)

        project_path = str(project_dir.resolve())
        service = RelatedContextService(temp_db, project_dir)
        result = service.get_related_context(project_path, "hub.py", hops=2)

        assert len(result.related_files) <= service.MAX_FILES
        if len(result.related_files) == service.MAX_FILES:
            assert result.truncated is True

    def test_signatures_extracted_not_implementations(self, temp_db, sample_project):
        """Test that file context contains only signatures, not full implementations.

        Property 11: Each file context should contain only signatures, not full implementations.
        """
        scan_and_save_project(sample_project, temp_db)

        project_path = str(sample_project.resolve())
        service = RelatedContextService(temp_db, sample_project)
        result = service.get_related_context(project_path, "utils.py", hops=1)

        # Check target file signatures
        assert len(result.target_file.signatures) > 0, "Should have signatures"

        for sig in result.target_file.signatures:
            # Signatures should not contain function body
            assert "{" not in sig.signature or "=>" in sig.signature, (
                f"Signature should not contain body: {sig.signature}"
            )
            # Signatures should be reasonably short (not full implementations)
            assert len(sig.signature.split('\n')) <= 5, (
                f"Signature too long, might contain body: {sig.signature}"
            )

    def test_hop_parameter_clamped(self, temp_db, sample_project):
        """Test that hops parameter is clamped between 1 and MAX_HOPS."""
        scan_and_save_project(sample_project, temp_db)

        project_path = str(sample_project.resolve())
        service = RelatedContextService(temp_db, sample_project)

        # Test with hops = 0 (should be clamped to 1)
        result_0 = service.get_related_context(project_path, "main.py", hops=0)
        result_1 = service.get_related_context(project_path, "main.py", hops=1)
        assert len(result_0.related_files) == len(result_1.related_files)

        # Test with hops = 10 (should be clamped to MAX_HOPS)
        result_10 = service.get_related_context(project_path, "main.py", hops=10)
        result_max = service.get_related_context(
            project_path, "main.py", hops=service.MAX_HOPS
        )
        assert len(result_10.related_files) == len(result_max.related_files)

    def test_relation_types_correct(self, temp_db, sample_project):
        """Test that relation types are set correctly."""
        scan_and_save_project(sample_project, temp_db)

        project_path = str(sample_project.resolve())
        service = RelatedContextService(temp_db, sample_project)
        result = service.get_related_context(project_path, "utils.py", hops=2)

        assert result.target_file.relation == "target"

        valid_relations = {"imports", "imported_by"}
        for related_file in result.related_files:
            assert related_file.relation in valid_relations, (
                f"Invalid relation: {related_file.relation}"
            )

    def test_project_not_found(self, temp_db, sample_project):
        """Test handling of non-existent project."""
        service = RelatedContextService(temp_db, sample_project)
        result = service.get_related_context("/nonexistent/path", "main.py", hops=1)

        assert result.target_file.file_path == "main.py"
        assert "not found" in result.message.lower()

    def test_file_not_found(self, temp_db, sample_project):
        """Test handling of non-existent file."""
        scan_and_save_project(sample_project, temp_db)

        project_path = str(sample_project.resolve())
        service = RelatedContextService(temp_db, sample_project)
        result = service.get_related_context(
            project_path, "nonexistent.py", hops=1
        )

        assert result.target_file.file_path == "nonexistent.py"
        assert "not found" in result.message.lower()

    def test_to_dict_serialization(self, temp_db, sample_project):
        """Test that to_dict produces valid serializable output."""
        scan_and_save_project(sample_project, temp_db)

        project_path = str(sample_project.resolve())
        service = RelatedContextService(temp_db, sample_project)
        result = service.get_related_context(project_path, "main.py", hops=2)
        serialized = service.to_dict(result)

        assert "target_file" in serialized
        assert "related_files" in serialized
        assert "total_files" in serialized
        assert "truncated" in serialized

        # Check target file structure
        assert "file_path" in serialized["target_file"]
        assert "signatures" in serialized["target_file"]
        assert "relation" in serialized["target_file"]

        # Check signatures structure
        for sig in serialized["target_file"]["signatures"]:
            assert "name" in sig
            assert "type" in sig
            assert "signature" in sig
            assert "line" in sig


class TestRelatedContextServiceProperty:
    """Property-based tests for RelatedContextService."""

    @given(
        num_files=st.integers(min_value=2, max_value=20),
        hops=st.integers(min_value=1, max_value=3)
    )
    @settings(max_examples=30, deadline=None)
    def test_no_duplicates_in_results(self, num_files, hops, tmp_path_factory):
        """Property 11: No duplicate files should appear in the response.

        Feature: code-knowledge-graph-enhancement
        Property 11: Related Code Context Correctness
        """
        tmp_path = tmp_path_factory.mktemp("project")
        project_dir = tmp_path / "test_project"
        project_dir.mkdir()

        # Create files with chain dependencies
        for i in range(num_files):
            if i == 0:
                content = "def func_0(): pass"
            else:
                content = f"from file_{i-1} import func_{i-1}\ndef func_{i}(): pass"
            (project_dir / f"file_{i}.py").write_text(content)

        # Create storage and scan
        db_path = str(tmp_path / "test.db")
        storage = SQLiteStorage(db_path)

        scan_and_save_project(project_dir, storage)

        project_path = str(project_dir.resolve())
        service = RelatedContextService(storage, project_dir)
        result = service.get_related_context(project_path, "file_0.py", hops=hops)

        # Collect all file paths
        all_paths = [result.target_file.file_path]
        all_paths.extend([f.file_path for f in result.related_files])

        # Check for duplicates
        assert len(all_paths) == len(set(all_paths)), (
            f"Duplicate files found: {all_paths}"
        )

        storage.close()

    @given(
        hops=st.integers(min_value=1, max_value=3)
    )
    @settings(max_examples=20, deadline=None)
    def test_hop_distance_constraint(self, hops, tmp_path_factory):
        """Property 11: All related files should be within the specified hop distance.

        Feature: code-knowledge-graph-enhancement
        Property 11: Related Code Context Correctness
        """
        tmp_path = tmp_path_factory.mktemp("project")
        project_dir = tmp_path / "test_project"
        project_dir.mkdir()

        # Create a simple dependency chain
        (project_dir / "a.py").write_text("def a(): pass")
        (project_dir / "b.py").write_text("from a import a\ndef b(): pass")
        (project_dir / "c.py").write_text("from b import b\ndef c(): pass")
        (project_dir / "d.py").write_text("from c import c\ndef d(): pass")

        db_path = str(tmp_path / "test.db")
        storage = SQLiteStorage(db_path)

        scan_and_save_project(project_dir, storage)

        project_path = str(project_dir.resolve())
        service = RelatedContextService(storage, project_dir)
        result = service.get_related_context(project_path, "a.py", hops=hops)

        # All related files should have hop_distance <= hops
        for related_file in result.related_files:
            assert related_file.hop_distance <= hops, (
                f"File {related_file.file_path} has hop_distance "
                f"{related_file.hop_distance} > {hops}"
            )

        storage.close()

    @given(
        target_idx=st.integers(min_value=0, max_value=4)
    )
    @settings(max_examples=10, deadline=None)
    def test_target_always_included(self, target_idx, tmp_path_factory):
        """Property 11: Target file should always be included in response.

        Feature: code-knowledge-graph-enhancement
        Property 11: Related Code Context Correctness
        """
        tmp_path = tmp_path_factory.mktemp("project")
        project_dir = tmp_path / "test_project"
        project_dir.mkdir()

        # Create 5 files
        filenames = ["alpha.py", "beta.py", "gamma.py", "delta.py", "epsilon.py"]
        for i, name in enumerate(filenames):
            if i == 0:
                content = "def func(): pass"
            else:
                content = f"from {filenames[i-1][:-3]} import func\ndef func(): pass"
            (project_dir / name).write_text(content)

        db_path = str(tmp_path / "test.db")
        storage = SQLiteStorage(db_path)

        scan_and_save_project(project_dir, storage)

        project_path = str(project_dir.resolve())
        service = RelatedContextService(storage, project_dir)
        target_file = filenames[target_idx]
        result = service.get_related_context(project_path, target_file, hops=2)

        # Target file should always be present
        assert result.target_file is not None
        assert result.target_file.file_path == target_file
        assert result.target_file.relation == "target"
        assert result.target_file.hop_distance == 0

        storage.close()

    @given(
        num_files=st.integers(min_value=2, max_value=60)
    )
    @settings(max_examples=10, deadline=None)
    def test_max_files_limit(self, num_files, tmp_path_factory):
        """Property 11: Result count should be <= MAX_FILES (50).

        Feature: code-knowledge-graph-enhancement
        Property 11: Related Code Context Correctness
        """
        tmp_path = tmp_path_factory.mktemp("project")
        project_dir = tmp_path / "test_project"
        project_dir.mkdir()

        # Create a long chain of dependencies that will be resolved
        # file_0 -> file_1 -> file_2 -> ... -> file_n
        for i in range(num_files):
            if i == 0:
                content = "def func_0(): pass"
            else:
                content = f"from file_{i-1} import func_{i-1}\ndef func_{i}(): pass"
            (project_dir / f"file_{i}.py").write_text(content)

        db_path = str(tmp_path / "test.db")
        storage = SQLiteStorage(db_path)

        scan_and_save_project(project_dir, storage)

        project_path = str(project_dir.resolve())
        service = RelatedContextService(storage, project_dir)

        # Query with max hops from middle of chain to potentially get many files
        middle_idx = num_files // 2
        result = service.get_related_context(
            project_path, f"file_{middle_idx}.py", hops=3
        )

        # Core property: result should always respect MAX_FILES limit
        assert len(result.related_files) <= service.MAX_FILES

        # If we have MAX_FILES related files, truncated should be True
        if len(result.related_files) == service.MAX_FILES:
            assert result.truncated is True

        storage.close()


class TestSignatureExtraction:
    """Tests for signature extraction functionality."""

    @pytest.fixture
    def temp_db(self, tmp_path):
        """Create a temporary database."""
        db_path = str(tmp_path / "test.db")
        storage = SQLiteStorage(db_path)
        yield storage
        storage.close()

    def test_python_function_signatures(self, temp_db, tmp_path):
        """Test extraction of Python function signatures."""
        project_dir = tmp_path / "py_project"
        project_dir.mkdir()

        (project_dir / "funcs.py").write_text('''
def simple_func(x):
    """Simple function."""
    return x * 2

async def async_func(a: int, b: str) -> dict:
    """Async function with type hints."""
    return {"a": a, "b": b}

def multi_line_func(
    param1: int,
    param2: str,
    param3: list
) -> tuple:
    """Function with multi-line signature."""
    return (param1, param2, param3)
''')

        scan_and_save_project(project_dir, temp_db)

        project_path = str(project_dir.resolve())
        service = RelatedContextService(temp_db, project_dir)
        result = service.get_related_context(project_path, "funcs.py", hops=1)

        # Check signatures were extracted
        func_names = {sig.name for sig in result.target_file.signatures}
        assert "simple_func" in func_names
        assert "async_func" in func_names
        assert "multi_line_func" in func_names

        # Check signature types
        for sig in result.target_file.signatures:
            assert sig.entity_type == "function"
            assert sig.line > 0

    def test_python_class_signatures(self, temp_db, tmp_path):
        """Test extraction of Python class signatures."""
        project_dir = tmp_path / "py_project"
        project_dir.mkdir()

        (project_dir / "classes.py").write_text('''
class SimpleClass:
    """A simple class."""

    def method_one(self):
        """First method."""
        pass

    def method_two(self, x: int) -> str:
        """Second method."""
        return str(x)

class ChildClass(SimpleClass):
    """A child class."""

    def child_method(self):
        """Child method."""
        pass
''')

        scan_and_save_project(project_dir, temp_db)

        project_path = str(project_dir.resolve())
        service = RelatedContextService(temp_db, project_dir)
        result = service.get_related_context(project_path, "classes.py", hops=1)

        # Check class and method signatures
        sig_names = {sig.name for sig in result.target_file.signatures}
        assert "SimpleClass" in sig_names
        assert "ChildClass" in sig_names
        assert "SimpleClass.method_one" in sig_names
        assert "SimpleClass.method_two" in sig_names
        assert "ChildClass.child_method" in sig_names

    def test_javascript_signatures(self, temp_db, tmp_path):
        """Test extraction of JavaScript function and class signatures."""
        project_dir = tmp_path / "js_project"
        project_dir.mkdir()

        (project_dir / "funcs.js").write_text('''
function regularFunction(a, b) {
    return a + b;
}

const arrowFunc = (x) => {
    return x * 2;
};

async function asyncFunction(data) {
    return await fetch(data);
}

class JsClass {
    constructor(value) {
        this.value = value;
    }

    getValue() {
        return this.value;
    }
}
''')

        scan_and_save_project(project_dir, temp_db)

        project_path = str(project_dir.resolve())
        service = RelatedContextService(temp_db, project_dir)
        result = service.get_related_context(project_path, "funcs.js", hops=1)

        # Check signatures were extracted
        sig_names = {sig.name for sig in result.target_file.signatures}
        assert "regularFunction" in sig_names
        assert "arrowFunc" in sig_names
        assert "asyncFunction" in sig_names
        assert "JsClass" in sig_names
