"""
License GPLv3 or higher.

(C) 2025 Created by Maikel Mardjan - https://nocomplexity.com/

This program is free software: you can redistribute it and/or modify it under the terms of the GNU General Public License as published by the Free Software Foundation, either version 3 of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for more details.

You should have received a copy of the GNU General Public License along with this program. If not, see <https://www.gnu.org/licenses/>. 


CLI functions for codeaudit
"""
import fire  # for working CLI with this PoC-thing (The Google way)
import sys
from codeaudit import __version__
from codeaudit.reporting import overview_report ,report_module_information , scan_report , report_implemented_tests

codeaudit_ascii_art=r"""
----------------------------------------------------
 _                    __             _             
|_) \/_|_|_  _ __    /   _  _| _    |_|    _| o _|_
|   /  |_| |(_)| |   \__(_)(_|(/_   | ||_|(_| |  |_
----------------------------------------------------
"""

     
def display_version():
    """Prints the module version. Or use codeaudit [-v] [--v] [-version] or [--version]."""
    print(f"version: {__version__}")


def display_help():
    """Shows detailed help for using codeaudit tool."""    
    print(codeaudit_ascii_art)
    print("Python Code Audit - A modern Python security source code analyzer based on distrust.\n")
    print("Commands to evaluate Python source code:")
    print('Usage: codeaudit COMMAND <directory|package>  [report.html] \n')
    print('Depending on the command, you must specify a local directory, a Python file, or a package name hosted on PyPI.org.Reporting: The results are generated as a static HTML report for viewing in a web browser.\n')
    print('Commands:')
    commands = ["overview", "filescan", "modulescan",  "checks","version"]  # commands on CLI
    functions = [overview_report, scan_report, report_module_information, report_implemented_tests,display_version]  # Related functions relevant for help
    for command, function in zip(commands, functions):
        docstring = function.__doc__.strip().split('\n')[0] or ""  
        summary = docstring.split("\n", 1)[0]
        print(f"  {command:<20} {summary}")        
    print("\nUse the Python Code Audit documentation (https://codeaudit.nocomplexity.com) to audit and secure your Python programmes. Explore further essential open-source security tools at https://simplifysecurity.nocomplexity.com/\n")

def main():
    if "-?" in sys.argv:      # Normalize help flags BEFORE Fire sees them: fire module treats anything starting with - as a flag/value, not as a help alias.
        sys.argv[sys.argv.index("-?")] = "--help"
    if "-help" in sys.argv:      # Normalize help flags BEFORE Fire sees them
        sys.argv[sys.argv.index("-help")] = "--help"        
    if len(sys.argv) > 1 and sys.argv[1] in ("-v", "--v", "--version", "-version"):
        display_version()
    elif len(sys.argv) > 1 and sys.argv[1] in ("-help", "--help", "-h"):
        display_help()
    elif len(sys.argv) == 1:
        display_help()
    else:
        fire.Fire(
            {
                "overview": overview_report,
                "modulescan": report_module_information,
                "filescan": scan_report,
                "checks": report_implemented_tests,
                "version": display_version,
            }
        )



if __name__ == "__main__":
    main()

                                            
                                                 

