"""Base client implementation for CodeMie SDK."""

from typing import Optional

from ..auth.credentials import KeycloakCredentials
from ..services.admin import AdminService
from ..services.analytics import AnalyticsService
from ..services.assistant import AssistantService
from ..services.conversation import ConversationService
from ..services.datasource import DatasourceService
from ..services.llm import LLMService
from ..services.integration import IntegrationService
from ..services.task import TaskService
from ..services.user import UserService
from ..services.workflow import WorkflowService
from ..services.files import FileOperationService
from ..services.webhook import WebhookService
from ..services.vendor_assistant import VendorAssistantService
from ..services.vendor_workflow import VendorWorkflowService
from ..services.vendor_knowledgebase import VendorKnowledgeBaseService
from ..services.vendor_guardrail import VendorGuardrailService
from ..services.codemie_guardrails import CodemieGuardrailService


class CodeMieClient:
    """Main client class for interacting with CodeMie API."""

    def __init__(
        self,
        auth_server_url: str,
        auth_realm_name: str,
        codemie_api_domain: str,
        auth_client_id: Optional[str] = None,
        auth_client_secret: Optional[str] = None,
        username: Optional[str] = None,
        password: Optional[str] = None,
        verify_ssl: bool = True,
    ):
        """Initialize CodeMie client with authentication credentials.

        Args:
            auth_server_url: Keycloak server URL
            auth_realm_name: Realm name for authentication
            codemie_api_domain: CodeMie API domain
            auth_client_id: Client ID for authentication (optional if using username/password)
            auth_client_secret: Client secret for authentication (optional if using username/password)
            username: Username/email for password grant (optional if using client credentials)
            password: Password for password grant (optional if using client credentials)
            verify_ssl: Whether to verify SSL certificates (default: True)
        """
        self.auth = KeycloakCredentials(
            server_url=auth_server_url,
            realm_name=auth_realm_name,
            client_id=auth_client_id,
            client_secret=auth_client_secret,
            username=username,
            password=password,
            verify_ssl=verify_ssl,
        )

        self._token: Optional[str] = None
        self._api_domain = codemie_api_domain.rstrip("/")
        self._is_localhost = self._is_localhost_domain(self._api_domain)
        self._verify_ssl = verify_ssl
        if not verify_ssl:
            import requests
            from urllib3.exceptions import InsecureRequestWarning

            requests.packages.urllib3.disable_warnings(InsecureRequestWarning)

        # Initialize token
        self._token = "" if self._is_localhost else self.auth.get_token()

        # Initialize services with verify_ssl parameter and token
        self.admin = AdminService(self._api_domain, self._token, verify_ssl=verify_ssl)
        self.analytics = AnalyticsService(
            self._api_domain, self._token, verify_ssl=verify_ssl
        )
        self.assistants = AssistantService(
            self._api_domain, self._token, verify_ssl=verify_ssl
        )
        self.llms = LLMService(self._api_domain, self._token, verify_ssl=verify_ssl)
        self.integrations = IntegrationService(
            self._api_domain, self._token, verify_ssl=verify_ssl
        )
        self.tasks = TaskService(self._api_domain, self._token, verify_ssl=verify_ssl)
        self.users = UserService(self._api_domain, self._token, verify_ssl=verify_ssl)
        self.datasources = DatasourceService(
            self._api_domain, self._token, verify_ssl=verify_ssl
        )
        self.workflows = WorkflowService(
            self._api_domain, self._token, verify_ssl=self._verify_ssl
        )
        self.conversations = ConversationService(
            self._api_domain, self._token, verify_ssl=self._verify_ssl
        )
        self.files = FileOperationService(
            self._api_domain, self._token, verify_ssl=self._verify_ssl
        )
        self.webhook = WebhookService(
            self._api_domain, self._token, verify_ssl=self._verify_ssl
        )
        self.vendor_assistants = VendorAssistantService(
            self._api_domain, self._token, verify_ssl=self._verify_ssl
        )
        self.vendor_workflows = VendorWorkflowService(
            self._api_domain, self._token, verify_ssl=self._verify_ssl
        )
        self.vendor_knowledgebases = VendorKnowledgeBaseService(
            self._api_domain, self._token, verify_ssl=self._verify_ssl
        )
        self.vendor_guardrails = VendorGuardrailService(
            self._api_domain, self._token, verify_ssl=self._verify_ssl
        )
        self.codemie_guardrails = CodemieGuardrailService(
            self._api_domain, self._token, verify_ssl=self._verify_ssl
        )

    @property
    def token(self) -> str:
        """Get current token or fetch new one if not available."""
        self._token = "" if self._is_localhost else self.auth.get_token()
        return self._token

    @staticmethod
    def _is_localhost_domain(domain: str) -> bool:
        """Check if the domain is a localhost variant."""
        domain_lower = domain.lower()
        localhost_patterns = [
            "localhost",
            "127.0.0.1",
            "0.0.0.0",
            "192.168",
        ]
        return any(pattern in domain_lower for pattern in localhost_patterns)

    def refresh_token(self) -> str:
        """Force token refresh."""
        self._token = "" if self._is_localhost else self.auth.get_token()
        # Update token in services
        self.admin = AdminService(
            self._api_domain, self._token, verify_ssl=self._verify_ssl
        )
        self.analytics = AnalyticsService(
            self._api_domain, self._token, verify_ssl=self._verify_ssl
        )
        self.assistants = AssistantService(
            self._api_domain, self._token, verify_ssl=self._verify_ssl
        )
        self.llms = LLMService(
            self._api_domain, self._token, verify_ssl=self._verify_ssl
        )
        self.integrations = IntegrationService(
            self._api_domain, self._token, verify_ssl=self._verify_ssl
        )
        self.tasks = TaskService(
            self._api_domain, self._token, verify_ssl=self._verify_ssl
        )
        self.users = UserService(
            self._api_domain, self._token, verify_ssl=self._verify_ssl
        )
        self.datasources = DatasourceService(
            self._api_domain, self._token, verify_ssl=self._verify_ssl
        )
        self.workflows = WorkflowService(
            self._api_domain, self._token, verify_ssl=self._verify_ssl
        )
        self.conversations = ConversationService(
            self._api_domain, self._token, verify_ssl=self._verify_ssl
        )
        self.files = FileOperationService(
            self._api_domain, self._token, verify_ssl=self._verify_ssl
        )
        self.webhook = WebhookService(
            self._api_domain, self._token, verify_ssl=self._verify_ssl
        )
        self.vendor_assistants = VendorAssistantService(
            self._api_domain, self._token, verify_ssl=self._verify_ssl
        )
        self.vendor_workflows = VendorWorkflowService(
            self._api_domain, self._token, verify_ssl=self._verify_ssl
        )
        self.vendor_knowledgebases = VendorKnowledgeBaseService(
            self._api_domain, self._token, verify_ssl=self._verify_ssl
        )
        self.vendor_guardrails = VendorGuardrailService(
            self._api_domain, self._token, verify_ssl=self._verify_ssl
        )
        self.codemie_guardrails = CodemieGuardrailService(
            self._api_domain, self._token, verify_ssl=self._verify_ssl
        )
        return self._token
