import importlib.resources
import logging
import logging.config
import os
import traceback
from typing import Optional

import yaml
from liquid2.exceptions import TemplateNotFoundError
from requests.exceptions import RequestException

import codeplain_REST_api as codeplain_api
import file_utils
import plain_file
import plain_spec
from event_bus import EventBus
from module_renderer import ModuleRenderer
from plain2code_arguments import parse_arguments
from plain2code_console import console
from plain2code_exceptions import PlainSyntaxError
from plain2code_logger import (
    CrashLogHandler,
    IndentedFormatter,
    RetryOnlyFilter,
    TuiLoggingHandler,
    dump_crash_logs,
    get_log_file_path,
)
from plain2code_state import RunState
from system_config import system_config
from tui.plain2code_tui import Plain2CodeTUI

TEST_SCRIPT_EXECUTION_TIMEOUT = 120  # 120 seconds

DEFAULT_TEMPLATE_DIRS = importlib.resources.files("standard_template_library")

MAX_UNITTEST_FIX_ATTEMPTS = 20
MAX_CONFORMANCE_TEST_FIX_ATTEMPTS = 20
MAX_CONFORMANCE_TEST_RUNS = 20
MAX_REFACTORING_ITERATIONS = 5
MAX_UNIT_TEST_RENDER_RETRIES = 2

MAX_ISSUE_LENGTH = 10000  # Characters.

UNRECOVERABLE_ERROR_EXIT_CODES = [69]
TIMEOUT_ERROR_EXIT_CODE = 124


class InvalidFridArgument(Exception):
    pass


def get_render_range(render_range, plain_source):
    render_range = render_range.split(",")
    range_end = render_range[1] if len(render_range) == 2 else render_range[0]

    return _get_frids_range(plain_source, render_range[0], range_end)


def get_render_range_from(start, plain_source):
    return _get_frids_range(plain_source, start)


def _get_frids_range(plain_source, start, end=None):
    frids = list(plain_spec.get_frids(plain_source))

    start = str(start)

    if start not in frids:
        raise InvalidFridArgument(f"Invalid start functional requirement ID: {start}. Valid IDs are: {frids}.")

    if end is not None:
        end = str(end)
        if end not in frids:
            raise InvalidFridArgument(f"Invalid end functional requirement ID: {end}. Valid IDs are: {frids}.")

        end_idx = frids.index(end) + 1
    else:
        end_idx = len(frids)

    start_idx = frids.index(start)
    if start_idx >= end_idx:
        raise InvalidFridArgument(
            f"Start functional requirement ID: {start} must be before end functional requirement ID: {end}."
        )

    return frids[start_idx:end_idx]


def setup_logging(
    args,
    event_bus: EventBus,
    log_to_file: bool,
    log_file_name: str,
    plain_file_path: Optional[str],
):
    # Set default level to INFO for everything not explicitly configured
    logging.getLogger().setLevel(logging.INFO)
    logging.getLogger("urllib3").setLevel(logging.WARNING)
    logging.getLogger("httpx").setLevel(logging.WARNING)
    logging.getLogger("httpcore").setLevel(logging.WARNING)
    logging.getLogger("anthropic").setLevel(logging.WARNING)
    logging.getLogger("langsmith").setLevel(logging.WARNING)
    logging.getLogger("git").setLevel(logging.WARNING)
    logging.getLogger("anthropic._base_client").setLevel(logging.WARNING)
    logging.getLogger("services.langsmith.langsmith_service").setLevel(logging.WARNING)
    logging.getLogger("repositories").setLevel(logging.WARNING)
    logging.getLogger("transitions").setLevel(logging.ERROR)
    logging.getLogger("transitions.extensions.diagrams").setLevel(logging.ERROR)

    log_file_path = get_log_file_path(plain_file_path, log_file_name)

    # Try to load logging configuration from YAML file
    if args.logging_config_path and os.path.exists(args.logging_config_path):
        try:
            with open(args.logging_config_path, "r") as f:
                config = yaml.safe_load(f)
                logging.config.dictConfig(config)
                console.info(f"Loaded logging configuration from {args.logging_config_path}")
        except Exception as e:
            console.warning(f"Failed to load logging configuration from {args.logging_config_path}: {str(e)}")

    # Allow detailed retry logs for anthropic if needed
    logging.getLogger("anthropic._base_client").setLevel(logging.DEBUG)
    if logging.getLogger("anthropic._base_client").level == logging.DEBUG:
        logging.getLogger("anthropic._base_client").addFilter(RetryOnlyFilter())

    # The IndentedFormatter provides better multiline log readability.
    # We add the TuiLoggingHandler to the root logger.
    # CRITICAL: We must remove existing handlers (like StreamHandler) to prevent double-logging
    # that spills into the TUI dashboard.
    root_logger = logging.getLogger()
    for h in root_logger.handlers[:]:
        root_logger.removeHandler(h)

    handler = TuiLoggingHandler(event_bus)
    formatter = IndentedFormatter("%(levelname)s:%(name)s:%(message)s")
    handler.setFormatter(formatter)
    root_logger.addHandler(handler)

    if log_to_file and log_file_path:
        try:
            file_handler = logging.FileHandler(log_file_path, mode="w")
            file_handler.setFormatter(formatter)
            root_logger.addHandler(file_handler)
        except Exception as e:
            console.warning(f"Failed to setup file logging to {log_file_path}: {str(e)}")
    else:
        # If file logging is disabled, use CrashLogHandler to buffer logs in memory
        # in case we need to dump them on crash.
        crash_handler = CrashLogHandler()
        crash_handler.setFormatter(formatter)
        root_logger.addHandler(crash_handler)


def render(args, run_state: RunState, codeplain_api, event_bus: EventBus):  # noqa: C901
    # Check system requirements before proceeding
    system_config.verify_requirements()

    template_dirs = file_utils.get_template_directories(args.filename, args.template_dir, DEFAULT_TEMPLATE_DIRS)

    console.info(f"Rendering {args.filename} to target code.")

    codeplainAPI = codeplain_api.CodeplainAPI(args.api_key, console)
    codeplainAPI.verbose = args.verbose
    assert args.api is not None and args.api != "", "API URL is required"
    codeplainAPI.api_url = args.api

    module_renderer = ModuleRenderer(
        codeplainAPI,
        args.filename,
        args.render_range,
        template_dirs,
        args,
        run_state,
        event_bus,
    )

    app = Plain2CodeTUI(
        event_bus=event_bus,
        worker_fun=module_renderer.render_module,
        render_id=run_state.render_id,
        unittests_script=args.unittests_script,
        conformance_tests_script=args.conformance_tests_script,
        prepare_environment_script=args.prepare_environment_script,
        css_path="styles.css",
    )
    result = app.run()

    # If the app exited due to a worker error, re-raise it here
    # so it hits the exception handlers in main()
    if isinstance(result, Exception):
        raise result

    return


def main():
    args = parse_arguments()

    event_bus = EventBus()

    setup_logging(args, event_bus, args.log_to_file, args.log_file_name, args.filename)

    if not args.api:
        args.api = "https://api.codeplain.ai"

    run_state = RunState(spec_filename=args.filename, replay_with=args.replay_with)

    try:
        render(args, run_state, codeplain_api, event_bus)
    except InvalidFridArgument as e:
        console.error(f"Error rendering plain code: {str(e)}.\n")
        # No need to print render ID since this error is going to be thrown at the very start so user will be able to
        # see the render ID that's printed at the very start of the rendering process.
        dump_crash_logs(args)
    except FileNotFoundError as e:
        console.error(f"Error rendering plain code: {str(e)}\n")
        console.debug(f"Render ID: {run_state.render_id}")
        dump_crash_logs(args)
    except plain_file.InvalidPlainFileExtension as e:
        console.error(f"Error rendering plain code: {str(e)}\n")
        dump_crash_logs(args)
    except TemplateNotFoundError as e:
        console.error(f"Error: Template not found: {str(e)}\n")
        console.error(system_config.get_error_message("template_not_found"))
        dump_crash_logs(args)
    except PlainSyntaxError as e:
        console.error(f"Error rendering plain code: {str(e)}\n")
        dump_crash_logs(args)
    except KeyboardInterrupt:
        console.error("Keyboard interrupt")
        # Don't print the traceback here because it's going to be from keyboard interrupt and we don't really care about that
        console.debug(f"Render ID: {run_state.render_id}")
        dump_crash_logs(args)
    except RequestException as e:
        console.error(f"Error rendering plain code: {str(e)}\n")
        console.debug(f"Render ID: {run_state.render_id}")
        dump_crash_logs(args)
    except Exception as e:
        console.error(f"Error rendering plain code: {str(e)}\n")
        console.debug(f"Render ID: {run_state.render_id}")
        traceback.print_exc()
        dump_crash_logs(args)


if __name__ == "__main__":  # noqa: C901
    main()
