from pathlib import Path
from string import Template

from colcon_core.plugin_system import satisfies_version
from colcon_core.prefix_path import get_chained_prefix_path
from colcon_core.shell import ShellExtensionPoint, logger


def expand_py_template(src, dest, substitutions, *, is_entry_point=False):
    """
    Expand a template to a file.
    :param src: template to be expanded. $TOKEN or ${TOKEN} will be replaced as per string.Template
    :param dest: path to the output file
    :param substitutions: mapping from tokens to their values. NOTE, the values will be repr'd so only use simple
            objects with eval'able reprs
    :param is_entry_point: whether this is expected to be user-callable
    :return: None
    """
    template_text = Path(src).read_text()
    template = Template(template_text)
    result = template.substitute({k: repr(v) for k, v in substitutions.items()})

    Path(dest).parent.mkdir(parents=True, exist_ok=True)

    with Path(dest).open('w') as f:
        if is_entry_point:
            f.write('#! /usr/bin/env python3 -i\n')
        f.write('# template generated by {}\n'.format(__file__))
        f.write(result)
    if is_entry_point:
        Path(dest).chmod(0o755)


STEM_SCRIPT_PACKAGE = 'package'
STEM_SCRIPT_PREFIX = 'local_setup'
STEM_SCRIPT_PREFIX_CHAIN = 'setup_chain'
STEM_HOOK_PREPEND = 'hook_prepend_path'
EXT_SCRIPT = '.py'
PATH_TEMPLATES = Path(__file__).parent / 'template'
EXT_TEMPLATE = '.pytemplate'


class PythonShell(ShellExtensionPoint):
    """Generate `.py` scripts to extend the environment."""

    # Needs to be > ShellExtensionPoint.PRIORITY for colcon to call `create_hook_...` functions
    PRIORITY = 200

    def __init__(self):
        super().__init__()
        satisfies_version(ShellExtensionPoint.EXTENSION_POINT_VERSION, '^2.0')

    def create_package_script(self, prefix_path, pkg_name, hooks):
        out_path = (prefix_path / 'share' / pkg_name / STEM_SCRIPT_PACKAGE).with_suffix(EXT_SCRIPT)

        logger.info("Creating package script '{}'".format(out_path))

        template_path = (PATH_TEMPLATES / STEM_SCRIPT_PACKAGE).with_suffix(EXT_TEMPLATE)
        expand_py_template(template_path,
                           out_path,
                           dict(
                               pkg_name=pkg_name,
                               hooks=[hook for hook in hooks if hook[0].endswith(EXT_SCRIPT)]
                           ))

    def create_prefix_script(self, prefix_path, merge_install):
        out_path1 = (prefix_path / STEM_SCRIPT_PREFIX).with_suffix(EXT_SCRIPT)

        logger.info("Creating prefix script '{}'".format(out_path1))
        expand_py_template(PATH_TEMPLATES / (STEM_SCRIPT_PREFIX + EXT_TEMPLATE),
                           out_path1,
                           dict(
                               prefix=str(prefix_path),
                               package_script_name=STEM_SCRIPT_PACKAGE + EXT_SCRIPT,
                           ),
                           is_entry_point=True,
                           )

        out_path2 = (prefix_path / STEM_SCRIPT_PREFIX_CHAIN).with_suffix(EXT_SCRIPT)
        logger.info(
            "Creating prefix chain script '{}'".format(out_path2))
        expand_py_template(
            PATH_TEMPLATES / (STEM_SCRIPT_PREFIX_CHAIN + EXT_TEMPLATE),
            out_path2,
            dict(prefixes=get_chained_prefix_path(skip=prefix_path),
                 prefix_script_name=STEM_SCRIPT_PREFIX + EXT_SCRIPT),
            is_entry_point=True,
        )

    def create_hook_prepend_value(
            self, env_hook_name, prefix_path, pkg_name, name, subdirectory,
    ):
        out_path = (prefix_path / 'share' / pkg_name / 'hook' / env_hook_name).with_suffix(EXT_SCRIPT)
        logger.info("Creating environment hook {}".format(out_path))
        expand_py_template((PATH_TEMPLATES / STEM_HOOK_PREPEND).with_suffix(EXT_TEMPLATE),
                           out_path,
                           dict(name=name,
                                subdirectory=subdirectory),
                           )
        return out_path
