from cast_common.restAPI import RestCall
from cast_common.logger import Logger, INFO,DEBUG
from cast_common.util import format_table

from requests import codes
from pandas import ExcelWriter,DataFrame,json_normalize,concat
from typing import List
from json import loads,dumps

class Highlight(RestCall):

    _data = {}
    _third_party = {}
    _cves = {}
    _license = {}

    _tags = []
    _cloud = []
    _oss = []
    _elegance = []
    _apps = None
    _apps_full_list = None
    _instance_id = 0
    _base_url = None
    _hl_basic_auth = None
    _hl_user = None
    _hl_pswd = None

    log = None

    grades = ['softwareHealth',
                'softwareResiliency',
                'softwareAgility',
                'softwareElegance',
                'cloudReady',
                'openSourceSafety']

    def __init__(self,  
                 hl_user:str=None, hl_pswd:str=None,hl_basic_auth=None, hl_instance:int=0,
                 hl_apps:str=[],hl_tags:str=[], 
                 hl_base_url:str=None, 
                 log_level=INFO, timer_on=False):

        # general message to be used by ValueError exception
        msg='must be supplied with the first Highlight class instance'

        if Highlight.log is None:
            Highlight._log_level = log_level
            Highlight.log = Logger('Highlight',log_level)

        reset_data = False
        if Highlight._instance_id == 0:
            if hl_instance == 0:
                raise ValueError(f'Domain id {msg}') 

            Highlight._instance_id = hl_instance
            Highlight._data = {}
            reset_data = True
        
        #set the base url, if not already set from a prev instance of the Highlight class
        if Highlight._base_url is None:
            if hl_base_url is None:
                raise ValueError(f'Base url {msg}') 

            Highlight._base_url = hl_base_url
            if Highlight._base_url.endswith('/'):
                Highlight._base_url=Highlight._base_url[:-1]
                            
            if not Highlight._base_url.endswith('/WS2'):
                Highlight._base_url=f'{Highlight._base_url}/WS2/'

        #set the user name/password or Basic Autorization 
        if Highlight._hl_basic_auth is None and \
            Highlight._hl_user is None and \
            Highlight._hl_pswd is None:

            if hl_user is None and \
                hl_pswd is None and \
                hl_basic_auth is None:
                raise ValueError(f'UserName/password or Basic Authorization {msg}')

            if not hl_user is None:
                Highlight._hl_user=hl_user

            if not hl_pswd is None:
                Highlight._hl_pswd=hl_pswd

            if not hl_basic_auth is None:
                Highlight._hl_basic_auth=hl_basic_auth

        super().__init__(base_url=Highlight._base_url, user=Highlight._hl_user, password=Highlight._hl_pswd, 
                         basic_auth=Highlight._hl_basic_auth,track_time=timer_on,log_level=Highlight._log_level)

        # self._business = self._get_top_metric('businessValue')
        # self._cloud = self._get_top_metric('cloudValue')
        # self._oss = self._get_top_metric('openSourceSafty')
        # #self._elegance = self._get_top_metric('softwareElegance')

        if reset_data:
            Highlight._tags = self._get_tags()
            Highlight._apps_full_list = []

            # retrieve all applications from HL REST API
            Highlight._apps_full_list = DataFrame(self._get_applications())
#            Highlight._apps_full_list.dropna(subset=['metrics'],inplace=True)
            self.info(f'Found {len(Highlight._apps_full_list)} analyzed applications: {",".join(Highlight._apps_full_list["name"])}')
    
            # if the apps is not already of list type then convert it now
            if not isinstance(hl_apps,list):
                hl_apps=list(hl_apps.split(','))

            #if hl_apps is empty, include all applications prevously retrieved
            #otherwise filter the list down to include only the selected applications
            if len(hl_apps) > 0:
                Highlight._apps = Highlight._apps_full_list[Highlight._apps_full_list['name'].isin(hl_apps)]
            else:
                Highlight._apps = Highlight._apps_full_list

            if Highlight._apps is not None:
                self.info(f'{len(Highlight._apps)} applications selected: {",".join(Highlight._apps["name"])}')

        pass

    @property
    def app_list(self) -> DataFrame:
        return Highlight._apps

    def _get_application_data(self,app:str=None):
        self.debug('Retrieving all HL application specific data')
        
        if str is None:
            df = Highlight._apps
        else:
            df = Highlight._apps[Highlight._apps['name']==app]

        for idx,app in df.iterrows():
            app_name = app['name']
            if app_name not in Highlight._data:
                try:
                    if not app_name in Highlight._data:
                        self.info(f'Loading Highlight data for {app_name}')
                        Highlight._data[app_name]=self._get_app_from_rest(app_name)
                except KeyError as ke:
                    self.warning(str(ke))
                    pass
                except Exception as ex:
                    self.error(str(ex))
        pass

    def _get_metrics(self,app_name:str) -> dict:
        try:
            if app_name not in Highlight._apps['name'].to_list():
                raise ValueError(f'{app_name} is not a selected application')

            self._get_application_data(app_name)
            data = Highlight._data[app_name]
            metrics = data['metrics'][0]
            return metrics
        except KeyError as ke:
            self.warning(f'{app_name} has no Metric Data')
            return {}

    def _get_third_party_data(self,app_name:str=None):
        self.debug('Retrieving third party HL data')

        if app_name not in Highlight._apps['name'].to_list():
            raise ValueError(f'{app_name} is not a selected application')

        if str is None:
            df = Highlight._apps
        else:
            df = Highlight._apps[Highlight._apps['name']==app_name]

        for idx,app in df.iterrows():
            app_name = app['name']
            if app_name not in Highlight._third_party:
                try:
                    if not app_name in Highlight._third_party:
                        self.info(f'Loading Highlight component data for {app_name}')
                        data = self._get_third_party_from_rest(app_name)
                        Highlight._third_party[app_name]=data
                except KeyError as ke:
                    self.warning(str(ke))
                    pass
                except Exception as ex:
                    self.error(str(ex))
        pass

    def _get_third_party(self,app_name:str) -> dict:
        try:
            if app_name not in Highlight._apps['name'].to_list():
                raise ValueError(f'{app_name} is not a selected application')

            self._get_third_party_data(app_name)
            data = Highlight._third_party[app_name]
#            third_party = data['metrics'][0]
            return data
        except KeyError as ke:
            self.error(f'{app_name} has no Metric Data')
            raise ke

    def _get_third_party_from_rest(self,app:str) -> dict:
        return self._get(f'domains/{Highlight._instance_id}/applications/{self.get_app_id(app)}/thirdparty')

    def _get(self,url:str,header=None) -> DataFrame:
        (status, json) = self.get(url,header)
        if status == codes.ok:
#            return DataFrame(json)
            return json
        else:
            raise KeyError (f'Server returned a {status} while accessing {url}')

    def _get_applications(self):
        return DataFrame(self._get(f'domains/{Highlight._instance_id}/applications/', {'Accept': 'application/vnd.castsoftware.api.basic+json'} ))
        
    def _get_tags(self) -> DataFrame:
        return DataFrame(self._get(f'domains/{Highlight._instance_id}/tags/'))

    # def _get_top_metric(self,metric:str) -> DataFrame:
    #     return DataFrame(self.post(f'domains/{Highlight._instance_id}/metrics/top?metric=cloudReady&order=desc',header={'Content-type':'application/json'}))

    def _get_app_from_rest(self,app:str) -> dict:
        return DataFrame(self._get(f'domains/{Highlight._instance_id}/applications/{self.get_app_id(app)}'))

    def get_app_id(self,app_name:str) -> int:
        """get the application id

        Args:
            app_name (str): application name 

        Raises:
            KeyError: application not found

        Returns:
            int: highlight application id
        """
        if len(Highlight._apps)==0:
            Highlight._apps = self._get_applications()

        series = Highlight._apps[Highlight._apps['name']==app_name]
        if series.empty:
            raise KeyError (f'Highlight application not found: {app_name}')
        else:
            return int(series.iloc[0]['id'])
            
    def get_tag_id(self,tag_name:str):
        """get the application id

        Args:
            tag_name (str): tag name 

        Raises:
            KeyError: tag not found

        Returns:
            int: highlight tag id
        """
        if len(Highlight._tags)==0:
            Highlight._tags = self._get_tags()

        series = Highlight._tags[Highlight._tags['label']==tag_name]
        if series.empty:
            raise KeyError (f'Highlight tag not found: {tag_name}')
        else:
            return int(series.iloc[0]['id'])
            
    def add_tag(self,app_name,tag_name) -> bool:
        """Add tag for application 

        Args:
            app_name (_type_): applicaton name
            tag_name (_type_): tag name
        Returns:
            bool: True if tag added
        """
        app_id = self.get_app_id(app_name)
        tag_id = self.get_tag_id(tag_name)

        url = f'domains/{Highlight._instance_id}/applications/{app_id}/tags/{tag_id}'
        (status,json) = self.post(url)
        if status == codes.ok or status == codes.no_content:
            return True
        else:
            return False 
    
    def get_tech_debt_advisor(self,order:List[str],tag:str=None) -> DataFrame:
        #https://rpa.casthighlight.com/WS2/domains/1271/technicalDebt/aggregated?tagIds=685&activePagination=false&pageOffset=0&maxEntryPerPage=9999&maxPage=999&order=healthFactor&order=alert
        tag_part = ""
        if tag is not None:
            tag_part = f'tagIds={self.get_tag_id(tag)}&'
        url = f'domains/{Highlight._instance_id}/technicalDebt/aggregated?{tag_part}&activePagination=false&pageOffset=0&maxEntryPerPage=9999&maxPage=999'
        for o in order:
            url=f'{url}&order={o}'
        
        (status, json) = self.get(url)
        if status == codes.ok:
            d1 = json_normalize(json,['subLevel'], max_level=1)
            d1.rename(columns={'id':'Health Factor'},inplace=True)
            d1.drop(columns=['levelType'], inplace=True)

            d2=d1.explode('subLevel')
            d2=json_normalize(d2.to_dict('records'), max_level=1)
            d2=d2.rename(columns={'subLevel.id':'Alert','subLevel.detail':'detail'})
            d2=d2.drop(columns=['subLevel.levelType'])

            d3=json_normalize(d2.to_dict('records'),['detail'],meta=['Health Factor','Alert'])
            d3=d3.rename(columns={'application_id':'App Id','effort':'Effort'})
            d3=d3.drop(columns=['counter'])

            return d3

        else:
            raise KeyError (f'Server returned a {status} while accessing {url}')

    """ **************************************************************************************************************
                                            Third Party Component Data 
    ************************************************************************************************************** """
    def get_component_total(self,app_name:str) -> int:
        third_party = len(self._get_third_party(app_name)['thirdParties'])
        return int(third_party)

    def get_cve_data(self,app_name:str) -> DataFrame:
        third_party = self._get_third_party(app_name)['thirdParties']
        
        if not app_name in Highlight._cves:
            cves = DataFrame()
            for tp in third_party:
                if 'cve' in tp:
                    cve_df = json_normalize(tp['cve']['vulnerabilities'])
                    cve_df.rename(columns={'name':'cve'},inplace=True)
                    
                    cve_df['component']=tp['name']
                    cve_df['version']=tp['version']
                    cve_df['languages']=tp['languages']
                    cve_df['release']=tp['release']
                    cve_df['origin']=tp['origin']
                    cve_df['lastVersion']=tp['lastVersion']

                    cves=concat([cves,cve_df],ignore_index=True)
            
            if not cves.empty and 'component' in cves.columns:
                cves=cves[['component','version','languages','release','origin','lastVersion','cve', 'description', 'cweId', 'cweLabel', 'criticity', 'cpe']]

            Highlight._cves[app_name]=cves
        else:
            cves = Highlight._cves[app_name]
        return cves            

    def get_cve_critical(self, app_name:str) -> DataFrame:
        cves = self.get_cve_data(app_name)
        return cves[cves['criticity']=='CRITICAL']
    def get_cve_high(self, app_name:str) -> DataFrame:
        cves = self.get_cve_data(app_name)
        return cves[cves['criticity']=='HIGH']
    def get_cve_medium(self, app_name:str) -> DataFrame:
        cves = self.get_cve_data(app_name)
        return cves[cves['criticity']=='MEDIUM']

    def get_license_data(self,app_name:str) -> DataFrame:
        third_party = self._get_third_party(app_name)['thirdParties']
        
        if not app_name in Highlight._license:
            lic = DataFrame()
            for tp in third_party:
                if 'licenses' in tp:
                    lic_df = json_normalize(tp['licenses'], \
                        meta = ['name','version','languages','release','origin','lastVersion'])
                    lic_df.rename(columns={'name':'license'},inplace=True)
                    lic_df['component']=tp['name']
                    load_df_element(tp,lic_df,'version')
                    load_df_element(tp,lic_df,'languages')
                    load_df_element(tp,lic_df,'release')
                    load_df_element(tp,lic_df,'origin')
                    load_df_element(tp,lic_df,'lastVersion')
                    lic=concat([lic,lic_df],ignore_index=True)
            
            if not lic.empty and  'component' in lic.columns:
                lic=lic[['component','version','languages','release','origin','lastVersion','license','compliance']] 
                lic['compliance']=lic['compliance'].str.replace('compliant','high')
                lic['compliance']=lic['compliance'].str.replace('partial','medium')
                lic['compliance']=lic['compliance'].str.replace('notCompliant','low')

            Highlight._license[app_name]=lic
        else:
            lic = Highlight._license[app_name]
        return lic            

    def get_license_high(self,app_name:str) -> DataFrame:
        lic = self.get_license_data(app_name)
        return lic[lic['compliance']=='high'] 

    def get_license_medium(self,app_name:str) -> DataFrame:
        lic = self.get_license_data(app_name)
        return lic[lic['compliance']=='medium']   

    def get_license_low(self,app_name:str) -> DataFrame:
        lic = self.get_license_data(app_name)
        return lic[lic['compliance']=='low']   

    """ **************************************************************************************************************
                                            Cloud Ready Data 
    ************************************************************************************************************** """
    def get_cloud_detail(self,app_name:str)->DataFrame:
        """Highlight cloud ready data

        Args:
            app_name (str): name of the application

        Returns:
            DataFrame: flattened version of the Highlight cloud ready data
        """
        try:
            return json_normalize(self._get_metrics(app_name)['cloudReadyDetail'],['cloudReadyDetails'],meta=['technology','cloudReadyScan'])
        except KeyError as ke:
            self.warning(f'{app_name} has no Cloud Ready Data')
            return None

    """ **************************************************************************************************************
                                            Cloud Ready Data 
    ************************************************************************************************************** """
    def get_green_detail(self,app_name:str)->DataFrame:
        """Highlight green impact data

        Args:
            app_name (str): name of the application

        Returns:
            DataFrame: flattened version of the Highlight cloud ready data
        """
        try:
            return json_normalize(self._get_metrics(app_name)['greenDetail'],'greenIndexDetails')
        except KeyError as ke:
            self.warning(f'{app_name} has no Green Impact Data')
            return None

    """ **************************************************************************************************************
                                            General Metrics Data 
    ************************************************************************************************************** """
    def get_technology(self,app_name:str) -> DataFrame:
        """Highlight application technology data

        Args:
            app_name (str): _description_

        Returns:
            DataFrame: _description_
        """
        df = json_normalize(self._get_metrics(app_name))
        if not df.empty:
            tech = json_normalize(df['technologies'])
            tech = tech.transpose()
            tech = json_normalize(tech[0])
            return tech.sort_values(by=['totalLinesOfCode'],ascending=False)
        else:
            return DataFrame()

    def get_total_lines_of_code(self,app_name:str) -> int:
        return json_normalize(self._get_metrics(app_name))['totalLinesOfCode']    

    """ ******************************************************************************
                        Highlight Scores
    ****************************************************************************** """
    def get_software_health_score(self,app_name:str) -> float:
        metrics = self._get_metrics(app_name)
        return float(metrics['softwareHealth'])*100

    def get_software_health_hml(self,app_name:str=None,score=None) -> str:
        if score is None:
            score = self.get_software_health_score(app_name)
        if score > 75:
            return 'high'
        elif score > 52:
            return 'medium'
        else:
            return 'low'

    def get_software_agility_score(self,app_name:str) -> float:
        metrics = self._get_metrics(app_name)
        return float(metrics['softwareAgility'])*100

    def get_software_agility_hml(self,app_name:str) -> str:
        score = self.get_software_agility_score(app_name)
        if score > 69:
            return 'high'
        elif score > 54:
            return 'medium'
        else:
            return 'low'

    def get_software_elegance_score(self,app_name:str) -> float:
        metrics = self._get_metrics(app_name)
        return float(metrics['softwareElegance'])*100

    def get_software_elegance_hml(self,app_name:str) -> str:
        score = self.get_software_elegance_score(app_name)
        if score > 70:
            return 'high'
        elif score > 39:
            return 'medium'
        else:
            return 'low'

    def get_software_resiliency_score(self,app_name:str) -> float:
        metrics = self._get_metrics(app_name)
        return float(metrics['softwareResiliency'])*100
    
    def get_software_resiliency_hml(self,app_name:str) -> str:
        score = self.get_software_resiliency_score(app_name)
        if score > 87:
            return 'high'
        elif score > 65:
            return 'medium'
        else:
            return 'low'

    def get_software_cloud_score(self,app_name:str) -> float:
        metrics = self._get_metrics(app_name)
        return float(metrics['cloudReady'])*100
    def get_get_cloud_hml(self,app_name:str=None,score=None) -> str:
        if score is None:
            score = self.get_software_cloud_score(app_name)
        if score > 75:
            return 'high'
        elif score > 52:
            return 'medium'
        else:
            return 'low'

    def get_software_oss_safty_score(self,app_name:str) -> float:
        metrics = self._get_metrics(app_name)
        return float(metrics['openSourceSafety'])*100
    def get_get_software_oss_risk(self,app_name:str=None,score=None) -> str:
        if score is None:
            score = self.get_software_oss_safty_score(app_name)
        if score > 75:
            return 'low'
        elif score > 52:
            return 'medium'
        else:
            return 'high'

    def get_software_oss_license_score(self,app_name:str) -> float:
        metrics = self._get_metrics(app_name)
        return float(metrics['openSourceLicense'])*100

    def get_software_oss_obsolescence_score(self,app_name:str) -> float:
        metrics = self._get_metrics(app_name)
        return float(metrics['openSourceObsolescence'])*100

    def get_software_green_score(self,app_name:str) -> float:
        metrics = self._get_metrics(app_name)
        green_index = metrics['greenIndex']
        if green_index is None: green_index = 0
        return float(green_index)*100

    def get_software_green_hml(self,app_name:str=None,score=None) -> str:
        if score is None:
            score = self.get_software_green_score(app_name)
        if score > 75:
            return 'high'
        elif score > 52:
            return 'moderate'
        else:
            return 'low'

def load_df_element(src,dst,name):
    if not (src.get(name) is None):
        dst[name]=src[name] 


