import argparse
import subprocess
from .generator import gerar_mensagem_commit
from .configure import api_key, get_configured_model
import sys
import getpass

def main():
    parser = argparse.ArgumentParser(description="Gerador de mensagens de commit com IA")
    parser.add_argument("-c", "--commit", action="store_true", help="Commits with the generated message")
    parser.add_argument("-cp", "--commitpush",  action="store_true", help="Commits and pushes with the generated message")
    parser.add_argument("-cf", "--configure", action="store_true", help="Configures the GEMINI_API_KEY environment variable")
    parser.add_argument("-s", "--status", action="store_true", help="Shows current configuration status")
    args = parser.parse_args()

    if args.status:
        from .configure import get_configured_model
        import os
        from dotenv import load_dotenv
        
        load_dotenv()
        key = os.getenv("GEMINI_API_KEY")
        model = get_configured_model()
        
        print("\nCurrent Configuration:")
        print(f"API Key: {'✓ Set' if key else '✗ Not set'}")
        print(f"Model: {model}")
        return

    if not args.configure:
        mensagem = gerar_mensagem_commit()

        if "No changes detected" in mensagem:
            print(mensagem)
            return

        print("\nGenerated commit message:\n" + mensagem)

    if args.commit or args.commitpush:
        print("\nCommitting changes...")
        subprocess.run(["git", "commit", "-m", mensagem])

    if args.commitpush:
        print("\nPushing changes...")
        subprocess.run(["git", "push"])
    
    if args.configure:
        print("\nPlease input your API KEY\nThis is directly set in the .env file")
        key = getpass.getpass()
        
        # Model selection
        print("\nAvailable models:")
        print("1. gemini-2.0-flash (default, fast and efficient)")
        print("2. gemini-1.5-flash (good balance of speed and quality)")
        print("3. gemini-1.5-pro (highest quality, slower)")
        print("4. gemini-2.0-flash-exp (experimental)")
        print("5. gemini-2.5-flash (latest, fast and efficient)")
        print("6. gemini-2.5-pro (latest, highest quality)")
        
        model_choice = input("\nSelect model (1-6) or press Enter for default (1): ").strip()
        
        model_map = {
            "1": "gemini-2.0-flash",
            "2": "gemini-1.5-flash", 
            "3": "gemini-1.5-pro",
            "4": "gemini-2.0-flash-exp",
            "5": "gemini-2.5-flash",
            "6": "gemini-2.5-pro"
        }
        
        selected_model = model_map.get(model_choice, "gemini-2.0-flash")
        
        api_key(key, selected_model)
        print(f"\nAPI KEY and model ({selected_model}) saved in .env file\n")
    
    if len(sys.argv) == 1:
        print("\nRemoving staged changes (git reset)...")
        subprocess.run(["git", "reset"])
