import os
import shutil
import stat
from pathlib import Path
from typing import Optional

import keyring
import questionary
import typer
from rich import print
from rich.table import Table
from typing_extensions import Annotated

from commitzilla.characters import CharacterDict
from commitzilla.config import ConfigSchema, CzConfig

app = typer.Typer()

DEFAULT_MODEL = "gpt-4o-mini"

HOOK_DIR_PATH = Path.cwd() / ".git" / "hooks"
HOOK_PATH = HOOK_DIR_PATH / "prepare-commit-msg"
CONFIG_PATH = HOOK_DIR_PATH / "cz-config.ini"
CHARACTERS_PATH = HOOK_DIR_PATH / "cz_characters.json"

question_style = questionary.Style(
    [
        ("answer", "fg:#008000 bold"),
        ("highlighted", "fg:#0000ff bold"),
        ("pointer", "fg:#0000ff bold"),
    ]
)


@app.command()
def install():
    if _is_hook_installed():
        print(
            "[yellow]commitzilla is already installed in this repository, use [bold]commitzilla uninstall[/bold] to uninstall it[/yellow]"
        )
        return

    print(
        ":rocket: [yellow]Installing [bold]commitzilla[/bold] hook...[/yellow] :cowboy_hat_face:"
    )

    if not keyring.get_password("commitzilla", "api_key"):
        api_key = questionary.password(
            "Enter your OpenAI API key", style=question_style
        ).ask()
        _update_values(api_key=api_key)

    CharacterDict()
    character_name, character_prompt = _input_character()

    config_schema = ConfigSchema(
        model=DEFAULT_MODEL,
        character_name=character_name,
        character_prompt=character_prompt,
        prefix="no",
        enabled="yes",
    )
    config = CzConfig()
    config.write(config_schema)

    _move_hook_file()

    print(
        ":white_check_mark: [green]Successfully installed [bold]commitzilla[/bold][/green]!"
    )


@app.command(
    help="Enable or disable the character name being added to the commit message, with this enabled, a commit will look like: '[character] message'"
)
def prefix():
    config = CzConfig()
    choices = ["Enable", "Disable"]
    selection = questionary.select(
        "Do you want to enable or disable the character name prefix? This will add the character name to the commit message.",
        choices=choices,
        style=question_style,
    ).ask()

    enable = selection == choices[0]

    config_schema = ConfigSchema(
        prefix="yes" if enable else "no",
    )
    config.write(config_schema)

    print(
        f":white_check_mark: [green]{'Enabled' if enable else 'Disabled'} character prefix. {'Your commits will now include the character name.' if enable else 'Your commits will no longer include the character name.'}[/green]"
    )


@app.command()
def uninstall():
    if _is_hook_installed():
        confirmation = questionary.confirm(
            "Are you sure you want to remove the commitzilla hook?",
            style=question_style,
        ).ask()

        if confirmation:
            print(
                ":warning: [yellow]Removing [bold]commitzilla[/bold] hook...[/yellow] :sleepy:"
            )
            _remove_hook()
            print(
                ":white_check_mark: [green]Successfully removed [bold]commitzilla[/bold] hook... have a nice life without me...[/green]"
            )
        else:
            print(
                ":grey_question: YAY! [bold]commitzilla[/bold] hook was saved :smirk:"
            )
        return

    print(
        "[yellow]commitzilla is not installed in this repository, use [bold]commitzilla install[/bold] to install it[/yellow]"
    )


@app.command()
def configure(
    model: Annotated[Optional[str], typer.Option(help="OpenAI model to use")] = None,
    api_key: Annotated[Optional[str], typer.Option(help="OpenAI API key")] = None,
):
    _update_values(model, api_key)

    if updated_values := [
        t[0] for t in zip(("model", "api_key"), (model, api_key)) if t[1]
    ]:
        print(
            f":white_check_mark: [green]Successfully updated [bold]commitzilla[/bold] values: {', '.join(updated_values)}[/green]"
        )
        return
    print(
        ":grey_question: No configuration parameters were specified, try running [bold]commitzilla configure --help[/bold] for more information"
    )


@app.command()
def character(
    character: Annotated[Optional[str], typer.Argument()] = None,
    list: Annotated[bool, typer.Option()] = False,
):
    if list:
        characters = CharacterDict()
        table = Table(title="Available characters")
        table.add_column("Character")
        [table.add_row(c) for c in characters.keys()]
        print(table)
        return

    if character:
        characters = CharacterDict()
        if character not in characters:
            print(
                "[red]Character not found, use [bold]commitzilla character --list[/bold] to see available characters[/red]"
            )
            return

        character_prompt = characters[character]
        character_name = character
    else:
        character_name, character_prompt = _input_character()

    config = CzConfig()
    config.write(
        ConfigSchema(character_name=character_name, character_prompt=character_prompt)
    )
    print(f":white_check_mark: [green]Now using '{character_name}' character[/green]")


@app.command()
def check():
    if _is_hook_installed():
        print(":white_check_mark: [green][bold]commitzilla[/bold] is installed[/green]")
    else:
        print(":x: [red][bold]commitzilla[/bold] is not installed[/red]")


@app.command()
def enable():
    _toggle_enabled(True)


@app.command()
def disable():
    _toggle_enabled(False)


def _toggle_enabled(enable: bool):
    config = CzConfig()
    config.write(ConfigSchema(enabled="yes" if enable else "no"))
    print(
        f":white_check_mark: [green][bold]commitzilla[/bold] has been {'enabled' if enable else 'disabled'}[/green]"
    )


def _update_values(model: Optional[str] = None, api_key: Optional[str] = None):
    config_schema = ConfigSchema(model=model)
    config = CzConfig()
    config.write(config_schema)

    if api_key:
        keyring.set_password("commitzilla", "api_key", api_key)


def _is_hook_installed():
    return HOOK_PATH.exists()


def _remove_hook():
    HOOK_PATH.unlink(missing_ok=True)
    CONFIG_PATH.unlink(missing_ok=True)
    CHARACTERS_PATH.unlink(missing_ok=True)


def _move_hook_file():
    local_path = Path(__file__).parent.resolve() / "prepare-commit-msg.py"

    shutil.copy(local_path, HOOK_PATH)

    # We need to edit the permissions of the file, so git can execute it
    os.chmod(
        HOOK_PATH,
        stat.S_IRUSR
        | stat.S_IWUSR
        | stat.S_IXUSR
        | stat.S_IRGRP
        | stat.S_IXGRP
        | stat.S_IROTH
        | stat.S_IXOTH,
    )


def _input_character():
    choices = ["Preconfigured", "Custom"]
    selection = questionary.select(
        "Do you want to use a preconfigured character, or prompt your own?",
        choices=choices,
        style=question_style,
    ).ask()

    characters = CharacterDict()
    character_prompt = None

    if selection == choices[0]:
        character_name = questionary.select(
            "Which character do you want to use?",
            choices=sorted(characters.keys()),
            style=question_style,
        ).ask()

        character_prompt = characters[character_name]

    else:
        character_name = questionary.text(
            "What's the name of this character (does not affect the prompt):",
            style=question_style,
        ).ask()
        character_prompt = questionary.text(
            f"What's the custom prompt for '{character_name}' (treat this as a description of the character):",
            style=question_style,
        ).ask()

        characters[character_name] = character_prompt

    return (character_name, character_prompt)
