import os
from typing import Any, Dict

import subprocess
import h5py
import pytest
from compas_python_utils.cosmic_integration.binned_cosmic_integrator.bbh_population import \
    generate_mock_bbh_population_file

HERE = os.path.dirname(__file__)
TEST_CONFIG_DIR = os.path.join(HERE, "test_data")
TEST_BASH = os.path.join(TEST_CONFIG_DIR, "run.sh")
TEST_ARCHIVE_DIR = os.path.join(HERE, "test_artifacts")
REPO_ROOT = os.getenv("COMPAS_ROOT_DIR", os.path.abspath(os.path.join(HERE, "..")))
COMPAS_EXE = os.path.join(REPO_ROOT, "src", "COMPAS")

@pytest.fixture()
def example_compas_output_path(clean=False):
    """Get the path to the COMPAS output file generated by the test suite.

    (This is a fixture so it can passed as a parameter to other tests)
    """
    compas_data_path = os.path.join(
        TEST_CONFIG_DIR, "COMPAS_Output/COMPAS_Output.h5"
    )

    if not os.path.exists(compas_data_path) or clean:  # Check if path exists
        curr_dir = os.getcwd()
        os.chdir(TEST_CONFIG_DIR)
        # run the command in shell "compas_run_submit {TEST_CONFIG_FNAME}" with subprocess
        subprocess.run(TEST_BASH, shell=True, check=True)
        os.chdir(curr_dir)
        print("Generated COMPAS test data")


    return compas_data_path


@pytest.fixture()
def test_archive_dir():
    """Get the path to the test archive directory."""
    os.makedirs(TEST_ARCHIVE_DIR, exist_ok=True)
    return TEST_ARCHIVE_DIR


def get_compas_data(path: str) -> Dict[str, Any]:
    """Reads in a COMPAS h5 file and returns a dict with some data from the file."""
    data = {}
    with h5py.File(path, "r") as f:
        data["groups"] = list(f.keys())
        for group in data["groups"]:
            data[f"{group}_n_cols"] = len(f[group])
            if "SEED" in f[group]:
                data[f"{group}_SEED"] = f[group]["SEED"][:]
    return data


@pytest.fixture
def fake_compas_output(tmpdir) -> str:
    fname = f"{tmpdir}/COMPAS_mock_output.h5"
    generate_mock_bbh_population_file(
        filename=fname,
    )
    return fname