import datetime
import tempfile
import typing as t
from collections import OrderedDict
from pathlib import Path

import pytest


MY_DIR = Path(__file__).parent
C_KEY = 'cookiecutter'  # COOKIECUTTER_KEY
# offest by 2 hours to match Jinja 'now' expression: {% now 'utc', '%Y-%m-%d' %}
# RELEASE_DATE = (datetime.datetime.now() - datetime.timedelta(hours=2)).strftime('%Y-%m-%d')
RELEASE_DATE = datetime.datetime.now().strftime('%Y-%m-%d')


# AUTOMATED TESTs for Cookiecutter Context Generation
#  - GIVEN Inputs to cookiecutter (callable)
#     - Cookiecutter Template: Dir with cookiecutter.json file and {{ cookiecutter.project_name }}/ dir
#     - User Config YAML: File with User Config data passed as --config-file to cookiecutter
#  - GIVEN expectations:
#     - Jinja Context object expected to be generated by cookiecutter under the hood

# Verifies cookiecutter.generate_context is called with expected input parameter values
# Verifies the Jinja Context object generated has the 'cookiecutter' and '_cookiecutter' keys
# Verifies values of keys are == "exactly equal" to expected values


# TEST CASE 2 depends on finding the .github/biskotaki.yaml in local checkout,
# because its automatic discovery mechanism relies on relative path from this file

# CONSTANTS

PROD_TEMPLATE = 'PROD_TEMPLATE'


TEST_TIME_BISKOTAKI_CONFIG = None
SHOULD_SKIP = False

if not (MY_DIR.parent.parent / '.github' / 'biskotaki.yaml').exists():
    SHOULD_SKIP = True
    # Create a temporary file to use as a test config and PRESERVE it on close!
    with tempfile.NamedTemporaryFile(mode='w+b', delete=False) as fp:
        fp.write(
            b"""
default_context:
    project_name: Biskotaki
    project_type: module
    project_slug: biskotaki
    pkg_name: biskotaki
    repo_name: biskotaki
    readthedocs_project_slug: biskotaki
    docker_image: biskotaki
    full_name: Konstantinos Lampridis
    author: Konstantinos Lampridis
    email: k.lampridis@hotmail.com
    author_email: k.lampridis@hotmail.com
    github_username: boromir674
    project_short_description: Project generated using https://github.com/boromir674/cookiecutter-python-package
    initialize_git_repo: 'no'
    interpreters: {"supported-interpreters": ["3.7", "3.8", "3.9", "3.10", "3.11"]}
    ## Documentation Config ##
    docs_builder: "sphinx"
    ## READ THE DOCS CI Config ##
    rtd_python_version: "3.10"
    cicd: 'experimental'

"""
        )
        fp.close()
        TEST_TIME_BISKOTAKI_CONFIG = Path(fp.name)


@pytest.fixture(
    params=[
        (
            # TEST CASE 1 - Simple Template with Cookiecutter Choice Variable included
            # GIVEN a simple Cookiecutter Template (cookiecutter.json + jinja Template project)
            MY_DIR / '..' / 'data' / 'rendering' / 'only_list_template',
            # GIVEN a User Config YAML
            MY_DIR / '..' / 'data' / 'rendering' / 'user_config.yml',
            # THEN expected Context that cookiecutter will generate under the hood is
            OrderedDict(
                [
                    # 1st Item mapped in Jinja context with dedicated key
                    (
                        'cookiecutter',
                        OrderedDict(
                            [
                                ('project_dir_name', 'unit-test-new-project'),
                                ('some_setting', 'another_option'),
                                # ('_template', str(cookie)),
                            ]
                        ),
                    ),
                    # 2nd Item mapped in Jinja context with dedicated key
                    (
                        '_cookiecutter',
                        {
                            'project_dir_name': 'unit-test-new-project',
                            'some_setting': [
                                'another_option',
                                'some_option',
                            ],  # NOTE Difference to 1st Item
                        },
                    ),
                ]
            ),
        ),
        pytest.param(
            # TEST CASE 2 - Production Template included in Distribution
            # whenever src/cookiecutter_python/cookiecutter.json is updated, the below needs syncing
            (
                PROD_TEMPLATE,  # GIVEN the prod Template (cookiecutter.json + jinja Template project)
                'BISKOTAKI_CONFIG',  # and the .github/biskotaki.yaml User Config YAML
                # EXPECTED CONTEXT
                OrderedDict(
                    [
                        # 1st Item mapped in Jinja context with dedicated key
                        (
                            'cookiecutter',
                            OrderedDict(
                                [  # KEY, Default/First choice
                                    ('project_name', 'Biskotaki'),
                                    ('project_type', 'module'),
                                    ("project_slug", "biskotaki"),
                                    ("pkg_name", "biskotaki"),
                                    ("repo_name", "biskotaki"),
                                    ("readthedocs_project_slug", "biskotaki"),
                                    ("docker_image", "biskotaki"),
                                    ("full_name", "Konstantinos Lampridis"),
                                    ("author", "Konstantinos Lampridis"),
                                    ("author_email", 'k.lampridis@hotmail.com'),
                                    ("github_username", 'boromir674'),
                                    (
                                        "project_short_description",
                                        "Project generated using https://github.com/boromir674/cookiecutter-python-package",
                                    ),
                                    (
                                        "pypi_subtitle",
                                        "Project generated using https://github.com/boromir674/cookiecutter-python-package",
                                    ),
                                    ("release_date", RELEASE_DATE),
                                    ("year", str(datetime.datetime.now().year)),
                                    ("version", "0.0.1"),
                                    ("initialize_git_repo", "no"),
                                    (
                                        "interpreters",
                                        {
                                            "supported-interpreters": [
                                                "3.6",
                                                "3.7",
                                                "3.8",
                                                "3.9",
                                                "3.10",
                                                "3.11",
                                            ]
                                        },
                                    ),
                                    ("docs_builder", "sphinx"),
                                    ("rtd_python_version", "3.10"),
                                    ('cicd', 'experimental'),
                                    ('include_observability', 'no'),
                                    # since the below is expected to be put in the extra context before calling cookiecutter, it gets below the rest of Variables
                                    # ('_template', str(cookie)),
                                ]
                            ),
                        ),
                        # 2nd Item mapped in Jinja context with dedicated key _cookiecutter
                        (
                            '_cookiecutter',
                            {
                                'project_name': 'Biskotaki',
                                'project_type': [
                                    'module',
                                    'module+cli',
                                    'pytest-plugin',
                                ],  # NOTE Difference to 1st Item
                                "project_slug": "biskotaki",
                                "pkg_name": "biskotaki",
                                "repo_name": "biskotaki",
                                "readthedocs_project_slug": "biskotaki",
                                "docker_image": "biskotaki",
                                "full_name": "Konstantinos Lampridis",
                                "author": "Konstantinos Lampridis",
                                "author_email": 'k.lampridis@hotmail.com',
                                "github_username": 'boromir674',
                                "project_short_description": "Project generated using https://github.com/boromir674/cookiecutter-python-package",
                                "pypi_subtitle": "{{ cookiecutter.project_short_description }}",
                                # current date in format '2024-03-04'
                                # "release_date": datetime.datetime.now().strftime('%Y-%m-%d'),
                                "release_date": "{% now 'utc', '%Y-%m-%d' %}",
                                "year": "{% now 'utc', '%Y' %}",
                                "version": "0.0.1",
                                "initialize_git_repo": [
                                    'no',
                                    'yes',
                                ],  # NOTE Difference to 1st Item
                                "interpreters": {
                                    "supported-interpreters": [
                                        "3.7",
                                        "3.8",
                                        "3.9",
                                        "3.10",
                                        "3.11",
                                    ]
                                },
                                "docs_builder": [
                                    'sphinx',
                                    'mkdocs',
                                ],  # NOTE Difference to 1st Item
                                "rtd_python_version": [
                                    "3.10",
                                    "3.8",
                                    "3.9",
                                    "3.11",
                                    "3.12",
                                ],
                                'cicd': ['experimental', 'stable'],
                                'include_observability': ['no', 'yes'],
                            },
                        ),
                    ]
                ),
            ),
            marks=pytest.mark.skipif(
                SHOULD_SKIP,
                reason=f"'Test is impossible outside of project, automatically replaced by 'test_prod_template' test case. Current path: {MY_DIR}'",
            ),
        ),
        pytest.param(
            # TEST CASE 3 - Production Template included in Distribution
            # whenever src/cookiecutter_python/cookiecutter.json is updated, the below needs syncing
            (
                PROD_TEMPLATE,  # GIVEN the prod Template (cookiecutter.json + jinja Template project)
                'TEST_TIME_BISKOTAKI_CONFIG',  # and a User Config File that "should be identical" to .github/biskotaki.yaml
                # EXPECTED CONTEXT
                OrderedDict(
                    [
                        # 1st Item mapped in Jinja context with dedicated key
                        (
                            'cookiecutter',
                            OrderedDict(
                                [
                                    ('project_name', 'Biskotaki'),
                                    ('project_type', 'module'),
                                    ("project_slug", "biskotaki"),
                                    ("pkg_name", "biskotaki"),
                                    ("repo_name", "biskotaki"),
                                    ("readthedocs_project_slug", "biskotaki"),
                                    ("docker_image", "biskotaki"),
                                    ("full_name", "Konstantinos Lampridis"),
                                    ("author", "Konstantinos Lampridis"),
                                    ("author_email", 'k.lampridis@hotmail.com'),
                                    ("github_username", 'boromir674'),
                                    (
                                        "project_short_description",
                                        "Project generated using https://github.com/boromir674/cookiecutter-python-package",
                                    ),
                                    (
                                        "pypi_subtitle",
                                        "Project generated using https://github.com/boromir674/cookiecutter-python-package",
                                    ),
                                    ("release_date", RELEASE_DATE),
                                    ("year", str(datetime.datetime.now().year)),
                                    ("version", "0.0.1"),
                                    ("initialize_git_repo", "no"),
                                    (
                                        "interpreters",
                                        {
                                            "supported-interpreters": [
                                                "3.6",
                                                "3.7",
                                                "3.8",
                                                "3.9",
                                                "3.10",
                                                "3.11",
                                            ]
                                        },
                                    ),
                                    ("docs_builder", "sphinx"),
                                    ("rtd_python_version", "3.10"),
                                    ('cicd', 'experimental'),
                                    ('include_observability', 'no'),
                                    # since the below is expected to be put in the extra context before calling cookiecutter, it gets below the rest of Variables
                                    # ('_template', str(cookie)),
                                ]
                            ),
                        ),
                        # 2nd Item mapped in Jinja context with dedicated key _cookiecutter
                        (
                            '_cookiecutter',
                            {
                                'project_name': 'Biskotaki',
                                'project_type': [
                                    'module',
                                    'module+cli',
                                    'pytest-plugin',
                                ],  # NOTE Difference to 1st Item
                                "project_slug": "biskotaki",
                                "pkg_name": "biskotaki",
                                "repo_name": "biskotaki",
                                "readthedocs_project_slug": "biskotaki",
                                "docker_image": "biskotaki",
                                "full_name": "Konstantinos Lampridis",
                                "author": "Konstantinos Lampridis",
                                "author_email": 'k.lampridis@hotmail.com',
                                "github_username": 'boromir674',
                                # "project_short_description": "Project generated using https://github.com/boromir674/cookiecutter-python-package",
                                "project_short_description": "Project generated from https://github.com/boromir674/cookiecutter-python-package/",
                                "pypi_subtitle": "{{ cookiecutter.project_short_description }}",
                                # current date in format '2024-03-04'
                                # "release_date": datetime.datetime.now().strftime('%Y-%m-%d'),
                                "release_date": "{% now 'utc', '%Y-%m-%d' %}",
                                "year": "{% now 'utc', '%Y' %}",
                                "version": "0.0.1",
                                "initialize_git_repo": [
                                    'no',
                                    'yes',
                                ],  # NOTE Difference to 1st Item
                                "interpreters": {
                                    "supported-interpreters": [
                                        "3.7",
                                        "3.8",
                                        "3.9",
                                        "3.10",
                                        "3.11",
                                    ]
                                },
                                "docs_builder": [
                                    'sphinx',
                                    'mkdocs',
                                ],  # NOTE Difference to 1st Item
                                "rtd_python_version": [
                                    "3.10",
                                    "3.8",
                                    "3.9",
                                    "3.11",
                                    "3.12",
                                ],
                                'cicd': ['experimental', 'stable'],
                                'include_observability': ['no', 'yes'],
                            },
                        ),
                    ]
                ),
            ),
            marks=pytest.mark.skipif(
                not SHOULD_SKIP,
                reason="Test is redundant, since we are running inside project",
            ),
        ),
        (
            # TEST CASE 4 - Production Template + Gold Standard User Config
            # whenever src/cookiecutter_python/cookiecutter.json is updated, the below needs syncing
            PROD_TEMPLATE,  # GIVEN the prod Template (cookiecutter.json + jinja Template project)
            'GOLD_STANDARD_CONFIG',  # and the tests/data/gold-standard.yml
            # EXPECTED CONTEXT
            OrderedDict(
                [
                    # 1st Item mapped in Jinja context with dedicated key
                    (
                        C_KEY,
                        OrderedDict(
                            [
                                ('project_name', 'Biskotaki Gold Standard'),
                                ('project_type', 'module+cli'),
                                ("project_slug", "biskotaki-gold-standard"),
                                ("pkg_name", "biskotakigold"),
                                ("repo_name", "biskotaki-gold"),
                                ("readthedocs_project_slug", "biskotaki-gold"),
                                ("docker_image", "bgs"),
                                ("full_name", "Konstantinos Lampridis"),
                                ("author", "Konstantinos Lampridis"),
                                ("author_email", 'k.lampridis@hotmail.com'),
                                ("github_username", 'boromir674'),
                                (
                                    "project_short_description",
                                    "Project generated from https://github.com/boromir674/cookiecutter-python-package/",
                                ),
                                (
                                    "pypi_subtitle",
                                    "Project generated from https://github.com/boromir674/cookiecutter-python-package/",
                                ),
                                ("release_date", RELEASE_DATE),
                                ("year", str(datetime.datetime.now().year)),
                                ("version", "0.0.1"),
                                ("initialize_git_repo", "no"),
                                (
                                    "interpreters",
                                    {
                                        "supported-interpreters": [
                                            "3.10",
                                            "3.11",
                                            "3.12",
                                        ]
                                    },
                                ),
                                ("docs_builder", "mkdocs"),
                                ("rtd_python_version", "3.11"),
                                ('cicd', 'experimental'),
                                ('include_observability', 'no'),
                            ]
                        ),
                    ),
                    # 2nd Item mapped in Jinja context with dedicated key _cookiecutter
                    (
                        '_cookiecutter',
                        {
                            'project_name': 'Biskotaki Gold Standard',
                            'project_type': [
                                'module+cli',
                                'module',
                                'pytest-plugin',
                            ],  # NOTE Difference to 1st Item
                            "project_slug": "biskotaki-gold-standard",
                            "pkg_name": "biskotakigold",
                            "repo_name": "biskotaki-gold",
                            "readthedocs_project_slug": "biskotaki-gold",
                            "docker_image": "bgs",
                            "full_name": "Konstantinos Lampridis",
                            "author": "Konstantinos Lampridis",
                            "author_email": 'k.lampridis@hotmail.com',
                            "github_username": 'boromir674',
                            # "project_short_description": "Project generated using https://github.com/boromir674/cookiecutter-python-package",
                            "project_short_description": "Project generated from https://github.com/boromir674/cookiecutter-python-package/",
                            "pypi_subtitle": "{{ cookiecutter.project_short_description }}",
                            # current date in format '2024-03-04'
                            # "release_date": datetime.datetime.now().strftime('%Y-%m-%d'),
                            # "release_date": RELEASE_DATE,
                            "release_date": "{% now 'utc', '%Y-%m-%d' %}",
                            "year": "{% now 'utc', '%Y' %}",
                            # "year": str(datetime.datetime.now().year),
                            "version": "0.0.1",
                            "initialize_git_repo": [
                                'no',
                                'yes',
                            ],  # NOTE Difference to 1st Item
                            "interpreters": {
                                "supported-interpreters": ["3.10", "3.11", "3.12"]
                            },
                            "docs_builder": [
                                'mkdocs',
                                'sphinx',
                            ],  # NOTE Difference to 1st Item
                            "rtd_python_version": [
                                "3.11",
                                "3.8",
                                "3.9",
                                "3.10",
                                "3.12",
                            ],
                            'cicd': ['experimental', 'stable'],
                            'include_observability': ['no', 'yes'],
                        },
                    ),
                ]
            ),
        ),
    ],
    ids=(
        'simple_template',  # TEST CASE 1
        'prod_template',  # TEST CASE 2 (mutually exclusive with Test Case 3)
        'test_prod_template',  # TEST CASE 3
        'gold_standard',  # TEST CASE 4
    ),
)
def template_test_case(request, distro_loc: Path):
    # handles cookiecutters dedicated for testing and the one included in the distribution
    COOKIE_TEMPLATE_DIR: Path = (
        distro_loc if request.param[0] == 'PROD_TEMPLATE' else request.param[0]
    )

    # Set User Config YAML
    ALIAS_TO_CONFIG_FILE: t.Dict[str, t.Optional[Path]] = {
        'BISKOTAKI_CONFIG': MY_DIR / '..' / '..' / '.github' / 'biskotaki.yaml',
        'TEST_TIME_BISKOTAKI_CONFIG': TEST_TIME_BISKOTAKI_CONFIG,
        'GOLD_STANDARD_CONFIG': MY_DIR / '..' / 'data' / 'gold-standard.yml',
        'PYTEST_PLUGIN_CONFIG': MY_DIR / '..' / 'data' / 'pytest-fixture.yaml',
    }

    USER_CONFIG_FILE = ALIAS_TO_CONFIG_FILE.get(request.param[1], request.param[1])

    # Prepare Expected Context, produced at runtime by cookiecutter (under the hood)
    expected_context: OrderedDict = request.param[2]

    # GIVEN the parent dir we expect the cookiecutter template to be in
    _expected_cookiecutter_parent_dir: str = str(COOKIE_TEMPLATE_DIR)

    # Solve issue of CI Windows, with a hack
    import os
    import sys

    testing_against_sdist: bool = any([x in os.environ for x in ('PY_SDIST', 'PY_WHEEL')])
    if sys.platform == 'win32' and testing_against_sdist:
        # now we allow only the 'expected_cookiecutter_parent_dir' to deviate by 1 letter !!!
        _expected_cookiecutter_parent_dir = _expected_cookiecutter_parent_dir.replace(
            'lib', 'Lib'
        )

    # ADD to Expected CONTEXT (ordered): template dir or url
    expected_context[C_KEY]['_template'] = _expected_cookiecutter_parent_dir

    # '_output_dir' key is ADDED HERE (expected_context[C_KEY]['_template']) dynamically

    # ADD to Expected CONTEXT (ordered): include repo dir or url in the context dict
    expected_context[C_KEY]['_repo_dir'] = _expected_cookiecutter_parent_dir

    # ADD to Expected CONTEXT (ordered):  include chec_keyout details in the context dict
    expected_context[C_KEY]['_checkout'] = False

    # manual JSON encoding of 'interpreters', when prod Template + Biskotaki Config
    from cookiecutter_python.backend.load_config import get_interpreters_from_yaml

    interpreters = get_interpreters_from_yaml(str(USER_CONFIG_FILE))

    print('\n---\n', interpreters)
    if interpreters:
        assert isinstance(interpreters, dict)
        assert isinstance(interpreters['supported-interpreters'], list)
        assert len(interpreters['supported-interpreters']) > 0
        # ADD to Expected CONTEXT (ordered): interpreters
        expected_context[C_KEY]['interpreters'] = interpreters

    # Sanity Check
    assert (
        request.param[0] != 'PROD_TEMPLATE'
        or isinstance(interpreters, dict)
        and isinstance(interpreters['supported-interpreters'], list)
        and len(interpreters['supported-interpreters']) > 0
    )

    # Helper Function for dynamically adding one key into expected context OrderedDict
    def get_expected_context(expected_context: t.Dict[str, t.Any], folder_path: Path):
        """Greate Expected Context as OrderedDict, given expected_context and generated project dir path"""

        def gen():
            """Generate key/values for expected context, ensuring _output_dir comes right after _template"""
            # generator keys an inject _output_dir
            for k, v in expected_context[C_KEY].items():
                # if key is not _template, yield it
                if k != '_template':
                    yield k, v
                # if key is _template, yield it and then yield _output_dir
                else:
                    yield k, v
                    yield '_output_dir', str(folder_path.absolute())

        return OrderedDict(
            [
                (C_KEY, OrderedDict([(k, v) for k, v in gen()])),
                ('_cookiecutter', expected_context['_cookiecutter']),
            ]
        )

    return {
        'cookie': COOKIE_TEMPLATE_DIR,
        'user_config': USER_CONFIG_FILE,
        'get_expected_context': lambda gen_proj_dir: get_expected_context(
            expected_context, gen_proj_dir
        ),
        'alias_of_template_used': request.param[0],
    }


CookiecutterCallable = t.Callable


@pytest.fixture
def cookiecutter_callable_mapping(
    mock_check, user_config
) -> t.Callable[[Path, Path], t.Dict[str, CookiecutterCallable]]:
    # Mapping for COOKIECUTTER_CALLABLE
    def prod_template_callable(config_file: Path):
        mock_check.config = user_config[config_file]  # read the config file
        mock_check('pypi', False)  # defer from network access and return False
        mock_check('readthedocs', False)  # defer from network access and return False

        from cookiecutter_python.backend.main import generate as generate_callback

        return generate_callback

    def test_template_callable(cookiecutter_template_dir: Path):
        from cookiecutter.main import cookiecutter

        def _generate_callback(**kwargs):
            return cookiecutter(str(cookiecutter_template_dir), **kwargs)

        return _generate_callback
        # return lambda **kwargs: cookiecutter(str(cookiecutter_template_dir), **kwargs)

    from collections import defaultdict

    def _get_cookiecutter_callable_mapping(config_file: Path, cookiecutter_template_dir: Path):
        cookiecutter_callable_mapping = defaultdict(
            # we call cookiecutter directly
            lambda: test_template_callable(cookiecutter_template_dir),
            **{
                # we call the cookiecutter_python.backend.main.generate  wrapper of cookiecutter callable
                PROD_TEMPLATE: prod_template_callable(config_file),
            },
        )
        return cookiecutter_callable_mapping

    return _get_cookiecutter_callable_mapping
