import datetime
import typing as t
from collections import OrderedDict
from pathlib import Path
from unittest.mock import patch


MY_DIR = Path(__file__).parent
C_KEY = 'cookiecutter'  # COOKIECUTTER_KEY
# offest by 2 hours to match Jinja 'now' expression: {% now 'utc', '%Y-%m-%d' %}
# RELEASE_DATE = (datetime.datetime.now() - datetime.timedelta(hours=2)).strftime('%Y-%m-%d')
RELEASE_DATE = datetime.datetime.now().strftime('%Y-%m-%d')


# AUTOMATED TESTs for Cookiecutter Context Generation
#  - GIVEN Inputs to cookiecutter (callable)
#     - Cookiecutter Template: Dir with cookiecutter.json file and {{ cookiecutter.project_name }}/ dir
#     - User Config YAML: File with User Config data passed as --config-file to cookiecutter
#  - GIVEN expectations:
#     - Jinja Context object expected to be generated by cookiecutter under the hood

# Verifies cookiecutter.generate_context is called with expected input parameter values
# Verifies the Jinja Context object generated has the 'cookiecutter' and '_cookiecutter' keys
# Verifies values of keys are == "exactly equal" to expected values


# TEST CASE 2 depends on finding the .github/biskotaki.yaml in local checkout,
# because its automatic discovery mechanism relies on relative path from this file


@patch('cookiecutter.main.generate_context')
def test_cookiecutter_generates_context_with_expected_values(
    generate_context_mock,  # magic mock object
    template_test_case,  # fixture with request.param, with Test Case Data
    cookiecutter_callable_mapping,  # support to call cookiecutter
    tmp_path: Path,  # pytest fixture
):
    """Verify generated Jinja Context, given Cookiecutter Template and User Config YAML."""
    import yaml

    # GIVEN a Cookiecutter Template: Dir with cookiecutter.json + {{ cookiecutter.project_name }}/
    cookie: Path = template_test_case['cookie']
    # GIVEN a User Config YAML, which overrides a default Choice Variable
    config_yaml: Path = template_test_case['user_config']

    # GIVEN target Gen Project dir has no files inside
    gen_proj_dir: Path = tmp_path
    assert gen_proj_dir.exists() and len(list(gen_proj_dir.iterdir())) == 0

    # GIVEN the EXPECTED CONTEXT DISCTIONARY
    EXPECTED_CONTEXT = template_test_case['get_expected_context'](gen_proj_dir)

    # GIVEN a way to "track" the input passed at runtime to cookiecutter's generate_context function
    from cookiecutter.config import get_config

    user_config_dict: t.MutableMapping[str, t.Any] = get_config(config_yaml)
    expected_default_context_passed: t.Dict = user_config_dict['default_context']
    # SANITY check User Config YAML data passed as Dict to 'default_context' kwarg of generate_context
    assert expected_default_context_passed == OrderedDict(
        [(k, v) for k, v in yaml.safe_load(config_yaml.read_text())['default_context'].items()]
    )

    expected_extra_context_passed = None

    # WHEN we call cookiecutter.generate_context like in production

    from cookiecutter.generate import generate_context

    # from cookiecutter.main import generate_context  # increases call_count by 1 !!!
    # SANITY no calls to generate_context yet
    assert generate_context_mock.call_count == 0

    # safe-keep output of production generate_context call
    prod_result = generate_context(
        context_file=str(cookie / 'cookiecutter.json'),
        default_context=expected_default_context_passed,  # user's config yaml
        extra_context=expected_extra_context_passed,  # None
    )

    import copy

    _gen_output_og_object = copy.deepcopy(prod_result)

    # SANITY no calls to generate_context yet
    assert generate_context_mock.call_count == 0

    # # THEN the production generate_context call returns expected Context
    # assert prod_result == EXPECTED_CONTEXT

    # AND the Template Variables in 'cookiecutter' key is an OrderedDict
    # AND the internal data in jinja context map under 'cookiecutter' key are as expected
    assert set(prod_result.keys()) == {C_KEY} and isinstance(prod_result[C_KEY], OrderedDict)

    # sanity that dict created by generate_context is being augmented before running jinja rendering
    assert len(prod_result[C_KEY]) == len(EXPECTED_CONTEXT[C_KEY]) - 4

    # SANITY generate_context does one Job and creates the cookiecutter key in context (used in downstream jinja rendering)

    # AND the internal generate context function received expected input parameter values

    # enable tracking of the "context" dict returned by the generate_context function
    # when called from cookiecutter

    # track returned value of generate_context from above: generator -> cookiecutter -> generate_context

    generate_context_mock.return_value = (
        prod_result  # exact output from app code generate_context function
    )
    # see cookiecutter.main where generate_context is called

    # Run cookiecutter directly, or by running Pkg Gen
    # in source code, cookiecutter.main:cookiecutter calls generate_context
    # but here our mock will trigger instead and ensure
    # 1. same object is returned from mock as from og generate_context source code
    # 2. prod_result references internal object x as in x = generate_context() !
    # 3. prod_result can be queried on-demand
    callback_mapping = cookiecutter_callable_mapping(config_yaml, cookie)
    callback_mapping[template_test_case['alias_of_template_used']](
        # str(cookie),  # template dir path
        config_file=str(config_yaml),
        default_config=False,
        output_dir=gen_proj_dir,
        # extra_context={'interpreters': EXPECTED_CONTEXT['cookiecutter']['interpreters']} if 'interpreters' in EXPECTED_CONTEXT['cookiecutter'] else None,
        no_input=True,  # non interactive
        checkout=False,
        replay=False,
    )

    # SANITY tracked generate_context was called once
    assert generate_context_mock.call_count == 1
    generate_context_mock.assert_called_once()

    # we check the runtime input parameters passed to cookiecutter's generate_context function
    # and verify that internal generate_context of coociecutter's was called with expected runtime input values

    # cookiecutter lib passes the expected arguments to the
    # cookiecutter.generate.generate_context function
    generate_context_mock.assert_called_with(
        context_file=str(
            Path(EXPECTED_CONTEXT['cookiecutter']['_template']) / 'cookiecutter.json'
        ),
        default_context=expected_default_context_passed,
        extra_context=expected_extra_context_passed,
    )

    # AND Cookiecutter inserts 2 keys into Jinja Context: 'cookiecutter' and '_cookiecutter'
    assert set(prod_result.keys()) == {C_KEY, '_cookiecutter'}

    # AND the Template Variables in 'cookiecutter' key is an OrderedDict
    assert isinstance(prod_result[C_KEY], OrderedDict)

    # AND the internal data in jinja context map under 'cookiecutter' key are as expected
    assert len(prod_result[C_KEY]) == len(EXPECTED_CONTEXT[C_KEY])
    for p1, p2 in zip(prod_result[C_KEY].items(), EXPECTED_CONTEXT[C_KEY].items()):
        assert p1[0] == p2[0], (
            "All PROD Keys: [\n"
            + '\n'.join(prod_result[C_KEY].keys())
            + '\n]\n\nAll TEST Keys: [\n'
            + '\n'.join(EXPECTED_CONTEXT[C_KEY].keys())
            + "\n]"
        )
        if p1[0] == 'release_date':
            # ACCOUNT for different timezones we accept date difference of 1 day
            runtime_date = p1[1]
            expected_date = p2[1]
            # verify diff is at most 1 day
            assert (
                abs(
                    datetime.datetime.strptime(runtime_date, '%Y-%m-%d')
                    - datetime.datetime.strptime(expected_date, '%Y-%m-%d')
                ).days
                <= 1
            ), f"Context Missmatch at '{C_KEY}' -> '{p1[0]}': Runtime: '{runtime_date}', Expected: '{expected_date}'"
        else:
            assert (
                p1[1] == p2[1]
            ), f"Context Missmatch at '{C_KEY}' -> '{p1[0]}': Runtime: '{p1[1]}', Expected: '{p2[1]}'"

    # THEN the internal data in jinja context map under 'cookiecutter' key are as expected
    assert prod_result[C_KEY] == dict(
        EXPECTED_CONTEXT[C_KEY],
        **(
            {
                'release_date': prod_result[C_KEY]['release_date'],
            }
            if 'release_date' in prod_result[C_KEY]
            else {}
        ),
    )

    # SANITY the back-up/copy of raw data is place under '_cookiecutter' key as Dict
    assert isinstance(prod_result['_cookiecutter'], dict)
    # AND the internal data in jinja context map under '_cookiecutter' key are as expected
    assert len(prod_result['_cookiecutter']) == len(EXPECTED_CONTEXT['_cookiecutter'])
    for p1, p2 in zip(
        prod_result['_cookiecutter'].items(),
        EXPECTED_CONTEXT['_cookiecutter'].items(),
    ):
        assert p1[0] == p2[0]
        # if p1[0] in {'project_short_description', 'pypi_subtitle'}:
        #     continue
        assert (
            p1[1] == p2[1]
        ), f"Error at key {p1[0]} with value '{p1[1]}'. Expected '{p2[1]}'!"

    # VERIFY that cookiecutter stores original output of generate_context (before adding 4 extra keys and converting choices to str or yes/no to bool)
    # under the '_cookiecutter' context key
    assert _gen_output_og_object['cookiecutter'] == EXPECTED_CONTEXT['_cookiecutter']
    assert set(prod_result.keys()) == {'cookiecutter', '_cookiecutter'}
    assert set(_gen_output_og_object.keys()) == {'cookiecutter'}

    # VERIFY that generate_context never converts to bool anything
    assert all([type(x) is not bool for x in _gen_output_og_object['cookiecutter'].values()])
    # exclude '_checkout' keys added iunder 'cookiecutter' context key
    assert all(
        [
            type(v) is not bool
            for k, v in prod_result['cookiecutter'].items()
            if k not in {'_checkout'}
        ]
    ), f"Keys {[k for k, v in prod_result['cookiecutter'].items() if type(v) is bool]} got converted to bool unexpectedly!"

    # ONLY in no_input = False, then cookiecutter converts to bool the yes/no !
    # very error-prone cookiecutter behavior !!!
