# -*- coding: utf-8 -*-

from enum import Enum

try:
    from typing import Self

except ImportError:
    # For earlier versions...
    from typing_extensions import Self


class HTTPStatus(Enum):
    """
    For backward compatibility because HTTPStatus class
    was added in Python 3.11...
    """

    CONTINUE = 100, "Continue"
    SWITCHING_PROTOCOLS = 101, "Switching Protocols"
    PROCESSING = 102, "Processing"
    EARLY_HINTS = 103, "Early Hints"

    OK = 200, "OK"
    CREATED = 201, "Created"
    ACCEPTED = 202, "Accepted"
    NON_AUTHORITATIVE_INFORMATION = 203, "Non-Authoritative Information"
    NO_CONTENT = 204, "No Content"
    RESET_CONTENT = 205, "Reset Content"
    PARTIAL_CONTENT = 206, "Partial Content"
    MULTI_STATUS = 207, "Multi-Status"
    ALREADY_REPORTED = 208, "Already Reported"
    IM_USED = 226, "IM Used"

    MULTIPLE_CHOICES = 300, "Multiple Choices"
    MOVED_PERMANENTLY = 301, "Moved Permanently"
    FOUND = 302, "Found"
    SEE_OTHER = 303, "See Other"
    NOT_MODIFIED = 304, "Not Modified"
    USE_PROXY = 305, "Use Proxy"
    TEMPORARY_REDIRECT = 307, "Temporary Redirect"
    PERMANENT_REDIRECT = 308, "Permanent Redirect"

    BAD_REQUEST = 400, "Bad Request"
    UNAUTHORIZED = 401, "Unauthorized"
    PAYMENT_REQUIRED = 402, "Payment Required"
    FORBIDDEN = 403, "Forbidden"
    NOT_FOUND = 404, "Not Found"
    METHOD_NOT_ALLOWED = 405, "Method Not Allowed"
    NOT_ACCEPTABLE = 406, "Not Acceptable"
    PROXY_AUTHENTICATION_REQUIRED = 407, "Proxy Authentication Required"
    REQUEST_TIMEOUT = 408, "Request Timeout"
    CONFLICT = 409, "Conflict"
    GONE = 410, "Gone"
    LENGTH_REQUIRED = 411, "Length Required"
    PRECONDITION_FAILED = 412, "Precondition Failed"
    PAYLOAD_TOO_LARGE = 413, "Payload Too Large"
    URI_TOO_LONG = 414, "URI Too Long"
    UNSUPPORTED_MEDIA_TYPE = 415, "Unsupported Media Type"
    RANGE_NOT_SATISFIABLE = 416, "Range Not Satisfiable"
    EXPECTATION_FAILED = 417, "Expectation Failed"
    IM_A_TEAPOT = 418, "I'm a teapot"
    MISDIRECTED_REQUEST = 421, "Misdirected Request"
    UNPROCESSABLE_ENTITY = 422, "Unprocessable Entity"
    LOCKED = 423, "Locked"
    FAILED_DEPENDENCY = 424, "Failed Dependency"
    TOO_EARLY = 425, "Too Early"
    UPGRADE_REQUIRED = 426, "Upgrade Required"
    PRECONDITION_REQUIRED = 428, "Precondition Required"
    TOO_MANY_REQUESTS = 429, "Too Many Requests"
    REQUEST_HEADER_FIELDS_TOO_LARGE = 431, "Request Header Fields Too Large"
    UNAVAILABLE_FOR_LEGAL_REASONS = 451, "Unavailable For Legal Reasons"

    INTERNAL_SERVER_ERROR = 500, "Internal Server Error"
    NOT_IMPLEMENTED = 501, "Not Implemented"
    BAD_GATEWAY = 502, "Bad Gateway"
    SERVICE_UNAVAILABLE = 503, "Service Unavailable"
    GATEWAY_TIMEOUT = 504, "Gateway Timeout"
    HTTP_VERSION_NOT_SUPPORTED = 505, "HTTP Version Not Supported"
    VARIANT_ALSO_NEGOTIATES = 506, "Variant Also Negotiates"
    INSUFFICIENT_STORAGE = 507, "Insufficient Storage"
    LOOP_DETECTED = 508, "Loop Detected"
    NOT_EXTENDED = 510, "Not Extended"
    NETWORK_AUTHENTICATION_REQUIRED = 511, "Network Authentication Required"

    def __init__(self, code, description):
        self._value_ = code
        self._description = description

    def __repr__(self):
        return f"<{self.__class__.__name__}.{self._value_}>"

    def __str__(self):
        return str(self.value)

    @property
    def description(self):
        return self._description

    @classmethod
    def by_code(cls, code: int) -> Self:
        for status in cls:
            if status.value == code:
                return status

        raise ValueError(f"No HTTPStatus found for code: {code}")


class HTTPMethod(Enum):
    """
    For backward compatibility because HTTPMethod class
    was added in Python 3.11...
    """

    CONNECT = "CONNECT", "Establish a connection to the server."
    DELETE = "DELETE", "Delete the resource."
    GET = "GET", "Retrieve the resource."
    HEAD = "HEAD", "Same as GET, but only retrieve the status line and header section."
    OPTIONS = "OPTIONS", "Describe the communication options for the resource."
    PATCH = "PATCH", "Apply partial modifications to a resource."
    POST = "POST", "Perform resource-specific processing with the request payload."
    PUT = "PUT", "Replace the resource with the request payload."
    TRACE = "TRACE", "Perform a message loop-back test along the path to the resource."

    def __init__(self, name: str, description: str):
        self._value_ = name
        self._description = description

    def __repr__(self):
        return f"<{self.__class__.__name__}.{self._value_}>"

    def __str__(self):
        return self._value_

    @property
    def description(self):
        return self._description

    @classmethod
    def by_name(cls, name: str) -> Self:
        name = name.upper()

        for http_method in cls:
            if http_method.value == name:
                return http_method

        raise ValueError(f"No HTTPMethod found for name: {name}")


class StrEnum(str, Enum):
    """
    For backward compatibility because StrEnum class
    was added in Python 3.11...
    """

    def __new__(cls, value, *args, **kwargs):
        if not isinstance(value, str):
            raise TypeError(f"{value} is not a string")

        obj = str.__new__(cls, value)
        obj._value_ = value
        return obj
