# -*- coding: utf-8 -*-

from __future__ import annotations

import asyncio
from typing import Any
from typing import Dict
from typing import Optional

try:
    from typing import Self

except ImportError:
    from typing_extensions import Self

try:
    from http import HTTPMethod

except ImportError:
    from core_https.utils import HTTPMethod

from aiohttp import (
    ClientResponse,
    ClientResponseError,
    ClientSession,
    ClientTimeout,
    TCPConnector,
)

from .base import IRequester


class AioHttpRequester(IRequester):
    """
    It uses `aiohttp` to make the requests.

    .. code-block:: python

        import aiohttp
        from core_https.requesters.aiohttp_ import AioHttpRequester
        from core_https.utils import HTTPMethod

        requester: AioHttpRequester = AioHttpRequester(raise_for_status=True)

        async def get():
            # This is optional as the client creates one session for you if not provided.
            session = aiohttp.ClientSession()

            try:
                response = await requester.request(
                    method=HTTPMethod.GET,
                    session=session,
                    url=url,
                    params={
                        "x-api-key": "..."
                    })

                return await response.text()

            except Exception as error:
                pass

            finally:
                await session.close()

        res = asyncio.run(get())
        print(res)
    ..
    """

    def __init__(
        self,
        session: Optional[ClientSession] = None,
        retries: Optional[int] = 3,
        **kwargs
    ) -> None:
        """
        :param session: The session to use for requests.
        :param retries: Retry strategy to apply. Pass zero (0) to avoid retries.
        """

        super().__init__(**kwargs)

        self._session = session
        self._session_lock = asyncio.Lock()
        self._owns_session = session is None
        self._timeout = ClientTimeout(total=self.timeout)
        self.retries = retries

    async def __aenter__(self) -> Self:
        await self._ensure_session()
        return self

    async def __aexit__(self, exc_type, exc_val, exc_tb) -> None:
        await self.close()

    @classmethod
    def engine(cls) -> str:
        return "aiohttp"

    async def _ensure_session(self) -> ClientSession:
        """ If the session doesn't exist, it creates it """

        if self._session is None:
            async with self._session_lock:
                if self._session is None:  # Double-check after acquiring lock...
                    self._session = ClientSession(
                        timeout=self._timeout,
                        connector=TCPConnector(
                            limit=self.connector_limit,
                            limit_per_host=self.connector_limit_per_host
                        ))

                    self._owns_session = True

        return self._session

    async def request(
        self,
        url: str,
        method: HTTPMethod = HTTPMethod.GET,
        session: Optional[ClientSession] = None,
        headers: Optional[Dict[str, Any]] = None,
        params: Optional[Dict[str, Any]] = None,
        timeout: Optional[float] = None,
        retries: Optional[int] = None,
        backoff_factor: Optional[int] = None,
        **kwargs
    ) -> ClientResponse:
        """
        It makes the request using the session (externally provided or created
        if required) and return the response...

        :returns: `aiohttp.ClientResponse` object.
        """

        session_ = session or await self._ensure_session()
        kwargs_ = kwargs.copy()

        if timeout is not None:
            kwargs_["timeout"] = ClientTimeout(total=timeout)

        retries = retries if retries is not None else self.retries
        attempts = 0

        backoff_factor = (
            backoff_factor
            if backoff_factor is not None
            else self.backoff_factor if self.backoff_factor is not None
            else 0.5
        )

        while True:
            attempts += 1

            try:
                response = await session_.request(
                    method=str(method),
                    url=url,
                    headers=headers,
                    params=params,
                    **kwargs_)

                if self.raise_for_status:
                    response.raise_for_status()

                return response

            except ClientResponseError as error:
                if attempts > retries:
                    self.raise_custom_exception(error.status, error.message)

                await asyncio.sleep(backoff_factor * attempts)

    async def close(self):
        if self._session and self._owns_session:
            await self._session.close()
            self._session = None
