# -*- coding: utf-8 -*-

from typing import Any, Dict, Optional
from unittest.mock import AsyncMock, Mock

from aiohttp import ClientResponseError
from multidict import CIMultiDict
from yarl import URL

from core_https.tests.base import BaseHttpTestCases
from core_https.tests.requests_ import BaseRequestsTestCases


class BaseAiohttpTestCases(BaseHttpTestCases):
    """ Base class for Test Cases related to HTTP requests using `aiohttp` """

    @classmethod
    def get_aiohttp_mock(
        cls,
        url: str = "https://example.com",
        method: str = "GET",
        json_response: Optional[Dict[str, Any]] = None,
        status: int = 200,
        headers: Optional[Dict[str, str]] = None,
        text_response: Optional[str] = None,
        content: Optional[bytes] = None,
        content_type: str = "application/json",
        charset: str = "utf-8",
        raise_for_status_exception: Optional[Exception] = None,
    ) -> Mock:
        """
        Create a mock for `aiohttp.ClientResponse` object
        for testing purposes...

        :param url: Response URL.
        :param method: HTTP method used.
        :param json_response: Dictionary to return from .json() method.
        :param status: HTTP status code (default: 200).
        :param headers: Response headers dictionary.
        :param text_response: Text content of response (auto-generated from json_response if not provided).
        :param content: Raw bytes content.
        :param content_type: Content-Type header value.
        :param charset: Character encoding.
        :param raise_for_status_exception: Exception to raise when .raise_for_status() is called.

        :returns: Mock object simulating aiohttp.ClientResponse.
        """

        mock = BaseRequestsTestCases.get_requests_mock()
        del mock.status_code

        mock.status = status
        mock.method = method
        mock.url = url
        mock.charset = charset
        mock.content_type = content_type

        if headers is None:
            headers = {
                "Content-Type": f"{content_type}; charset={charset}"
            }

        # Headers as CIMultiDict (case-insensitive)...
        mock.headers = CIMultiDict(headers)

        # Async methods - these need to be AsyncMock to allow `await`...
        mock.json = AsyncMock(return_value=json_response)
        mock.text = AsyncMock(return_value=text_response)
        mock.read = AsyncMock(return_value=content)

        # Context manager support...
        mock.__aenter__ = AsyncMock(return_value=mock)
        mock.__aexit__ = AsyncMock(return_value=None)

        mock.raise_for_status.return_value = None
        if raise_for_status_exception:
            mock.raise_for_status.side_effect = raise_for_status_exception

        # History and other attributes...
        mock.history = []
        mock.cookies = {}
        return mock

    @staticmethod
    def create_client_response_error(
        url="http://example.com",
        status: int = 404,
        message: str = "Not Found",
        history=None
    ) -> ClientResponseError:
        """ Simple way to create `ClientResponseError` with history """

        request_info = Mock()
        request_info.url = URL(url)
        request_info.method = "GET"
        request_info.headers = {}

        if history is None:
            history = []

        return ClientResponseError(
            request_info=request_info,
            history=tuple(history),
            status=status,
            message=message
        )
