# -*- coding: utf-8 -*-

import json
from typing import Any, Dict, Optional
from unittest.mock import Mock

from core_https.tests.base import BaseHttpTestCases


class BaseRequestsTestCases(BaseHttpTestCases):
    """ Base class for Test Cases related to HTTP requests using `requests` """
    
    @classmethod
    def get_requests_mock(
        cls,
        url: str = "https://example.com",
        encoding: str = "utf-8",
        headers: Optional[Dict[str, str]] = None,
        json_response: Optional[Dict[str, Any]] = None,
        text_response: Optional[str] = None,
        status_code: int = 200,
        content: Optional[bytes] = None,
        raise_for_status_exception: Optional[Exception] = None,
    ) -> Mock:
        """
        Create a mock requests.Response object for testing
        purposes...

        :param url: Response URL.
        :param encoding: Response encoding.
        :param headers: Response headers dictionary.
        :param json_response: Dictionary to return from .json() method.
        :param status_code: HTTP status code (default: 200).
        :param text_response: Text content of response (auto-generated from json_response if not provided).
        :param content: Raw bytes content.
        :param raise_for_status_exception: Exception to raise when .raise_for_status() is called.

        :returns: Mock object simulating `requests.Response`.
        """

        if json_response is None:
            json_response = {}

        if headers is None:
            headers = {
                "Content-Type": "application/json"
            }

        if text_response is None:
            text_response = json.dumps(json_response) if json_response else ""

        if content is None:
            content = text_response.encode(encoding)

        mock = Mock()

        mock.status_code = status_code
        mock.headers = headers
        mock.url = url
        mock.encoding = encoding
        mock.text = text_response
        mock.content = content

        mock.ok = status_code < 400
        mock.reason = cls.code_mapper.get(status_code)
        mock.json.return_value = json_response

        mock.raise_for_status.return_value = None
        if raise_for_status_exception:
            mock.raise_for_status.side_effect = raise_for_status_exception

        # Support for iter_content and iter_lines (common in streaming)...
        mock.iter_content.return_value = iter([content])
        mock.iter_lines.return_value = iter(text_response.splitlines())
        return mock
