# -*- coding: utf-8 -*-

import re
from abc import ABC
from abc import abstractmethod
from typing import Any, Dict, Optional

from core_mixins.interfaces.factory import IFactory

from core_https.exceptions import (
    AuthenticationException,
    AuthorizationException,
    InternalServerError,
    RateLimitException,
    RetryableException,
    ServiceException,
)

try:
    from http import HTTPMethod

except ImportError:
    from core_https.utils import HTTPMethod


class IRequester(IFactory, ABC):
    """ Base interface for all type of HTTP requesters """

    def __init__(
        self,
        encoding: str = "utf-8",
        raise_for_status: bool = False,
        retries: Optional[Any] = None,
        backoff_factor: Optional[int] = None,
        connector_limit: int = 100,
        connector_limit_per_host: int = 30,
        timeout: int = 10,
    ) -> None:
        """
        :param encoding: The encoding to use when decoding.
        :param raise_for_status: If True, `raise_for_status` will be executed.
        :param connector_limit: Maximum number of connections in pool.
        :param connector_limit_per_host: Maximum connections per host.
        :param timeout: How many seconds to wait for the server to send data.
        :param retries: Retry strategy to apply. Provide it here, to avoid passing it to each request.
        :param backoff_factor: Delay between successive retry attempts.
        """

        if timeout <= 0:
            raise ValueError("`timeout` must be positive!")

        if connector_limit <= 0:
            raise ValueError("`connector_limit` must be positive!")

        if connector_limit_per_host <= 0:
            raise ValueError("`connector_limit_per_host` must be positive!")

        if connector_limit_per_host > connector_limit:
            raise ValueError("`connector_limit_per_host` cannot exceed `connector_limit`!")

        self.backoff_factor = backoff_factor
        self.retries = retries

        self.encoding = encoding
        self.raise_for_status = raise_for_status
        self.timeout = timeout

        self.connector_limit = connector_limit
        self.connector_limit_per_host = connector_limit_per_host

    @classmethod
    def registration_key(cls) -> str:
        return cls.engine()

    @classmethod
    @abstractmethod
    def engine(cls) -> str:
        """ Must return the engine name like: `requests` or `urllib3` """

    @abstractmethod
    def request(
        self,
        url: str,
        method: HTTPMethod,
        headers: Optional[Dict[str, str]] = None,
        retries: Optional[Any] = None,
        backoff_factor: Optional[int] = None,
        **kwargs  # Each `engine`have its own attributes...
    ) -> Any:
        """
        Makes the request and returns the response.

        :param url: The url for the request.
        :param method: The method (verb) for the request.
        :param headers: The headers to add.

        :param retries:
            It defines the retry strategy. Pass `False` if you don't want to use a
            retry at all, otherwise a default one will be provided.

        :param backoff_factor: Delay between successive retry attempts.

        :param kwargs:
            Each concrete implementation will have its own attributes. Depending on
            the requester you are using, you can pass different attributes.
        """

    def _get_response_encoding(
        self,
        headers: Dict[str, str],
        default="utf-8",
    ) -> str:
        headers_ = {
            k.lower(): v
            for k, v in headers.items()
        }

        # First trying "charset" header directly (rare)...
        charset = headers_.get("charset")
        if charset:
            return charset.strip()

        # Then checking `Content-Type` for "charset="
        content_type = headers_.get("content-type", "")
        match = re.search(r"charset=([^\s;]+)", content_type, re.IGNORECASE)
        if match:
            return match.group(1).strip()

        return self.encoding or default

    @staticmethod
    def raise_custom_exception(status_code: int, details: str):
        """
        :raises: `ServiceException` for other 4XX status_code.
        :raises: `AuthenticationException` for status_code == 401.
        :raises: `AuthorizationException` for status_code == 403.
        :raises: `RateLimitException` for status_code == 429.
        :raises: `InternalServerError` for status_code >= 500.
        """

        error_cls = ServiceException

        if status_code == 401:
            error_cls = AuthenticationException

        elif status_code == 403:
            error_cls = AuthorizationException

        elif status_code == 429:
            error_cls = RateLimitException

        elif status_code in (429, 502, 503, 504):
            error_cls = RetryableException

        elif status_code >= 500:
            error_cls = InternalServerError

        raise error_cls(
            status_code=status_code,
            details=details,
        )
