# -*- coding: utf-8 -*-

from functools import wraps
from typing import Any, Dict, Optional
from unittest.mock import AsyncMock, Mock, patch

from .aiohttp_ import BaseAiohttpTestCases
from .requests_ import BaseRequestsTestCases
from .urllib3_ import BaseUrllib3TestCases


def patch_aiohttp(
    url: str = "https://example.com",
    method: str = "GET",
    json_response: Optional[Dict[str, Any]] = None,
    status: int = 200,
    headers: Optional[Dict[str, str]] = None,
    text_response: Optional[str] = None,
    content: Optional[bytes] = None,
    content_type: str = "application/json",
    charset: str = "utf-8",
    raise_for_status_exception: Optional[Exception] = None,
):
    """
    Decorator that patches `aiohttp.ClientSession._request` with
    an `AsyncMock` so it works with both async test functions
    and sync functions using asyncio.run().

    .. code-block:: python

        @patch_aiohttp(json_response={"id": 123}, status=201)
        def test_aiohttp_mock_json(self):
            async def test():
                async with aiohttp.ClientSession() as session:
                    async with session.get("https://example.com") as resp:
                        self.assertEqual(resp.status, 201)
                        self.assertEqual(await resp.json(), {"id": 123})

            asyncio.run(test())
    ..
    """

    def decorator(func):
        @wraps(func)
        def wrapper(*args, **kwargs):
            mock_response = BaseAiohttpTestCases.get_aiohttp_mock(
                url=url,
                method=method,
                json_response=json_response,
                status=status,
                headers=headers,
                text_response=text_response,
                content=content,
                content_type=content_type,
                charset=charset,
                raise_for_status_exception=raise_for_status_exception)

            with patch(
                target="aiohttp.client.ClientSession._request",
                new_callable=lambda: AsyncMock(return_value=mock_response)):

                return func(*args, **kwargs)

        return wrapper

    return decorator


def patch_requests(
    url: str = "https://example.com",
    encoding: str = "utf-8",
    headers: Optional[Dict[str, str]] = None,
    json_response: Optional[Dict[str, Any]] = None,
    text_response: Optional[str] = None,
    status_code: int = 200,
    content: Optional[bytes] = None,
    raise_for_status_exception: Optional[Exception] = None,
):
    """
    Decorator that patches requests.sessions.Session.request
    to return the result of BaseHttpTestCases.get_requests_mock()
    with the given parameters.

    .. code-block:: python

        @patch_requests(json_response={"id": 1})
        def test_get_request_mock_json(self):
            response = requests.get("https://example.com")
            self.assertEqual(response.json(), {"id": 1})
    ..
    """

    def decorator(func):
        @wraps(func)
        def wrapper(*args, **kwargs):
            with patch(
                target="requests.sessions.Session.request",
                return_value=BaseRequestsTestCases.get_requests_mock(
                    url=url,
                    encoding=encoding,
                    headers=headers,
                    json_response=json_response,
                    text_response=text_response,
                    status_code=status_code,
                    content=content,
                    raise_for_status_exception=raise_for_status_exception,
                )):

                return func(*args, **kwargs)

        return wrapper

    return decorator


def patch_urllib3(
    url: str = "https://example.com",
    method: str = "GET",
    status: int = 200,
    data: bytes = b'{"message": "success"}',
    headers: Optional[Dict[str, str]] = None,
    reason: Optional[str] = None,
    version: int = 11,
    preload_content: bool = True,
    decode_content: bool = True,
    version_string: str = "HTTP/1.1",
    original_response: Optional[Mock] = None,
    pool: Optional[Mock] = None,
    connection: Optional[Mock] = None,
    msg: Optional[Mock] = None,
    retries: Optional[Mock] = None,
    enforce_content_length: bool = False,
    with_json_attr: bool = True,
):
    """
    Decorator that patches urllib3.connectionpool.HTTPConnectionPool.urlopen
    to return the result of BaseHttpTestCases.get_urllib3_mock()
    with the given parameters.

    .. code-block:: python

        @patch_urllib3(status=404, data=b'{"error": "not found"}')
        def test_urllib3_mock_404(self):
            http = urllib3.PoolManager()
            response = http.request("GET", "https://example.com")
            self.assertEqual(response.status, 404)
            self.assertEqual(response.data, b'{"error": "not found"}')
    ..
    """

    def decorator(func):
        @wraps(func)
        def wrapper(*args, **kwargs):
            with patch(
                target="urllib3._request_methods.RequestMethods.request",
                return_value=BaseUrllib3TestCases.get_urllib3_mock(
                    url=url,
                    method=method,
                    status=status,
                    data=data,
                    headers=headers,
                    reason=reason,
                    version=version,
                    preload_content=preload_content,
                    decode_content=decode_content,
                    version_string=version_string,
                    original_response=original_response,
                    pool=pool,
                    connection=connection,
                    msg=msg,
                    retries=retries,
                    enforce_content_length=enforce_content_length,
                    with_json_attr=with_json_attr,
                )):

                return func(*args, **kwargs)

        return wrapper

    return decorator
