# -*- coding: utf-8 -*-

import json
from typing import Dict, Optional
from unittest.mock import Mock

from urllib3._collections import HTTPHeaderDict

from core_https.tests.base import BaseHttpTestCases


class BaseUrllib3TestCases(BaseHttpTestCases):
    """ Base class for Test Cases related to HTTP requests using `urllib3` """

    @classmethod
    def get_urllib3_mock(
        cls,
        url: str = "https://example.com",
        method: str = "GET",
        status: int = 200,
        data: bytes = b'{"message": "success"}',
        headers: Optional[Dict[str, str]] = None,
        reason: Optional[str] = None,
        version: int = 11,
        preload_content: bool = True,
        decode_content: bool = True,
        version_string = "HTTP/1.1",
        original_response: Optional[Mock] = None,
        pool: Optional[Mock] = None,
        connection: Optional[Mock] = None,
        msg: Optional[Mock] = None,
        retries: Optional[Mock] = None,
        enforce_content_length: bool = False,
        with_json_attr: bool = True
    ) -> Mock:
        """
        Returns a comprehensive mock HTTP response for urllib3.

        Args:
            status: HTTP status code.
            data: Response body (bytes).
            headers: Optional headers dictionary.
            reason: HTTP reason phrase.
            version: HTTP version (11 for HTTP/1.1, 20 for HTTP/2.0).
            version_string: HTTP version HTTP/1.1 or HTTP/2.0.
            original_response: Original response object.
            pool: Connection pool.
            preload_content: Whether content is preloaded.
            decode_content: Whether content should be decoded.
            connection: HTTP connection.
            msg: HTTP message object.
            retries: Retry configuration.
            enforce_content_length: Whether to enforce content-length.
            method: HTTP method used for request.
            url: Request URL.
            with_json_attr: To mimic previous versions.

        Returns:
            Mock urllib3.response.HTTPResponse object
        """

        if headers is None:
            headers = {"Content-Type": "application/json"}

        if reason is None:
            reason = cls.code_mapper.get(status)

        mock = Mock()

        mock.url = url
        mock._request_url = url
        mock.status = status

        mock.reason = reason
        mock.version_string = version_string
        mock.version = version

        mock.data = data if preload_content else None
        mock._body = data if preload_content else None
        mock.decode_content = decode_content

        # Headers - urllib3 uses HTTPHeaderDict
        header_dict = HTTPHeaderDict(headers)
        mock.headers = header_dict
        mock.msg = msg or Mock()

        # Request information
        mock.request_method = method

        # Connection and pool info...
        mock._original_response = original_response
        mock._pool = pool
        mock._connection = connection
        mock.retries = retries
        mock.enforce_content_length = enforce_content_length

        # Content handling flags...
        mock.closed = Mock(return_value=False)
        mock.readable = Mock(return_value=True)

        # Methods for reading content...
        def mock_read(amt=None, decode_content=None):
            """ Mock read method. """
            if preload_content:
                return data.decode() if decode_content else data
            else:
                # Simulate streaming read...
                res = data[:amt] if amt else data
                return res.decode() if decode_content else res

        def mock_read1(amt=None):
            """ Mock read1 method (reads up to amt bytes). """
            return mock_read(amt)

        def mock_readinto(b):
            """ Mock readinto method. """
            info = data[:len(b)]
            b[:len(info)] = info
            return len(info)

        def mock_readline(size=-1):
            lines = data.split(b"\n", 1)
            line = lines[0] + b"\n" if len(lines) > 1 else lines[0]
            return line[:size] if size > 0 else line

        def mock_readlines(hint=-1):
            return data.split(b"\n")

        # Assign read methods...
        mock.read = mock_read
        mock.read1 = mock_read1
        mock.readinto = mock_readinto
        mock.readline = mock_readline
        mock.readlines = mock_readlines

        # Stream methods
        def mock_stream(amt=2 ** 16, decode_content=None):
            if not data:
                return

            chunk_size = amt or 2 ** 16
            for i in range(0, len(data), chunk_size):
                yield data[i:i + chunk_size]

        mock.stream = mock_stream

        # Header methods...
        def mock_getheaders():
            return HTTPHeaderDict(header_dict)

        def mock_getheader(name, default=None):
            return header_dict.get(name, default)

        mock.getheaders = mock_getheaders
        mock.getheader = mock_getheader

        # Connection management
        def mock_close():
            """Mock close method."""
            mock.closed.return_value = True
            mock.readable.return_value = False

        def mock_release_conn():
            """Mock release_conn method."""
            pass

        mock.close = mock_close
        mock.release_conn = mock_release_conn

        # Context manager support
        mock.__enter__ = Mock(return_value=mock)
        mock.__exit__ = Mock(side_effect=mock_close())

        # Iterator support
        def mock_iter(self):
            """Support for iteration over response."""
            for line in data.split(b"\n"):
                yield line

        mock.__iter__ = mock_iter

        def mock_json():
            return json.loads(data.decode("utf-8"))

        mock.json = mock_json
        if not with_json_attr:
            delattr(mock, "json")

        return mock
