# -*- coding: utf-8 -*-

from aiohttp import ClientResponse
from aiolimiter import AsyncLimiter

from core_https.requesters.aiohttp_throttle import AioHttpThrottleRequester


class AioHttpRateLimitRequester(AioHttpThrottleRequester):
    """
    An HTTP requester that enforces a simple *rate limit* using
    :class:`aiolimiter.AsyncLimiter`. This class restricts how many
    requests may be *started* within a given time window. It does **not**
    limit concurrency; multiple requests may still run in parallel
    if they acquire permission within the same window.
    """

    def __init__(
        self,
        max_concurrency: int,
        max_rate: int,
        time_period: float,
        **kwargs
    ) -> None:
        """
        Initialize the rate-limited requester.

        :param max_rate: Maximum number of requests allowed within the time window.
        :param time_period: Duration (in seconds) of the rate-limiting window.
        :param kwargs: Additional keyword arguments passed to :class:`AioHttpRequester`.
        """

        if max_rate <= 0:
            raise ValueError("`max_rate` must be positive!")

        if time_period <= 0:
            raise ValueError("`time_period` must be positive!")

        super().__init__(max_concurrency=max_concurrency, **kwargs)

        self.max_rate = max_rate
        self.time_period = time_period

        self._limiter = AsyncLimiter(
            max_rate=self.max_rate,
            time_period=self.time_period,
        )

    async def request(self, *args, **kwargs) -> ClientResponse:
        """
        Execute an HTTP request subject to the configured rate limit. The
        coroutine will wait (non-blocking) until a rate slot becomes available,
        and then delegate the call to the base :class:`AioHttpRequester`.

        :return: The HTTP response object.
        :raises Exception: Any exception raised by the underlying HTTP client.
        """

        async with self._limiter:
            return await super().request(*args, **kwargs)
