def create_cpp_bindings(all_functions: dict, output_path: str):
    """Create the C++ bindings for exported functions.

    Args:
        all_functions (dict): Dictionary as produced by `parse_cpp_exports`.
        output_path (str): Path to store the output C++ bindings.

    Returns:
        A file is created at `output_path`. Nothing is returned.
    """

    all_function_names = list(all_functions.keys())
    all_function_names.sort()

    # Generating the C++ bindings for each function.
    with open(output_path, "w") as handle:
        handle.write("""/* DO NOT MODIFY: this is automatically generated by the ctypes-wrapper */

#include <cstring>
#include <stdexcept>

#ifdef _WIN32
#define PYAPI __declspec(dllexport)
#else
#define PYAPI
#endif

static char* copy_error_message(const char* original) {
    auto n = std::strlen(original);
    auto copy = new char[n + 1];
    std::strcpy(copy, original);
    return copy;
}""")

        externC = [
            """PYAPI void free_error_message(char** msg) {
    delete [] *msg;
}"""
        ]

        for k in all_function_names:
            restype, args = all_functions[k]
            
            types_only = [x.type.full_type for x in args]
            handle.write("\n\n" + restype.full_type + " " + k + "(" + ", ".join(types_only) + ");")

            all_args = [x.type.full_type + " " + x.name for x in args]
            all_args.append("int32_t* errcode")
            all_args.append("char** errmsg")

            names_only = [x.name for x in args]
            if restype.base_type == "void" and restype.pointer_level == 0:
                init_call = ""
                fun_call = k
                ret_call = ""
            else:
                init_call = "\n" + " " * 4 + restype.full_type + " output = "
                if restype.pointer_level:
                    init_call += "NULL;"
                else:
                    init_call += "0;"
                fun_call = "output = " + k
                ret_call = "\n" + " " * 4 + "return output;"

            externC.append("PYAPI " + restype.full_type + " py_" + k + "(" + ", ".join(all_args) + ") {" + init_call + """
    try {
        """ + fun_call + "(" + ", ".join(names_only) + """);
    } catch(std::exception& e) {
        *errcode = 1;
        *errmsg = copy_error_message(e.what());
    } catch(...) {
        *errcode = 1;
        *errmsg = copy_error_message("unknown C++ exception");
    }""" + ret_call + """
}"""
            )

        handle.write('\n\nextern "C" {')
        for cmd in externC:
            handle.write("\n\n" + cmd)
        handle.write("\n\n}\n")

    return
