### SOUNDFONT DOWNLOAD HELPERS

from pathlib import Path                # using Path to disambiguate between Windows and Unix path schemes
from platformdirs import user_data_dir
from pooch import retrieve              # secure file download
import os

APP = "CreativePython"
ORG = "CofC"

SOUNDFONT_NAME = "default.sf2"
CACHE_DIR = Path(user_data_dir(APP, ORG)) / "SoundFonts"
SOUNDFONT_PATH = CACHE_DIR / SOUNDFONT_NAME

SF2_URL = "https://www.dropbox.com/s/xixtvox70lna6m2/FluidR3%20GM2-2.SF2?dl=1"
SF2_SHA256 = "2ae766ab5c5deb6f7fffacd6316ec9f3699998cce821df3163e7b10a78a64066"

def find_soundfont(path=None):
   """
   Finds a soundfont 'default.sf2' in the cache, and returns its location.
   explicit can be another path that may have the soundfont needed.
   """
   candidates = []  # possible soundfont paths
   soundfont = None # actual soundfont path

   # was a path sent as an argument?
   if path:
      candidates.append(Path(path))
   
   # is there a PATH variable?
   env = os.getenv("CREATIVEPYTHON_SOUNDFONT")
   if env:
      candidates.append(Path(env))

   # CreativePython default sondfont directory
   candidates += [SOUNDFONT_PATH, Path.home() / "SoundFonts" / SOUNDFONT_NAME]

   # set the first confirmed soundfont
   for c in candidates:
      if c and c.exists() and soundfont is None:
         soundfont = str(c)

   return soundfont


def download_soundfont(destination):
   """Downloads a default soundfont (FluidR3 GM2-2) and saves it to the user's cache."""
   destination.parent.mkdir(parents=True, exist_ok=True)  # create destination directory
   path = retrieve(                                       # download soundfont
      url=SF2_URL,
      known_hash=f"sha256:{SF2_SHA256}",
      progressbar=True,
      fname=destination.name,
      path=str(destination.parent)
   )
   return Path(path)                                      # return the downloaded file's path


### MAIN METHODS

def run():
   # Check for soundfont file
   sf = find_soundfont()

   # Attempt to download missing soundfont, if needed
   if not sf:
      print("CreativePython setup warning: No soundfont found.")
      auto = os.getenv("CP_AUTO_DOWNLOAD") == "1"

      # ask user for permission to download soundfont
      if auto or input("Download the default soundfont now? [Y/n] ").strip().lower() in ("", "y", "yes"):
         path = download_soundfont()
         print(f"Downloaded to: {path}")

      # otherwise, instruct them how to add their own
      else:
         print(f"\nPlace your .sf2 soundfont at:\n   {SOUNDFONT_PATH}\nOr set CREATIVEPYTHON_SOUNDFONT to its location in your PATH.")
         print("\n(see https://pypi.org/project/CreativePython/ for more details)")


def playNote():
   from music import Note, Play, C4, HN  # can't use * within function, so naming needed pieces

   note = Note(C4, HN)        # create a middle C half note
   Play.midi(note)            # and play it!