
class Measurement:
   """
   This class encapsulates components of measurements generated by metrics.
   It uses a dictionary (Hashtable in Java).

   author Bill Manaris, Tim Hirzel, Luca Pellicoro
   version 1.0, June, 2006
   """

   def __init__(self, metricName, distribution, numHigherOrder):
      """
      Constructs a Measurement object used to hold results from a Metric (base and higher-order values).

      param metricName  holds the name of the metric that calculated these values
      param distribution  holds this Metric's distribution (for now, BY_RANK or BY_SIZE)
      param numHigherOrder  holds number of higher-order metrics (0 means only base metric)
      """
      self.metricName = metricName
      self.distribution = distribution
      self.numHigherOrder = numHigherOrder + 1 # plus one because of level zero
      self.capacity = self.numHigherOrder * 4  # for every order there are 4 values
      self.hashtable = {} # Measurement is really a wrapper to Hashtable (dict)
      self.keys = self.keysToArray()           # extract Measurement keys (once, for efficiency)

   def add(self, level, typeStr, slopeOrR2, value):
      """
      Adds values from a single higher-order metric.

      param level  holds the level of the metric (0 means base)
      param typeStr  holds the type of the metric (for now, either "" meaning regular, or "_LocalVariability")
      param slopeOrR2  holds the label of the value (for now, "slope" or "r2")
      param value  holds the value to be added
      """
      assert slopeOrR2 == "slope" or slopeOrR2 == "r2", "slopeOrR2 must be either slope or r2"
      assert level <= self.numHigherOrder, f"level must be inferior to the number of high order which is {self.numHigherOrder} instead it was {level}"

      # Java: hashtable.put(metricName + "_" + level + type + "_Slope", value);
      # Note: The Java code constructs key with "Slope" or "R2" capitalized, but asserts "slope" or "r2" lowercase.

      suffix = "Slope" if slopeOrR2 == "slope" else "R2"
      key = f"{self.metricName}_{level}{typeStr}_{suffix}"
      self.hashtable[key] = value

   def keysToArray(self):
      """
      Creates all the keys and stores them into a list (String array in Java)
      """
      result = []
      level = 0

      # Java loop: for (int index = 0; index < result.length ; index = index + 4)
      # result length is self.capacity.
      # Each iteration adds 4 keys for current level.

      for _ in range(0, self.capacity, 4):
         result.append(f"{self.metricName}_{level}_Slope")
         result.append(f"{self.metricName}_{level}_R2")
         result.append(f"{self.metricName}_{level}_LocalVariability_Slope")
         result.append(f"{self.metricName}_{level}_LocalVariability_R2")
         level += 1

      return result

   def valuesToArray(self):
      """
      Will get the keys from keysToArray() and retrieve the corresponding values.
      """
      result = []
      for key in self.keys:
         # Java: (Double) this.hashtable.get(this.keys[index])
         # In Python, dict.get returns None if missing. Java would return null, casting to Double might throw or be null.
         # The method seems to assume values exist.
         val = self.hashtable.get(key)
         # If val is None, this might cause issues downstream if floats are expected.
         # For now we preserve behavior.
         result.append(val)
      return result

   def init(self):
      """
      To keep reusing the same measurement with different songs
      """
      self.hashtable.clear()

   def getSize(self):
      """
      Get the total number of values contained in the measurement object
      """
      return self.capacity

   def put(self, key, value):
      """
      For testing purposes
      """
      self.hashtable[key] = value

   def __str__(self):
      """
      For testing purposes
      """
      return str(self.hashtable)

   def get(self, key):
      """
      For testing purposes
      """
      return self.hashtable.get(key)
