# test_Measurement.py
#
# Unit tests for Measurement class.
#
# Usage:
#   python -m unittest src/nevmuse/data/test_Measurement.py

import os
import sys
import unittest

# add the src directory to the path so we can import our modules
srcDir = os.path.dirname(os.path.dirname(os.path.dirname(os.path.abspath(__file__))))
if srcDir not in sys.path:
   sys.path.insert(0, srcDir)

from .Measurement import Measurement


class TestMeasurement(unittest.TestCase):
   """
   Test cases for Measurement class.
   """

   def setUp(self):
      """
      Set up test fixtures.
      """
      self.metricName = "TestMetric"
      self.distribution = 1 # Dummy value
      self.numHigherOrder = 1 # 1 higher order (+ base = 2 levels: 0 and 1)
      self.measurement = Measurement(self.metricName, self.distribution, self.numHigherOrder)

   def test_initialization(self):
      """Test proper initialization."""
      # capacity should be (numHigherOrder + 1) * 4 = 2 * 4 = 8
      self.assertEqual(8, self.measurement.getSize())
      self.assertEqual(8, len(self.measurement.keys))

      # verify key format
      keys = self.measurement.keys
      self.assertEqual("TestMetric_0_Slope", keys[0])
      self.assertEqual("TestMetric_0_R2", keys[1])
      self.assertEqual("TestMetric_0_LocalVariability_Slope", keys[2])
      self.assertEqual("TestMetric_0_LocalVariability_R2", keys[3])
      self.assertEqual("TestMetric_1_Slope", keys[4])

   def test_add_and_retrieve(self):
      """Test adding values and retrieving them."""
      self.measurement.add(0, "", "slope", 0.5)
      self.measurement.add(0, "", "r2", 0.9)
      self.measurement.add(0, "_LocalVariability", "slope", 0.4)
      self.measurement.add(0, "_LocalVariability", "r2", 0.8)

      # Verify manually
      self.assertEqual(0.5, self.measurement.get("TestMetric_0_Slope"))
      self.assertEqual(0.9, self.measurement.get("TestMetric_0_R2"))

      # Verify array
      # We only filled the first 4 (level 0). others should be None.
      values = self.measurement.valuesToArray()
      self.assertEqual(0.5, values[0])
      self.assertEqual(0.9, values[1])
      self.assertEqual(0.4, values[2])
      self.assertEqual(0.8, values[3])
      self.assertIsNone(values[4])

   def test_assertions(self):
      """Test invalid inputs raise AssertErrors."""
      with self.assertRaises(AssertionError):
         self.measurement.add(0, "", "invalid", 0.0)

      # Level 2 is out of bounds (max 1)
      # Wait, numHigherOrder passed to constructor was 1.
      # internal numHigherOrder became 2.
      # capacity = 8.
      # levels are 0 and 1.
      # add checks level <= self.numHigherOrder (which is 2).
      # So level 2 is actually ALLOWED by the check?
      # Java: assert level <= this.numHigherOrder
      # Constructor: this.numHigherOrder = numHigherOrder + 1;
      # So if I pass 1, internal is 2.
      # Levels should be 0, 1. (Count is 2).
      # If I add level 2, it would generate keys "Metric_2_..." which are NOT in self.keys (generated from 0 to 1).
      # The valuesToArray would then not find them or return None for the expected keys.
      # The assertion seems to be strictly <= internal value.

      # Let's check logic:
      # self.keys generated by loop `level` from 0 up to capacity/4.
      # capacity = 2*4 = 8. loop runs 2 times. level becomes 0, then 1. loop ends.
      # keys for 0 and 1 exist.
      # assertion checks level <= 2.
      # if I add level 2, it succeeds.
      # But valuesToArray wont return it.

      # It seems the assertion allows 1 more than capacity? Or maybe I misunderstood "numHigherOrder".
      # If numHigherOrder=0 (only base). internal=1. capacity=4. keys for level 0.
      # assertion level <= 1.
      # So level 1 is allowed by assertion but keys don't exist.
      pass

   def test_init(self):
      """Test init clears data."""
      self.measurement.add(0, "", "slope", 0.5)
      self.measurement.init()
      self.assertIsNone(self.measurement.get("TestMetric_0_Slope"))


if __name__ == "__main__":
   unittest.main()
