# AUTOGENERATED! DO NOT EDIT! File to edit: ../nbs/01_companies.ipynb.

# %% auto 0
__all__ = ['reformat_company_data', 'search']

# %% ../nbs/01_companies.ipynb 3
from . import core
from .core import set_headers
import pandas as pd
import pytz
from tqdm import tqdm

# %% ../nbs/01_companies.ipynb 4
def reformat_company_data(company_data:list, #Returned company data as list of dictionaries,
                          cf_fields:list = [], # list of custom fields on companies you would like to include
):
    """Clean Company Data

    Function to reformat/clean the raw company data recieved from Copper in its JSON format
    and return the result as a normalized diconary. This then can be used to make a pandas.DataFrame.

    Cleaning:
    - Change columns with 'date' in name from unix timestamp to human readable data (central time)
    - Change value in custom field columns from option id to the name of the otion
    - Discarding columns that are not desired using Custom_Fields list.

    Parameters:
      company_data:
        Raw data for an invdividual center in a JSON/dictonary format
    
    Returns:
      Company data as normalized dictonary, which can be made into a pandas.DataFrame
    """
    core.prc_cf_dicts()

    global custom_fields
    custom_fields = core.get_global_var('custom_fields')
    custom_fields_dict = core.get_global_var('custom_fields_dict')
    

    def clean_date(date):
        try:
            if date is None:
                return None  # Handle None values (empty rows)

            if isinstance(date, int) and len(str(date)) >= 6 and len(str(date)) <= 10:
                # If it's an integer with <= 10 digits, assume it's a Unix timestamp
                ct_timezone = pytz.timezone('US/Central')
                new_date = pd.to_datetime(date, unit='s',utc=True).tz_convert(ct_timezone)
                return new_date
            else:
                return date
        except Exception as e:
            return date

    native_items = ['id', 'name', 'address', 'assignee_id', 'contact_type_id']

    output_dict = {}

    for item in native_items:
        output_dict[item] = company_data.get(item, None)
    
    custom_field_data = company_data['custom_fields']

    for dict_item in custom_field_data:
        item_id = dict_item['custom_field_definition_id']
        item_name = custom_fields_dict.get(item_id, None)
        data_type = custom_fields[item_id].get('data_type')

        if item_name not in cf_fields or item_name is None:
            continue
        elif item_name is not None and ('options' in list(custom_fields[item_id].keys())):
            item_value = dict_item['value']
            value_name = core.cf_option_name(item_id,item_value)
            output_dict[item_name] = value_name
        elif data_type == 'Date':
            cleaned_date = clean_date(dict_item['value'])
            output_dict[item_name] = cleaned_date
        else:
            item_value = dict_item['value']
            output_dict[item_name] = item_value

    return output_dict

def search(search_params:dict={}, #search params for standard fields
            cf_search:list=[],#search params for custom fields
            clean_data:bool = True, #Whether to clean results or not
            cf_fields:list = [], # list of custom fields on companies you would like to include
            drop_cols:list = None, # Columns to drop
            )->pd.DataFrame:
    """Get Company Data

    Function to search copper companies, and return the resulting data.

    Parameters:
      search_params: dict
        dict of search parameters realted to native, copper fields (assignee id, address, etc.)
      
      cf_search: list[dict]

        list of dictonaires for search paramenters related to custom fields in Copper. Dictonary
        should include the `custom_field_definition_id` and the desired `value` (can be list of multiple values)
        
        Example of a dictionary in the list:
        {"custom_field_definition_id": 588393,"value": [1806073,1824709,1806067]}]

        Examle use:
        params = [{"custom_field_definition_id": 588393,"value": [1806073]}]
        test= = get_company_data(cf_search = params)
      
      cf_fields: list
        list of copper custom fields you would like to be returned  
      
      clean_data: bool
        Whether data is cleaned prior to return

      drop_cols: list
        list of columns to drop, not include, in the returned dataframe. Names must match the copper field names exactly.
    
    Returns:
      pandas.DataFrame
    """

    global copper_headers

    if "copper_headers" not in globals() and core.get_global_var('copper_headers') is None:
        raise ValueError('header information must be set with set_headers(). \nSee help(core.set_headers) for more info.')
    
    if "copper_headers" not in globals() and core.get_global_var('copper_headers') is not None:
        copper_headers = core.get_global_var('copper_headers')
    
    total_pages = page = 1
    combined_results = []
    
    Sess = core.get_session(copper_headers)

    while page <= total_pages:
        page_params = {
            "page_size": 100,
            "page_number": page,
            }
        
        if search_params != {}: page_params.update(search_params)

        if cf_search != {}:
            cf_addition = {"custom_fields":cf_search}
            page_params.update(cf_addition) 

        result = Sess.post('https://api.copper.com/developer_api/v1/companies/search',json=page_params)
    
        if result.status_code == 200:
            total_pages = (int(result.headers['X-PW-TOTAL'])//100)+1
            
            # Creatig Progress Bar:
            if page == 1: progress_bar = tqdm(total=total_pages,desc='Searching Copper')
            progress_bar.update(1)  # Update the progress bar
            
            result_json = result.json()
            combined_results.extend(result_json)
            page +=1

        else:
            print(f"Issue with page {page}. Stopping.")
            break

    progress_bar.close()  # Close the progress bar when done
 
    # Cleaning results
    if clean_data:
        print('\nData Recieved. Cleaning results:')
        cleaned_data = []

        for idx in range(len(combined_results)):
            results = reformat_company_data(combined_results[idx],cf_fields)
            cleaned_data.append(results)

        cleaned_data_df = pd.DataFrame(cleaned_data)
        print('Returning cleaned results')
        if isinstance(drop_cols,list): 
            return cleaned_data_df.drop(columns=drop_cols,inplace=True)
        else: return cleaned_data_df 
    else:
        print('Returning raw results')
        return combined_results
