import pytest
from pydantic import ValidationError

from cuenca_validations.types.enums import SessionType
from cuenca_validations.types.requests import (
    SessionRequest,
    UserTOSAgreementRequest,
)
from cuenca_validations.typing import DictStrAny


@pytest.mark.parametrize('environment', ['stage', 'sandbox', 'api'])
def test_file_cuenca_url(environment: str) -> None:
    request_data: DictStrAny = dict(
        tos_id='TS67dcae8e74e81bba5a77bf47',
        location=(19.432607, -99.133209),
        signature_image_url=(
            f'https://{environment}.cuenca.com/files/EFQL8_ohvoRp-PkOTYgvQYFA'
        ),
    )
    utos = UserTOSAgreementRequest(**request_data)
    assert utos.signature_image_url is not None
    assert utos.signature_image_url.file_id == 'EFQL8_ohvoRp-PkOTYgvQYFA'


def test_file_cuenca_url_invalid() -> None:
    request_data: DictStrAny = dict(
        tos_id='TS67dcae8e74e81bba5a77bf47',
        location=(19.432607, -99.133209),
        signature_image_url=(
            'https://cuenca.com/files/EFQL87ohvoRp-PkOTYgvQYFA/invalid'
        ),
    )
    with pytest.raises(ValidationError):
        UserTOSAgreementRequest(**request_data)


def test_session_request_without_metadata():
    request_data = {
        'user_id': 'USWqY5cvkISJOxHyEKjAKf8w',
        'type': 'session.curp_validation',
    }
    request = SessionRequest(**request_data)
    assert request.type == SessionType.curp_validation
    assert request.user_id == 'USWqY5cvkISJOxHyEKjAKf8w'


def test_session_request_with_metadata():
    request_data = {
        'user_id': 'USWqY5cvkISJOxHyEKjAKf8w',
        'type': 'session.metamap_verification',
        'metadata': {'verification_id': 'some_verification'},
    }
    request = SessionRequest(**request_data)
    assert request.type == SessionType.metamap_verification
    assert request.user_id == 'USWqY5cvkISJOxHyEKjAKf8w'
    assert request.metadata
    assert request.metadata['verification_id'] == 'some_verification'


def test_metamap_session_request_without_metadata():
    request_data = {
        'user_id': 'USWqY5cvkISJOxHyEKjAKf8w',
        'type': 'session.metamap_verification',
    }

    with pytest.raises(ValidationError) as exception:
        SessionRequest(**request_data)

    assert 'Metadata expected for this session' in str(exception)
