# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['cyclonedx',
 'cyclonedx.exception',
 'cyclonedx.model',
 'cyclonedx.output',
 'cyclonedx.parser',
 'cyclonedx.utils']

package_data = \
{'': ['*'], 'cyclonedx': ['schema/*', 'schema/ext/*']}

install_requires = \
['packageurl-python>=0.9',
 'requirements_parser>=0.2',
 'setuptools>=47.0.0',
 'toml>=0.10.0,<0.11.0',
 'types-setuptools>=57.0.0',
 'types-toml>=0.10.0,<0.11.0']

extras_require = \
{':python_version < "3.8"': ['importlib-metadata>=3.4',
                             'typing-extensions>=3.10,<4.0']}

setup_kwargs = {
    'name': 'cyclonedx-python-lib',
    'version': '0.12.2',
    'description': 'A library for producing CycloneDX SBOM (Software Bill of Materials) files.',
    'long_description': '# Python Library for generating CycloneDX\n\n[![shield_gh-workflow-test]][link_gh-workflow-test]\n[![shield_pypi-version]][link_pypi]\n[![shield_license]][license_file]  \n[![shield_website]][link_website]\n[![shield_slack]][link_slack]\n[![shield_groups]][link_discussion]\n[![shield_twitter-follow]][link_twitter]\n\n----\n\nThis CycloneDX module for Python can generate valid CycloneDX bill-of-material document containing an aggregate of all\nproject dependencies.\n\nThis module is not designed for standalone use.  \nIf you\'re looking for a CycloneDX tool to run to generate (SBOM) software bill-of-materials documents, why not checkout: [CycloneDX Python][cyclonedx-python]\n\nAdditionally, the following tool can be used as well (and this library was written to help improve it) [Jake][jake].\n\nAdditionally, you can use this module yourself in your application to programmatically generate SBOMs.\n\nCycloneDX is a lightweight BOM specification that is easily created, human-readable, and simple to parse.\n\n## Installation\n\nInstall from pypi.org as you would any other Python module:\n\n```shell\npip install cyclonedx-python-lib\n```\n\n## Architecture\n\nThis module break out into three key areas:\n\n1. **Parser**: Use a parser that suits your needs to automatically gather information about your environment or\n   application\n2. **Model**: Internal models used to unify data from different parsers\n3. **Output**: Choose and configure an output which allows you to define output format as well as the CycloneDX schema\n   version\n\n### Parsing\n\nYou can use one of the parsers to obtain information about your project or environment. Available parsers:\n\n| Parser | Class / Import | Description |\n| ------- | ------ | ------ |\n| CondaListJsonParser | `from cyclonedx.parser.conda import CondaListJsonParser` | Parses input provided as a `str` that is output from `conda list --json` |\n| CondaListExplicitParser | `from cyclonedx.parser.conda import CondaListExplicitParser` | Parses input provided as a `str` that is output from `conda list --explicit` or `conda list --explicit --md5` |\n| Environment | `from cyclonedx.parser.environment import EnvironmentParser` | Looks at the packaged installed in your current Python environment. |\n| PipEnvParser | `from cyclonedx.parser.pipenv import PipEnvParser` | Parses `Pipfile.lock` content passed in as a string. |\n| PipEnvFileParser | `from cyclonedx.parser.pipenv import PipEnvFileParser` | Parses the `Pipfile.lock` file at the supplied path. |\n| PoetryParser | `from cyclonedx.parser.poetry import PoetryParser` | Parses `poetry.lock` content passed in as a string. |\n| PoetryFileParser | `from cyclonedx.parser.poetry import PoetryFileParser` | Parses the `poetry.lock` file at the supplied path. |\n| RequirementsParser | `from cyclonedx.parser.requirements import RequirementsParser` | Parses a multiline string that you provide that conforms to the `requirements.txt` [PEP-508] standard. |\n| RequirementsFileParser | `from cyclonedx.parser.requirements import RequirementsFileParser` | Parses a file that you provide the path to that conforms to the `requirements.txt` [PEP-508] standard. |\n\n#### Example\n\n```py\nfrom cyclonedx.parser.environment import EnvironmentParser\n\nparser = EnvironmentParser()\n```\n\n#### Notes on Requirements parsing\n\nCycloneDX software bill-of-materials require pinned versions of requirements. If your `requirements.txt` does not have\npinned versions, warnings will be recorded and the dependencies without pinned versions will be excluded from the\ngenerated CycloneDX. CycloneDX schemas (from version 1.0+) require a component to have a version when included in a\nCycloneDX bill of materials (according to schema).\n\nIf you need to use a `requirements.txt` in your project that does not have pinned versions an acceptable workaround\nmight be to:\n\n```shell\npip install -r requirements.txt\npip freeze > requirements-frozen.txt\n```\n\nYou can then feed in the frozen requirements from `requirements-frozen.txt` _or_ use the `Environment` parser one you\nhave `pip install`ed your dependencies.\n\n### Modelling\n\nYou can create a BOM Model from either a Parser instance or manually using the methods avaialbel directly on the `Bom` class.\n\nThe model also supports definition of vulnerabilities for output using the CycloneDX schema extension for\n[Vulnerability Disclosures](https://cyclonedx.org/use-cases/#vulnerability-disclosure) as of version 0.3.0.\n\n**Note:** Known vulnerabilities associated with Components can be sourced from various data sources, but this library\nwill not source them for you. Perhaps look at [Jake][jake] if you\'re interested in this.\n\n#### Example from a Parser\n\n```py\nfrom cyclonedx.model.bom import Bom\nfrom cyclonedx.parser.environment import EnvironmentParser\n\nparser = EnvironmentParser()\nbom = Bom.from_parser(parser=parser)\n```\n\n### Generating Output\n\nOnce you have an instance of a `Bom` you can produce output in either `JSON` or `XML` against any of the supporting CycloneDX schema versions as you require.\n\nWe provide two helper methods:\n\n* Output to string (for you to do with as you require)\n* Output directly to a filename you provide\n\n#### Example as JSON\n\n```py\nfrom cyclonedx.output import get_instance, OutputFormat\n\noutputter = get_instance(bom=bom, output_format=OutputFormat.JSON)\noutputter.output_as_string()\n```\n\n#### Example as XML\n\n```py\nfrom cyclonedx.output import get_instance, SchemaVersion\n\noutputter = get_instance(bom=bom, schema_version=SchemaVersion.V1_2)\noutputter.output_to_file(filename=\'/tmp/sbom-v1.2.xml\')\n```\n\n## Library API Documentation\n\nThe Library API Documentation is available online at [https://cyclonedx.github.io/cyclonedx-python-lib/](https://cyclonedx.github.io/cyclonedx-python-lib/).\n\n## Schema Support\n\nThis library is a work in progress and complete support for all parts of the CycloneDX schema will come in future releases.\n\nHere is a summary of the parts of the schema supported by this library:\n\n_Note: We refer throughout using XPath, but the same is true for both XML and JSON output formats._\n\n<table width="100%">\n   <thead>\n      <tr>\n         <th>XPath</th>\n         <th>Support v1.3</th>\n         <th>Support v1.2</th>\n         <th>Support v1.1</th>\n         <th>Support v1.0</th>\n         <th>Notes</th>\n      </tr>\n   </thead>\n   <tbody>\n      <tr>\n         <td><code>/bom</code></td>\n         <td>Y</td><td>Y</td><td>Y</td><td>Y</td>\n         <td>\n            This is the root element and is supported with all it\'s defined attributes.\n         </td>\n      </tr>\n      <tr>\n         <td><code>/bom/metadata</code></td>\n         <td>Y</td><td>Y</td><td>N/A</td><td>N/A</td>\n         <td>\n            <code>timestamp</code> and <code>tools</code> are currently supported \n         </td>\n      </tr>\n      <tr>\n         <td><code>/bom/components</code></td>\n         <td>Y</td><td>Y</td><td>Y</td><td>Y</td>\n         <td>&nbsp;</td>\n      </tr>\n      <tr>\n         <th colspan="6"><strong><code>/bom/components/component</code></strong></th>\n      </tr>\n      <tr>\n         <td><code>./author</code></td>\n         <td>Y</td><td>Y</td><td>N/A</td><td>N/A</td>\n         <td>&nbsp;</td>\n      </tr>\n      <tr>\n         <td><code>./name</code></td>\n         <td>Y</td><td>Y</td><td>Y</td><td>Y</td>\n         <td>&nbsp;</td>\n      </tr>\n      <tr>\n         <td><code>./version</code></td>\n         <td>Y</td><td>Y</td><td>Y</td><td>Y</td>\n         <td>&nbsp;</td>\n      </tr>\n      <tr>\n         <td><code>./purl</code></td>\n         <td>Y</td><td>Y</td><td>Y</td><td>Y</td>\n         <td>&nbsp;</td>\n      </tr>\n      <tr>\n         <td><code>./externalReferences</code></td>\n         <td>Y</td><td>Y</td><td>Y</td><td>N/A</td>\n         <td>Not all Parsers have this information. It will be populated where there is information available.</td>\n      </tr>\n      <tr>\n         <td><code>./hashes</code></td>\n         <td>Y</td><td>Y</td><td>Y</td><td>Y</td>\n         <td>\n            These are supported when programmatically creating a <code>Bom</code> - these will not currently be \n            automatically populated when using a <code>Parser</code>.\n         </td>\n      </tr>\n   </tbody>\n</table>\n\n### Notes on Schema Support\n\n* N/A is where the CycloneDX standard does not include this\n* If the table above does not refer to an element, it is not currently supported\n\n## Python Support\n\nWe endeavour to support all functionality for all [current actively supported Python versions](https://www.python.org/downloads/).\nHowever, some features may not be possible/present in older Python versions due to their lack of support.\n\n## Changelog\n\nSee our [CHANGELOG][chaneglog_file].\n\n## Contributing\n\nFeel free to open issues, bugreports or pull requests.  \nSee the [CONTRIBUTING][contributing_file] file for details.\n\n## Copyright & License\n\nCycloneDX Python Lib is Copyright (c) OWASP Foundation. All Rights Reserved.  \nPermission to modify and redistribute is granted under the terms of the Apache 2.0 license.  \nSee the [LICENSE][license_file] file for the full license.\n\n[cyclonedx-python]: https://github.com/CycloneDX/cyclonedx-python\n[jake]: https://github.com/sonatype-nexus-community/jake\n\n[license_file]: https://github.com/CycloneDX/cyclonedx-python-lib/blob/master/LICENSE\n[chaneglog_file]: https://github.com/CycloneDX/cyclonedx-python-lib/blob/master/CHANGELOG.md\n[contributing_file]: https://github.com/CycloneDX/cyclonedx-python-lib/blob/master/CONTRIBUTING.md\n\n[shield_gh-workflow-test]: https://img.shields.io/github/workflow/status/CycloneDX/cyclonedx-python-lib/Python%20CI/main?logo=GitHub&logoColor=white "build"\n[shield_pypi-version]: https://img.shields.io/pypi/v/cyclonedx-python-lib?logo=Python&logoColor=white&label=PyPI "PyPI"\n[shield_license]: https://img.shields.io/github/license/CycloneDX/cyclonedx-python-lib "license"\n[shield_website]: https://img.shields.io/badge/https://-cyclonedx.org-blue.svg "homepage"\n[shield_slack]: https://img.shields.io/badge/slack-join-blue?logo=Slack&logoColor=white "slack join"\n[shield_groups]: https://img.shields.io/badge/discussion-groups.io-blue.svg "groups discussion"\n[shield_twitter-follow]: https://img.shields.io/badge/Twitter-follow-blue?logo=Twitter&logoColor=white "twitter follow"\n[link_gh-workflow-test]: https://github.com/CycloneDX/cyclonedx-python-lib/actions/workflows/poetry.yml?query=branch%3Amain\n[link_pypi]: https://pypi.org/project/cyclonedx-python-lib/\n[link_website]: https://cyclonedx.org/\n[link_slack]: https://cyclonedx.org/slack/invite\n[link_discussion]: https://groups.io/g/CycloneDX\n[link_twitter]: https://twitter.com/CycloneDX_Spec\n\n[PEP-508]: https://www.python.org/dev/peps/pep-0508/\n',
    'author': 'Paul Horton',
    'author_email': 'phorton@sonatype.com',
    'maintainer': 'Paul Horton',
    'maintainer_email': 'phorton@sonatype.com',
    'url': 'https://github.com/CycloneDX/cyclonedx-python-lib',
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'extras_require': extras_require,
    'python_requires': '>=3.6,<4.0',
}


setup(**setup_kwargs)
