# encoding: utf-8

# This file is part of CycloneDX Python Lib
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#     http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.
#
# SPDX-License-Identifier: Apache-2.0
# Copyright (c) OWASP Foundation. All Rights Reserved.

from enum import Enum
from typing import List, Union
from urllib.parse import ParseResult, urlparse

"""
This set of classes represents the data that is possible under the CycloneDX extension
schema for Vulnerabilties (version 1.0).

See: https://github.com/CycloneDX/specification/blob/master/schema/ext/vulnerability-1.0.xsd
"""


class VulnerabilitySourceType(Enum):
    """
    Represents <xs:simpleType name="scoreSourceType">
    """
    CVSS_V2 = 'CVSSv2'
    CVSS_V3 = 'CVSSv3'
    OWASP = 'OWASP Risk'
    OPEN_FAIR = 'Open FAIR'
    OTHER = 'Other'


class VulnerabilitySeverity(Enum):
    """
    Represents <xs:simpleType name="severityType">
    """
    NONE = 'None'
    LOW = 'Low'
    MEDIUM = 'Medium'
    HIGH = 'High'
    CRITICAL = 'Critical'
    UNKNOWN = 'Unknown'


class VulnerabilityRating:
    """
    Represents <xs:complexType name="scoreType">
    """
    _score_base: float
    _score_impact: float
    _score_exploitability: float
    _severity: VulnerabilitySeverity
    _method: VulnerabilitySourceType
    _vector: str

    def __init__(self, score_base: float = None, score_impact: float = None, score_exploitability=None,
                 severity: VulnerabilitySeverity = None, method: VulnerabilitySourceType = None,
                 vector: str = None):
        self._score_base = score_base
        self._score_impact = score_impact
        self._score_exploitability = score_exploitability
        self._severity = severity
        self._method = method
        self._vector = vector

    def get_base_score(self) -> float:
        return self._score_base

    def get_impact_score(self) -> float:
        return self._score_impact

    def get_exploitability_score(self) -> float:
        return self._score_exploitability

    def get_severity(self) -> Union[VulnerabilitySeverity, None]:
        return self._severity

    def get_method(self) -> Union[VulnerabilitySourceType, None]:
        return self._method

    def get_vector(self) -> Union[str, None]:
        return self._vector

    def has_score(self) -> bool:
        return (None, None, None) != (self._score_base, self._score_impact, self._score_exploitability)


class Vulnerability:
    """
    Represents <xs:complexType name="vulnerability">
    """
    _id: str
    _source_name: str
    _source_url: ParseResult
    _ratings: List[VulnerabilityRating] = []
    _cwes: List[int] = []
    _description: str = None
    _recommendations: List[str] = []
    _advisories: List[str] = []

    def __init__(self, id: str, source_name: str = None, source_url: str = None,
                 ratings: List[VulnerabilityRating] = [], cwes: List[int] = [], description: str = None,
                 recommendations: List[str] = [], advisories: List[str] = []):
        self._id = id
        self._source_name = source_name
        self._source_url = urlparse(source_url) if source_url else None
        self._ratings = ratings
        self._cwes = cwes
        self._description = description
        self._recommendations = recommendations
        self._advisories = advisories

    def get_id(self) -> str:
        return self._id

    def get_source_name(self) -> Union[str, None]:
        return self._source_name

    def get_source_url(self) -> Union[ParseResult, None]:
        return self._source_url

    def get_ratings(self) -> List[VulnerabilityRating]:
        return self._ratings

    def get_cwes(self) -> List[int]:
        return self._cwes

    def get_description(self) -> Union[str, None]:
        return self._description

    def get_recommendations(self) -> Union[List[str], None]:
        return self._recommendations

    def get_advisories(self) -> Union[List[str], None]:
        return self._advisories

    def has_ratings(self) -> bool:
        return len(self.get_ratings()) > 0

    def has_cwes(self) -> bool:
        return len(self._cwes) > 0

    def has_recommendations(self) -> bool:
        return len(self._recommendations) > 0

    def has_advisories(self) -> bool:
        return len(self._advisories) > 0
