import pathlib
from typing import Optional

import click

from cycode.cli.commands.report.sbom.path.path_command import path_command
from cycode.cli.commands.report.sbom.repository_url.repository_url_command import repository_url_command
from cycode.cli.config import config
from cycode.cli.sentry import add_breadcrumb
from cycode.cyclient.report_client import ReportParameters


@click.group(
    commands={
        'path': path_command,
        'repository_url': repository_url_command,
    },
    short_help='Generate SBOM report for remote repository by url or local directory by path.',
)
@click.option(
    '--format',
    '-f',
    help='SBOM format.',
    type=click.Choice(config['scans']['supported_sbom_formats']),
    required=True,
)
@click.option(
    '--output-format',
    '-o',
    default='json',
    help='Specify the output file format (the default is json).',
    type=click.Choice(['json']),
    required=False,
)
@click.option(
    '--output-file',
    help='Output file (the default is autogenerated filename saved to the current directory).',
    default=None,
    type=click.Path(resolve_path=True, writable=True, path_type=pathlib.Path),
    required=False,
)
@click.option(
    '--include-vulnerabilities',
    is_flag=True,
    default=False,
    help='Include vulnerabilities.',
    type=bool,
    required=False,
)
@click.option(
    '--include-dev-dependencies',
    is_flag=True,
    default=False,
    help='Include dev dependencies.',
    type=bool,
    required=False,
)
@click.pass_context
def sbom_command(
    context: click.Context,
    format: str,
    output_format: Optional[str],
    output_file: Optional[pathlib.Path],
    include_vulnerabilities: bool,
    include_dev_dependencies: bool,
) -> int:
    """Generate SBOM report."""
    add_breadcrumb('sbom')

    sbom_format_parts = format.split('-')
    if len(sbom_format_parts) != 2:
        raise click.ClickException('Invalid SBOM format.')

    sbom_format, sbom_format_version = sbom_format_parts

    report_parameters = ReportParameters(
        entity_type='SbomCli',
        sbom_report_type=sbom_format,
        sbom_version=sbom_format_version,
        output_format=output_format,
        include_vulnerabilities=include_vulnerabilities,
        include_dev_dependencies=include_dev_dependencies,
    )
    context.obj['report_parameters'] = report_parameters
    context.obj['output_file'] = output_file

    return 1
