# Copyright 2025 PT Espay Debit Indonesia Koe
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#     http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.

# coding: utf-8

"""
    OpenAPI Configuration for DANA Open API type endpoints
    Generated by OpenAPI Generator (https://openapi-generator.tech)

    Do not edit the class manually.
"""

import copy
import http.client as httplib
import logging
from logging import FileHandler
import multiprocessing
import sys
from typing import Any, ClassVar, Dict, List, Literal, Optional, TypedDict, Union
from typing_extensions import NotRequired, Self
from enum import Enum

from dana.base.configuration import BaseConfiguration

import urllib3

from dana.utils.snap_configuration import (
    JSON_SCHEMA_VALIDATION_KEYWORDS, ServerVariablesT, GenericAuthSetting,
    OAuth2AuthSetting, APIKeyAuthSetting, BasicAuthSetting, BearerFormatAuthSetting,
    BearerAuthSetting, HTTPSignatureAuthSetting, Env, HostSettingVariable, HostSetting
)

OpenApiAuthSettings = TypedDict(
    "OpenApiAuthSettings",
    {
        "CLIENT_SECRET": APIKeyAuthSetting,
        "CLIENT_ID": APIKeyAuthSetting,
        "DANA_ENV": APIKeyAuthSetting,
        "ENV": APIKeyAuthSetting,
        "ACCESS_TOKEN": APIKeyAuthSetting,
        "PRIVATE_KEY": APIKeyAuthSetting,
        "PRIVATE_KEY_PATH": APIKeyAuthSetting,
        "X_PARTNER_ID": APIKeyAuthSetting,
    },
    total=False,
)


class OpenApiConfiguration(BaseConfiguration):
    """Configuration class for DANA Open API endpoints.

    This class handles authentication and configuration for Open API type endpoints
    which use CLIENT_SECRET authentication instead of SNAP signature authentication.

    :param host: Base url.
    :param ignore_operation_servers
      Boolean to ignore operation servers for the API client.
      OpenApiConfiguration will use `host` as the base url regardless of the operation servers.
    :param api_key: Dict to store API key(s).
      Each entry in the dict specifies an API key.
      The dict key is the name of the security scheme in the OAS specification.
      The dict value is the API key secret.
    :param api_key_prefix: Dict to store API prefix (e.g. Bearer).
      The dict key is the name of the security scheme in the OAS specification.
      The dict value is an API key prefix when generating the auth data.
    :param username: Username for HTTP basic authentication.
    :param password: Password for HTTP basic authentication.
    :param access_token: Access token (optional for Open API).
    :param server_index: Index to servers configuration.
    :param server_variables: Mapping with string values to replace variables in
      templated server configuration.
    :param server_operation_index: Mapping from operation ID to an index to server
      configuration.
    :param server_operation_variables: Mapping from operation ID to a mapping with
      string values to replace variables in templated server configuration.
    :param ssl_ca_cert: str - the path to a file of concatenated CA certificates
      in PEM format.
    :param retries: Number of retries for API requests.
    :param ca_cert_data: verify the peer using concatenated CA certificate data
      in PEM (str) or DER (bytes) format.

    :Example:

    API Key Authentication Example for Open API:
    
    auth_settings = OpenApiAuthSettings(
        CLIENT_SECRET=os.getenv("CLIENT_SECRET"),
        CLIENT_ID=os.getenv("CLIENT_ID"),
        DANA_ENV=os.getenv("DANA_ENV"),
        ENV=Env.SANDBOX
    )
    config = OpenApiConfiguration(api_key=auth_settings)
    """

    _default: ClassVar[Optional[Self]] = None

    def __init__(
        self,
        host: Optional[str]=None,
        api_key: Optional[OpenApiAuthSettings]=None,
        api_key_prefix: Optional[Dict[str, str]]=None,
        username: Optional[str]=None,
        password: Optional[str]=None,
        access_token: Optional[str]=None,
        server_index: Optional[int]=None,
        server_variables: Optional[ServerVariablesT]=None,
        server_operation_index: Optional[Dict[int, int]]=None,
        server_operation_variables: Optional[Dict[int, ServerVariablesT]]=None,
        ignore_operation_servers: bool=False,
        ssl_ca_cert: Optional[str]=None,
        retries: Optional[int] = None,
        ca_cert_data: Optional[Union[str, bytes]] = None,
        *,
        debug: Optional[bool] = None,
    ) -> None:
        """Constructor"""
        
        # Initialize server configuration first to avoid circular reference
        self.server_index = 0 if server_index is None and host is None else server_index
        self.server_operation_index = server_operation_index or {}
        self.server_variables = server_variables or {}
        self.server_operation_variables = server_operation_variables or {}
        self.ignore_operation_servers = ignore_operation_servers
        
        # Set host based on environment
        if api_key and ("DANA_ENV" in api_key or "ENV" in api_key):
            self.host = "https://api.sandbox.dana.id" if (api_key["DANA_ENV"] == Env.SANDBOX or api_key["ENV"] == Env.SANDBOX) else \
                       "https://api.saas.dana.id" if (api_key["DANA_ENV"] == Env.PRODUCTION or api_key["ENV"] == Env.PRODUCTION) else None
            self._base_path = self.host
        else:
            self.host = host or "https://api.sandbox.dana.id"
            self._base_path = self.host

        self.temp_folder_path = None

        # Authentication Settings
        self.api_key = {}
        if api_key:
            self.api_key = api_key

        self.api_key_prefix = {}
        if api_key_prefix:
            self.api_key_prefix = api_key_prefix

        self.refresh_api_key_hook = None
        self.username = username
        self.password = password
        self.access_token = access_token

        # Logging Settings
        self.logger = {}
        self.logger["package_logger"] = logging.getLogger("dana")
        self.logger["urllib3_logger"] = logging.getLogger("urllib3")
        self.logger_format = '%(asctime)s %(levelname)s %(message)s'
        self.logger_stream_handler = None
        self.logger_file_handler: Optional[FileHandler] = None
        self.logger_file = None
        
        if debug is not None:
            self.debug = debug
        else:
            self.__debug = False

        self.verify_ssl = True
        self.ssl_ca_cert = ssl_ca_cert
        self.ca_cert_data = ca_cert_data
        self.cert_file = None
        self.key_file = None
        self.assert_hostname = None
        self.tls_server_name = None

        self.connection_pool_maxsize = multiprocessing.cpu_count() * 5
        self.proxy: Optional[str] = None
        self.proxy_headers = None
        self.safe_chars_for_path_param = ''
        self.retries = retries
        self.client_side_validation = True
        self.socket_options = None

        self.datetime_format = "%Y-%m-%dT%H:%M:%S.%f%z"
        self.date_format = "%Y-%m-%d"

    def __deepcopy__(self, memo: Dict[int, Any]) -> Self:
        cls = self.__class__
        result = cls.__new__(cls)
        memo[id(self)] = result
        for k, v in self.__dict__.items():
            if k not in ('logger', 'logger_file_handler'):
                setattr(result, k, copy.deepcopy(v, memo))
        # shallow copy of loggers
        result.logger = copy.copy(self.logger)
        # use setters to configure loggers
        result.logger_file = self.logger_file
        result.debug = self.debug
        return result

    def __setattr__(self, name: str, value: Any) -> None:
        object.__setattr__(self, name, value)

    @classmethod
    def set_default(cls, default: Optional[Self]) -> None:
        """Set default instance of configuration."""
        cls._default = default

    @classmethod
    def get_default_copy(cls) -> Self:
        """Deprecated. Please use `get_default` instead."""
        return cls.get_default()

    @classmethod
    def get_default(cls) -> Self:
        """Return the default configuration."""
        if cls._default is None:
            cls._default = cls()
        return cls._default

    @property
    def logger_file(self) -> Optional[str]:
        """The logger file."""
        return self.__logger_file

    @logger_file.setter
    def logger_file(self, value: Optional[str]) -> None:
        """The logger file."""
        self.__logger_file = value
        if self.__logger_file:
            self.logger_file_handler = logging.FileHandler(self.__logger_file)
            self.logger_file_handler.setFormatter(self.logger_formatter)
            for _, logger in self.logger.items():
                logger.addHandler(self.logger_file_handler)

    @property
    def debug(self) -> bool:
        """Debug status"""
        return self.__debug

    @debug.setter
    def debug(self, value: bool) -> None:
        """Debug status"""
        self.__debug = value
        if self.__debug:
            for _, logger in self.logger.items():
                logger.setLevel(logging.DEBUG)
            httplib.HTTPConnection.debuglevel = 1
        else:
            for _, logger in self.logger.items():
                logger.setLevel(logging.WARNING)
            httplib.HTTPConnection.debuglevel = 0

    @property
    def logger_format(self) -> str:
        """The logger format."""
        return self.__logger_format

    @logger_format.setter
    def logger_format(self, value: str) -> None:
        """The logger format."""
        self.__logger_format = value
        self.logger_formatter = logging.Formatter(self.__logger_format)

    def get_api_key_with_prefix(self, identifier: str, alias: Optional[str]=None) -> Optional[str]:
        """Gets API key (with prefix if set)."""
        if self.refresh_api_key_hook is not None:
            self.refresh_api_key_hook(self)
        key = self.api_key.get(identifier, self.api_key.get(alias) if alias is not None else None)
        if key:
            prefix = self.api_key_prefix.get(identifier)
            if prefix:
                return "%s %s" % (prefix, key)
            else:
                return key
        return None

    def get_basic_auth_token(self) -> Optional[str]:
        """Gets HTTP basic authentication header (string)."""
        username = ""
        if self.username is not None:
            username = self.username
        password = ""
        if self.password is not None:
            password = self.password
        return urllib3.util.make_headers(
            basic_auth=username + ':' + password
        ).get('authorization')

    def auth_settings(self) -> OpenApiAuthSettings:
        """Gets Auth Settings dict for Open API client."""
        auth: OpenApiAuthSettings = {}
        if 'CLIENT_SECRET' in self.api_key:
            auth['CLIENT_SECRET'] = {
                'type': 'api_key',
                'in': 'header',
                'key': 'clientSecret',
                'value': self.get_api_key_with_prefix('CLIENT_SECRET'),
            }
        if 'CLIENT_ID' in self.api_key:
            auth['CLIENT_ID'] = {
                'type': 'api_key',
                'in': 'header',
                'key': 'clientId',
                'value': self.get_api_key_with_prefix('CLIENT_ID'),
            }
        if 'X_PARTNER_ID' in self.api_key:
            auth['X_PARTNER_ID'] = {
                'type': 'api_key',
                'in': 'header',
                'key': 'X-PARTNER-ID',
                'value': self.get_api_key_with_prefix('X_PARTNER_ID'),
            }
        return auth

    def to_debug_report(self) -> str:
        """Gets the essential information for debugging."""
        return "Python SDK Debug Report:\n"\
               "OS: {env}\n"\
               "Python Version: {pyversion}\n"\
               "Version of the API: 1.0.0\n"\
               "SDK Package Version: 1.0.0".\
               format(env=sys.platform, pyversion=sys.version)

    def get_host_settings(self) -> List[HostSetting]:
        """Gets an array of host settings"""
        if self.api_key and ("DANA_ENV" in self.api_key or "ENV" in self.api_key):
            if self.api_key["DANA_ENV"] == Env.SANDBOX or self.api_key["ENV"] == Env.SANDBOX:
                return [{'url': "https://api.sandbox.dana.id", 'description': "Sandbox server"}]
            elif self.api_key["DANA_ENV"] == Env.PRODUCTION or self.api_key["ENV"] == Env.PRODUCTION:
                return [{'url': "https://api.saas.dana.id", 'description': "Production server"}]
        
        return [{'url': "https://api.sandbox.dana.id", 'description': "Default sandbox server"}]

    def get_host_from_settings(
        self,
        index: Optional[int],
        variables: Optional[ServerVariablesT]=None,
        servers: Optional[List[HostSetting]]=None,
    ) -> str:
        """Gets host URL based on the index and variables"""
        if index is None:
            return self._base_path

        variables = {} if variables is None else variables
        servers = self.get_host_settings() if servers is None else servers

        try:
            server = servers[index]
        except IndexError:
            raise ValueError(
                "Invalid index {0} when selecting the host settings. "
                "Must be less than {1}".format(index, len(servers)))

        url = server['url']

        # go through variables and replace placeholders
        for variable_name, variable in server.get('variables', {}).items():
            used_value = variables.get(
                variable_name, variable['default_value'])

            if 'enum_values' in variable \
                    and used_value not in variable['enum_values']:
                raise ValueError(
                    "The variable `{0}` in the host URL has invalid value "
                    "{1}. Must be {2}.".format(
                        variable_name, variables[variable_name],
                        variable['enum_values']))

            url = url.replace("{" + variable_name + "}", used_value)

        return url

    @property
    def host(self) -> str:
        """Return generated host."""
        return self.get_host_from_settings(self.server_index, variables=self.server_variables)

    @host.setter
    def host(self, value: str) -> None:
        """Fix base path."""
        self._base_path = value
        self.server_index = None 