"""Simple web admin dashboard for DarkCode Server.

Security considerations:
- Uses a 6-digit PIN generated at startup (shown in terminal)
- PIN is separate from WebSocket auth token for easier web access
- Served on the same port (HTTP upgrade for WebSocket, regular HTTP for admin)
- All admin actions require PIN authentication
- Read-only by default, write actions require explicit confirmation
"""

import base64
import html
import json
import secrets
import time
from datetime import datetime, timedelta
from typing import Optional
from http import HTTPStatus

# ASCII art logo with synthwave gradient styling
ASCII_LOGO = """
          .................                                                        ..........
         .#######-########+.         .+#####+.+###################+#####-       .########+.
         .#######- .+########.     -#########.####################+#####+     -########-
         .-------.     --------..------..---..------.       .-----.-----.  .--------..
         .#######-      -#############. +####.######.+############+#####+ -########.
         .#######-        +########+.   +####.######.+############+#####+########+.
         .#######-     .+########-......+####.######. -######+.   -#####+ .########-.
         .#######-   .+########.   .+########.######.   .-######- -#####+   .########+.
         .#######- .+#######+.       .-+++++-.-++++-.      .+++++++#####-     .+#######+.
         .#######+########+.                                      -##-.         .########+.
         .######+..................................................... .....................
         .###+-----------------------------------.--------------------.------------------.
       ..++---             .-----.        .-----.-----.        .-----------.............
       +####+.            .#####.        -#####-####+.       .+####-+##################.
     .#####-             .#####.        -####++####+.       .+####+#####-
    .###################+####################+###################.##################+.
   .+#################+-###################-+################+-..##################-.
"""

# Embedded favicon (32x32)
FAVICON_B64 = "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"

# HTML template for the admin dashboard
# NOTE: CSS curly braces are escaped ({{ and }}) to avoid Python .format() conflicts
ADMIN_HTML = """<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>DarkCode Server Admin</title>
    <link rel="icon" type="image/png" href="/favicon.ico">
    <style>
        :root {{
            --bg: #0a0a0f;
            --bg-card: #12121a;
            --border: #2a2a3a;
            --text: #e0e0e0;
            --text-dim: #888;
            --accent: #00d4ff;
            --accent-dim: #0088aa;
            --success: #00ff88;
            --warning: #ffaa00;
            --danger: #ff4466;
        }}

        * {{ box-sizing: border-box; margin: 0; padding: 0; }}

        body {{
            font-family: 'SF Mono', 'Fira Code', monospace;
            background: var(--bg);
            color: var(--text);
            min-height: 100vh;
            padding: 20px;
        }}

        .container {{ max-width: 1200px; margin: 0 auto; }}

        header {{
            display: flex;
            align-items: center;
            justify-content: space-between;
            padding: 20px 0;
            border-bottom: 1px solid var(--border);
            margin-bottom: 30px;
        }}

        .logo {{
            display: flex;
            align-items: center;
            gap: 12px;
            font-size: 24px;
            font-weight: bold;
            color: var(--accent);
        }}

        .logo img {{
            height: 40px;
            width: auto;
        }}

        .logo span {{ color: var(--text-dim); font-weight: normal; }}

        .ascii-logo {{
            font-family: 'Courier New', monospace;
            font-size: 6px;
            line-height: 1.1;
            white-space: pre;
            background: linear-gradient(135deg, #8b5cf6 0%, #06b6d4 50%, #a855f7 100%);
            -webkit-background-clip: text;
            -webkit-text-fill-color: transparent;
            background-clip: text;
            margin-bottom: 10px;
            text-shadow: 0 0 20px rgba(139, 92, 246, 0.5);
            filter: drop-shadow(0 0 8px rgba(6, 182, 212, 0.3));
        }}

        .ascii-logo-container {{
            text-align: center;
            padding: 20px;
            background: linear-gradient(180deg, rgba(139, 92, 246, 0.1) 0%, transparent 100%);
            border-radius: 12px;
            margin-bottom: 20px;
        }}

        .status-badge {{
            display: flex;
            align-items: center;
            gap: 8px;
            padding: 8px 16px;
            background: rgba(0, 255, 136, 0.1);
            border: 1px solid var(--success);
            border-radius: 20px;
            font-size: 14px;
        }}

        .status-badge::before {{
            content: '';
            width: 8px;
            height: 8px;
            background: var(--success);
            border-radius: 50%;
            animation: pulse 2s infinite;
        }}

        @keyframes pulse {{
            0%, 100% {{ opacity: 1; }}
            50% {{ opacity: 0.5; }}
        }}

        .grid {{
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(300px, 1fr));
            gap: 20px;
        }}

        .card {{
            background: var(--bg-card);
            border: 1px solid var(--border);
            border-radius: 12px;
            padding: 20px;
        }}

        .card h2 {{
            font-size: 14px;
            text-transform: uppercase;
            color: var(--text-dim);
            margin-bottom: 15px;
            letter-spacing: 1px;
        }}

        .stat {{
            display: flex;
            justify-content: space-between;
            padding: 10px 0;
            border-bottom: 1px solid var(--border);
        }}

        .stat:last-child {{ border-bottom: none; }}

        .stat-label {{ color: var(--text-dim); }}
        .stat-value {{ color: var(--accent); font-weight: bold; }}

        .sessions-list {{
            max-height: 300px;
            overflow-y: auto;
        }}

        .session-item {{
            padding: 12px;
            background: rgba(0, 212, 255, 0.05);
            border: 1px solid var(--border);
            border-radius: 8px;
            margin-bottom: 10px;
        }}

        .session-item:last-child {{ margin-bottom: 0; }}

        .session-id {{
            font-size: 12px;
            color: var(--text-dim);
            margin-bottom: 5px;
        }}

        .session-info {{
            display: flex;
            justify-content: space-between;
            font-size: 13px;
        }}

        .empty {{ color: var(--text-dim); font-style: italic; }}

        .qr-section {{
            text-align: center;
            padding: 20px;
        }}

        .qr-section img {{
            max-width: 200px;
            background: white;
            padding: 10px;
            border-radius: 8px;
        }}

        .token-display {{
            font-family: monospace;
            background: rgba(0, 0, 0, 0.5);
            padding: 15px;
            border-radius: 8px;
            word-break: break-all;
            color: var(--warning);
        }}

        .actions {{
            display: flex;
            flex-wrap: wrap;
            gap: 10px;
            margin-top: 15px;
        }}

        .btn {{
            padding: 10px 20px;
            border: 1px solid var(--border);
            border-radius: 8px;
            background: transparent;
            color: var(--text);
            font-family: inherit;
            cursor: pointer;
            transition: all 0.2s;
        }}

        .btn:hover {{
            background: rgba(255, 255, 255, 0.05);
            border-color: var(--accent);
        }}

        .btn-danger {{ border-color: var(--danger); color: var(--danger); }}
        .btn-danger:hover {{ background: rgba(255, 68, 102, 0.1); }}

        .refresh-note {{
            text-align: center;
            color: var(--text-dim);
            font-size: 12px;
            margin-top: 30px;
        }}

        .login-form {{
            max-width: 400px;
            margin: 100px auto;
        }}

        .login-form input {{
            width: 100%;
            padding: 15px;
            background: var(--bg-card);
            border: 1px solid var(--border);
            border-radius: 8px;
            color: var(--text);
            font-family: inherit;
            font-size: 24px;
            margin-bottom: 15px;
            text-align: center;
            letter-spacing: 8px;
        }}

        .login-form input:focus {{
            outline: none;
            border-color: var(--accent);
        }}

        .login-form button {{
            width: 100%;
            padding: 15px;
            background: var(--accent);
            border: none;
            border-radius: 8px;
            color: var(--bg);
            font-family: inherit;
            font-size: 16px;
            font-weight: bold;
            cursor: pointer;
        }}

        .error {{
            background: rgba(255, 68, 102, 0.1);
            border: 1px solid var(--danger);
            padding: 15px;
            border-radius: 8px;
            margin-bottom: 15px;
            color: var(--danger);
        }}
    </style>
</head>
<body>
    <div class="container">
        {content}
    </div>
</body>
</html>
"""

LOGIN_CONTENT = """
<div class="login-form">
    <div class="ascii-logo-container" style="margin-bottom: 20px;">
        <pre class="ascii-logo">{ascii_logo}</pre>
    </div>
    <div style="text-align: center; margin-bottom: 30px;">
        <h1 style="color: var(--accent);">Admin Login</h1>
    </div>
    {error}
    <form id="loginForm" method="GET" action="/admin/login">
        <input type="text" id="pinInput" name="pin" placeholder="000000" maxlength="6" pattern="[0-9]{{6}}" autofocus autocomplete="off">
        <button type="submit">Login</button>
    </form>
    <p style="text-align: center; margin-top: 20px; color: var(--text-dim); font-size: 12px;">
        Enter the 6-digit PIN shown in the terminal
    </p>
</div>
<script>
// Auto-focus
document.getElementById('pinInput').focus();
</script>
"""

DASHBOARD_CONTENT = """
<div class="ascii-logo-container">
    <pre class="ascii-logo">{ascii_logo}</pre>
</div>

<header>
    <div class="logo">
        DARKCODE <span>admin</span>
    </div>
    <div class="status-badge">Server Running</div>
</header>

<div class="grid">
    <div class="card">
        <h2>Server Status</h2>
        <div class="stat">
            <span class="stat-label">Uptime</span>
            <span class="stat-value">{uptime}</span>
        </div>
        <div class="stat">
            <span class="stat-label">Port</span>
            <span class="stat-value">{port}</span>
        </div>
        <div class="stat">
            <span class="stat-label">Working Directory</span>
            <span class="stat-value" title="{working_dir}">{working_dir_short}</span>
        </div>
        <div class="stat">
            <span class="stat-label">Server State</span>
            <span class="stat-value">{state}</span>
        </div>
        <div class="stat">
            <span class="stat-label">Device Lock</span>
            <span class="stat-value">{device_lock}</span>
        </div>
        <div class="stat">
            <span class="stat-label">Bound Device</span>
            <span class="stat-value">{bound_device}</span>
        </div>
        <div class="stat">
            <span class="stat-label">TLS</span>
            <span class="stat-value">{tls_status}</span>
        </div>
        {unbind_button}
    </div>

    <div class="card">
        <h2>Active Sessions ({session_count})</h2>
        <div class="sessions-list">
            {sessions_html}
        </div>
    </div>

    <div class="card">
        <h2>Authentication</h2>
        <p class="stat-label" style="margin-bottom: 10px;">Auth Token (masked)</p>
        <div class="token-display">{token_masked}</div>
        <div class="actions">
            <button class="btn" onclick="copyToken()">Copy Full Token</button>
        </div>
    </div>

    <div class="card">
        <h2>Connection Info</h2>
        <div class="stat">
            <span class="stat-label">Local IP</span>
            <span class="stat-value">{local_ip}</span>
        </div>
        {tailscale_row}
        <div class="stat">
            <span class="stat-label">WebSocket URL</span>
            <span class="stat-value">{ws_url}</span>
        </div>
    </div>
</div>

<p class="refresh-note">Auto-refreshing every 5 seconds | <a href="/admin/logout" style="color: var(--accent);">Logout</a></p>

<script>
    const TOKEN = '{token_full}';
    function copyToken() {{
        navigator.clipboard.writeText(TOKEN).then(() => {{
            alert('Token copied to clipboard');
        }});
    }}
    // Auto-refresh dashboard every 5 seconds
    setTimeout(() => location.reload(), 5000);
</script>
"""


def generate_web_pin() -> str:
    """Generate a 6-digit PIN for web admin login."""
    return ''.join(str(secrets.randbelow(10)) for _ in range(6))


class WebAdminHandler:
    """Handle HTTP requests for the web admin dashboard."""

    # Class-level PIN that persists across handler instances
    _web_pin: Optional[str] = None
    # Class-level authenticated sessions (must be class-level to persist across instances)
    _authenticated_sessions: set = set()
    _start_time: Optional[float] = None

    def __init__(self, config, server_instance=None):
        self.config = config
        self.server = server_instance

        # Set start time once on first handler creation
        if WebAdminHandler._start_time is None:
            WebAdminHandler._start_time = time.time()

        # Generate PIN once on first handler creation
        if WebAdminHandler._web_pin is None:
            WebAdminHandler._web_pin = generate_web_pin()

    @classmethod
    def get_web_pin(cls) -> str:
        """Get the current web PIN, generating one if needed."""
        if cls._web_pin is None:
            cls._web_pin = generate_web_pin()
        return cls._web_pin

    @classmethod
    def regenerate_pin(cls) -> str:
        """Regenerate the web PIN."""
        cls._web_pin = generate_web_pin()
        return cls._web_pin

    def _generate_session_cookie(self) -> str:
        """Generate a random session cookie."""
        return secrets.token_urlsafe(32)

    def _is_authenticated(self, cookies: dict) -> bool:
        """Check if the request has a valid session cookie."""
        session_id = cookies.get('darkcode_admin_session')
        return session_id in WebAdminHandler._authenticated_sessions

    def _verify_pin(self, pin: str) -> bool:
        """Verify the provided PIN matches the web PIN."""
        if WebAdminHandler._web_pin is None:
            return False
        # Simple string comparison - strip whitespace from input
        return pin.strip() == WebAdminHandler._web_pin

    def _parse_cookies(self, cookie_header: str) -> dict:
        """Parse cookies from header."""
        cookies = {}
        if cookie_header:
            for item in cookie_header.split(';'):
                if '=' in item:
                    key, value = item.strip().split('=', 1)
                    cookies[key] = value
        return cookies

    def _parse_form_data(self, body: bytes) -> dict:
        """Parse URL-encoded form data."""
        from urllib.parse import parse_qs
        data = parse_qs(body.decode('utf-8'))
        return {k: v[0] if len(v) == 1 else v for k, v in data.items()}

    def handle_request(self, path: str, method: str, headers: dict, body: bytes = b'') -> tuple:
        """Handle an HTTP request and return (status, headers, body).

        Returns:
            Tuple of (status_code, response_headers_dict, response_body_bytes)
        """
        from urllib.parse import urlparse, parse_qs

        cookies = self._parse_cookies(headers.get('Cookie', ''))

        # Parse path and query string
        parsed = urlparse(path)
        clean_path = parsed.path
        query_params = parse_qs(parsed.query)

        # Debug: log cookie handling
        import logging
        logging.info(f"[WebAdmin] Path: {clean_path}, Cookies received: {list(cookies.keys())}")
        logging.info(f"[WebAdmin] Session ID from cookie: {cookies.get('darkcode_admin_session', 'NONE')[:16] if cookies.get('darkcode_admin_session') else 'NONE'}...")
        logging.info(f"[WebAdmin] Known sessions: {[s[:8] for s in WebAdminHandler._authenticated_sessions]}")

        # Route requests
        if clean_path == '/admin' or clean_path == '/admin/':
            is_auth = self._is_authenticated(cookies)
            logging.info(f"[WebAdmin] Auth check result: {is_auth}")
            if is_auth:
                return self._dashboard_page()
            else:
                return self._login_page()

        elif clean_path == '/admin/logo':
            # Serve embedded logo
            return self._serve_logo()

        elif clean_path == '/admin/login':
            # Handle login - check for PIN in query params
            pin = ''
            if 'pin' in query_params:
                pin = query_params['pin'][0]
            elif body:
                form_data = self._parse_form_data(body)
                pin = form_data.get('pin', '')

            # Debug logging
            import logging
            logging.info(f"Login attempt - PIN provided: '{pin}', Expected: '{WebAdminHandler._web_pin}'")

            if pin:
                if self._verify_pin(pin):
                    session_cookie = self._generate_session_cookie()
                    WebAdminHandler._authenticated_sessions.add(session_cookie)
                    logging.info(f"Login successful, setting cookie: {session_cookie[:8]}...")
                    # Use HTTP 302 redirect with Set-Cookie header
                    return (
                        302,
                        {
                            'Location': '/admin',
                            'Set-Cookie': f'darkcode_admin_session={session_cookie}; Path=/; Max-Age=86400; SameSite=Lax',
                            'Cache-Control': 'no-store, no-cache, must-revalidate, max-age=0'
                        },
                        b''
                    )
                else:
                    logging.warning(f"Login failed - PIN mismatch")
                    return self._login_page(error="Invalid PIN")
            else:
                # Show login page
                return self._login_page()

        elif clean_path == '/admin/logout':
            session_id = cookies.get('darkcode_admin_session')
            if session_id:
                WebAdminHandler._authenticated_sessions.discard(session_id)
            return (
                302,
                {
                    'Location': '/admin',
                    'Set-Cookie': 'darkcode_admin_session=; HttpOnly; SameSite=Lax; Path=/; Max-Age=0'
                },
                b''
            )

        elif clean_path == '/admin/api/status':
            if not self._is_authenticated(cookies):
                return (401, {'Content-Type': 'application/json'}, b'{"error": "Unauthorized"}')
            return self._api_status()

        elif clean_path == '/admin/unbind':
            if not self._is_authenticated(cookies):
                return self._login_page()
            return self._unbind_device()

        else:
            return (404, {'Content-Type': 'text/html'}, b'Not Found')

    def _serve_logo(self) -> tuple:
        """Serve the embedded DarkCode logo."""
        logo_data = base64.b64decode(LOGO_B64)
        return (200, {'Content-Type': 'image/png', 'Cache-Control': 'max-age=3600'}, logo_data)

    def _login_page(self, error: str = '') -> tuple:
        """Render the login page."""
        error_html = f'<div class="error">{html.escape(error)}</div>' if error else ''
        content = LOGIN_CONTENT.format(error=error_html, ascii_logo=html.escape(ASCII_LOGO))
        page = ADMIN_HTML.format(content=content)
        return (200, {
            'Content-Type': 'text/html',
            'Cache-Control': 'no-store, no-cache, must-revalidate, max-age=0',
            'Pragma': 'no-cache'
        }, page.encode('utf-8'))

    def _dashboard_page(self) -> tuple:
        """Render the main dashboard."""
        # Calculate uptime
        uptime_secs = int(time.time() - (WebAdminHandler._start_time or time.time()))
        uptime = str(timedelta(seconds=uptime_secs))

        # Get session info
        sessions_html = '<p class="empty">No active sessions</p>'
        session_count = 0
        if self.server and hasattr(self.server, 'sessions'):
            session_count = len(self.server.sessions)
            if session_count > 0:
                sessions_html = ''
                for sid, session in self.server.sessions.items():
                    guest_badge = ' <span style="color: var(--warning);">[guest]</span>' if getattr(session, 'is_guest', False) else ''
                    sessions_html += f'''
                    <div class="session-item">
                        <div class="session-id">ID: {sid[:8]}...{guest_badge}</div>
                        <div class="session-info">
                            <span>IP: {getattr(session, 'client_ip', 'unknown')}</span>
                            <span>Msgs: {getattr(session, 'message_count', 0)}</span>
                        </div>
                    </div>
                    '''

        # Get server state
        state = 'running'
        if self.server and hasattr(self.server, 'state'):
            state = self.server.state.value

        # Get IPs
        local_ips = self.config.get_local_ips()
        local_ip = local_ips[0]['address'] if local_ips else '127.0.0.1'

        tailscale_ip = self.config.get_tailscale_ip()
        tailscale_row = ''
        if tailscale_ip:
            tailscale_row = f'''
            <div class="stat">
                <span class="stat-label">Tailscale IP</span>
                <span class="stat-value" style="color: var(--success);">{tailscale_ip}</span>
            </div>
            '''

        # Working dir (shortened)
        working_dir = str(self.config.working_dir)
        working_dir_short = working_dir if len(working_dir) <= 30 else '...' + working_dir[-27:]

        # WebSocket URL
        protocol = 'wss' if self.config.tls_enabled else 'ws'
        ws_url = f'{protocol}://{local_ip}:{self.config.port}'

        # Bound device info
        bound_device = 'None'
        unbind_button = ''
        if self.config.bound_device_id:
            bound_device = self.config.bound_device_id[:12] + '...'
            unbind_button = '''
            <div class="actions" style="margin-top: 15px;">
                <button class="btn btn-danger" onclick="if(confirm('Unbind device? This will allow a new device to connect.')) location.href='/admin/unbind'">Unbind Device</button>
            </div>
            '''

        content = DASHBOARD_CONTENT.format(
            ascii_logo=html.escape(ASCII_LOGO),
            uptime=uptime,
            port=self.config.port,
            working_dir=working_dir,
            working_dir_short=working_dir_short,
            state=state,
            device_lock='Enabled' if self.config.device_lock else 'Disabled',
            bound_device=bound_device,
            unbind_button=unbind_button,
            tls_status='Enabled (wss://)' if self.config.tls_enabled else 'Disabled (ws://)',
            session_count=session_count,
            sessions_html=sessions_html,
            token_masked=self.config.token[:4] + '*' * 20 + self.config.token[-4:],
            token_full=self.config.token,
            local_ip=local_ip,
            tailscale_row=tailscale_row,
            ws_url=ws_url,
        )

        page = ADMIN_HTML.format(content=content)
        return (200, {'Content-Type': 'text/html'}, page.encode('utf-8'))

    def _unbind_device(self) -> tuple:
        """Unbind the current device and redirect back to dashboard."""
        if self.server and hasattr(self.server, 'unbind_device'):
            self.server.unbind_device()
        elif self.config.bound_device_id:
            # Fallback: directly modify config
            self.config.bound_device_id = None
            self.config.save()

        return (
            302,
            {'Location': '/admin'},
            b''
        )

    def _api_status(self) -> tuple:
        """Return status as JSON for API consumers."""
        uptime_secs = int(time.time() - (WebAdminHandler._start_time or time.time()))
        session_count = 0
        if self.server and hasattr(self.server, 'sessions'):
            session_count = len(self.server.sessions)

        data = {
            'uptime_seconds': uptime_secs,
            'port': self.config.port,
            'session_count': session_count,
            'state': self.server.state.value if self.server and hasattr(self.server, 'state') else 'unknown',
            'device_lock': self.config.device_lock,
            'tls_enabled': self.config.tls_enabled,
        }

        return (200, {'Content-Type': 'application/json'}, json.dumps(data).encode('utf-8'))


def serve_favicon() -> tuple:
    """Serve the embedded favicon."""
    favicon_data = base64.b64decode(FAVICON_B64)
    return (200, {'Content-Type': 'image/png', 'Cache-Control': 'max-age=86400'}, favicon_data)
