# Copyright 2018-2026 contributors to the OpenLineage project
# SPDX-License-Identifier: Apache-2.0

from __future__ import annotations

from typing import ClassVar

import attr
from openlineage.client.generated.base import RunFacet


@attr.define
class ErrorMessageRunFacet(RunFacet):
    message: str
    """
    A human-readable string representing error message generated by observed system

    Example: org.apache.spark.sql.AnalysisException: Table or view not found: wrong_table_name; line 1
    pos 14
    """
    programmingLanguage: str  # noqa: N815
    """
    Programming language the observed system uses.

    Example: JAVA
    """
    stackTrace: str | None = attr.field(default=None)  # noqa: N815
    """
    A language-specific stack trace generated by observed system

    Example: Exception in thread "main" java.lang.RuntimeException: A test exception
    at io.openlineage.SomeClass.method(SomeClass.java:13)
    at io.openlineage.SomeClass.anotherMethod(SomeClass.java:9)
    """
    _additional_skip_redact: ClassVar[list[str]] = ["programmingLanguage"]

    @staticmethod
    def _get_schema() -> str:
        return (
            "https://openlineage.io/spec/facets/1-0-1/ErrorMessageRunFacet.json#/$defs/ErrorMessageRunFacet"
        )
