"""
Integration tests module
"""

import aiohttp
import pandas as pd
import pytest
from ddx._rust.common import ProductSymbol
from ddx._rust.common.enums import OrderSide, OrderType
from ddx._rust.decimal import Decimal
from tests.harness.account import Account
from tests.harness.market_data_driver.custom_market_data_driver import \
    CustomMarketDataDriver
from tests.harness.matchers.account import AccountMatcher
from tests.harness.matchers.tick import default_match_all_up_to
from tests.harness.store import Store
from tests.python_only import maybe_mock_client

ETHP = ProductSymbol("ETHP")
DDXP = ProductSymbol("DDXP")


# TODO low priority: doctest every comment


@pytest.mark.asyncio
async def test_deposit():
    async with aiohttp.ClientSession() as client:
        store: Store = Store.the_store()
        async with store.run_scenario(
            CustomMarketDataDriver.from_price_ranges(
                {ETHP: "ETH/USD"},
                "1min",
                client,
                start_timestamp=pd.Timestamp("2023-04-18"),
                duration="1m",
                price_ranges={ETHP: (Decimal("1_000"), Decimal("1_000"))},
            ),
            client,
        ):
            # Alice deposits 1_000 USDC.
            alice = AccountMatcher(Account(store.wallet.account_for_index(0)))
            await alice.should_deposit("main", Decimal("1_000"))


@pytest.mark.asyncio
async def test_min_deposit():
    async with aiohttp.ClientSession() as client:
        store: Store = Store.the_store()
        async with store.run_scenario(
            CustomMarketDataDriver.from_price_ranges(
                {ETHP: "ETH/USD"},
                "1min",
                client,
                start_timestamp=pd.Timestamp("2023-04-18"),
                duration="1m",
                price_ranges={ETHP: (Decimal("1_000"), Decimal("1_000"))},
            ),
            client,
        ):
            # Alice deposits 500 USDC. This amount should not pass the minimum deposit amount of 1_000 USDC.
            alice = AccountMatcher(Account(store.wallet.account_for_index(0)))
            await alice.should_not_deposit_min_amount_not_met("main", Decimal("500"))


@pytest.mark.asyncio
async def test_deposit_collateral_guard():
    async with aiohttp.ClientSession() as client:
        store: Store = Store.the_store()
        async with store.run_scenario(
            CustomMarketDataDriver.from_price_ranges(
                {ETHP: "ETH/USD"},
                "1min",
                client,
                start_timestamp=pd.Timestamp("2023-04-18"),
                duration="1m",
                price_ranges={ETHP: (Decimal("1_000"), Decimal("1_000"))},
            ),
            client,
        ):
            # Alice deposits 15_000 USDC. The collateral guard is for 10_000 USDC, so 10_000 should be in free collateral and the remaining 5_000 USDC in the frozen collateral
            alice = AccountMatcher(Account(store.wallet.account_for_index(0)))
            await alice.should_deposit("main", Decimal("15_000"))

            # TODO: explicit supplementary check on leaves here


@pytest.mark.asyncio
async def test_deposit_ddx():
    async with aiohttp.ClientSession() as client:
        store: Store = Store.the_store()
        async with store.run_scenario(
            CustomMarketDataDriver.from_price_ranges(
                {ETHP: "ETH/USD"},
                "1min",
                client,
                start_timestamp=pd.Timestamp("2023-04-18"),
                duration="1m",
                price_ranges={ETHP: (Decimal("1_000"), Decimal("1_000"))},
            ),
            client,
        ):
            # Alice deposits 1 DDX.
            alice = AccountMatcher(Account(store.wallet.account_for_index(0)))
            await alice.should_deposit_ddx(Decimal("1"))


# TODO: Withdrawal/checkpoint tests do not work in python-only mode since they require storing a checkpoint from the operator. As the operator does not exist in python-only mode, the checkpoint (and its epoch id) must be generated by the previous AdvanceEpoch request and stored on the python side.
@pytest.mark.asyncio
async def test_successful_withdraw():
    async with aiohttp.ClientSession() as client:
        store: Store = Store.the_store()
        async with store.run_scenario(
            CustomMarketDataDriver.from_price_ranges(
                {ETHP: "ETH/USD"},
                "1min",
                client,
                start_timestamp=pd.Timestamp("2023-04-18"),
                duration="1m",
                price_ranges={ETHP: (Decimal("1_000"), Decimal("1_000"))},
            ),
            client,
        ) as tick_generator:
            # Alice deposits 1_000 USDC.
            alice = AccountMatcher(Account(store.wallet.account_for_index(0)))
            await alice.should_deposit("main", Decimal("1_000"))

            # Alice intends to withdraw 500 USDC.
            await alice.should_intend_withdraw("main", Decimal("500"))

            current = await default_match_all_up_to(
                store.advance_until_next_epoch(
                    tick_generator,
                )
            )
            current.default_match_rest()

            # Alice submits a valid checkpoint and gets a merkle proof for withdrawal.
            _, strategy_proof = await alice.should_submit_valid_checkpoint("main")

            # Alice claims her withdrawal.
            await alice.should_claim_withdrawal("main", Decimal("500"), strategy_proof)


# TODO 3644: this requires the sequencer to be tested in the harness
# @pytest.mark.asyncio
# async def test_insufficient_balance_withdraw():
#     async with aiohttp.ClientSession() as client:
#         store: Store = Store.the_store()
#         async with store.run_scenario(
#             CustomMarketDataDriver.from_price_ranges(
#                 {ETHP: "ETH/USD"},
#                 "1min",
#                 client,
#                 start_timestamp=pd.Timestamp("2023-04-18"),
#                 duration="1m",
#                 price_ranges={ETHP: (Decimal("1_000"), Decimal("1_000"))},
#             ),
#             client,
#         ):
#             # Alice deposits 1_000 USDC.
#             alice = AccountMatcher(Account(store.wallet.account_for_index(0)))
#             await alice.should_deposit("main", Decimal("1_000"))
#
#             # Alice intends to withdraw 1_500 USDC, which is more than her balance.
#             await alice.should_reject_withdrawal_intent_insufficient_balance(
#                 "main", Decimal("1_500")
#             )


@pytest.mark.asyncio
async def test_invalid_checkpoint():
    async with aiohttp.ClientSession() as client:
        store: Store = Store.the_store()
        async with store.run_scenario(
            CustomMarketDataDriver.from_price_ranges(
                {ETHP: "ETH/USD"},
                "1min",
                client,
                start_timestamp=pd.Timestamp("2023-04-18"),
                duration="1m",
                price_ranges={ETHP: (Decimal("1_000"), Decimal("1_000"))},
            ),
            client,
        ) as tick_generator:
            # Alice deposits 1_000 USDC.
            alice = AccountMatcher(Account(store.wallet.account_for_index(0)))
            await alice.should_deposit("main", Decimal("1_000"))
            # Alice intends to withdraw 500 USDC.
            await alice.should_intend_withdraw("main", Decimal("500"))
            current = await default_match_all_up_to(
                store.advance_until_next_epoch(
                    tick_generator,
                )
            )
            current.default_match_rest()
            # Alice submits an invalid checkpoint.
            await alice.should_submit_invalid_checkpoint()


@pytest.mark.asyncio
async def test_intent_and_claim_withdraw_do_not_match():
    async with aiohttp.ClientSession() as client:
        store: Store = Store.the_store()
        async with store.run_scenario(
            CustomMarketDataDriver.from_price_ranges(
                {ETHP: "ETH/USD"},
                "1min",
                client,
                start_timestamp=pd.Timestamp("2023-04-18"),
                duration="1m",
                price_ranges={ETHP: (Decimal("1_000"), Decimal("1_000"))},
            ),
            client,
        ) as tick_generator:
            # Alice deposits 1_000 USDC.
            alice = AccountMatcher(Account(store.wallet.account_for_index(0)))
            await alice.should_deposit("main", Decimal("1_000"))
            # Alice intends to withdraw 500 USDC.
            await alice.should_intend_withdraw("main", Decimal("500"))
            current = await default_match_all_up_to(
                store.advance_until_next_epoch(
                    tick_generator,
                )
            )
            current.default_match_rest()
            # Alice submits a valid checkpoint.
            _, strategy_proof = await alice.should_submit_valid_checkpoint("main")
            # Alice's withdrawal claim is rejected because the amount she is claiming does not match the amount she intended to withdraw.
            await alice.should_reject_withdrawal_claim(
                "main", Decimal("600"), strategy_proof
            )


@pytest.mark.asyncio
async def test_successful_withdraw_ddx():
    async with aiohttp.ClientSession() as client:
        store: Store = Store.the_store()
        async with store.run_scenario(
            CustomMarketDataDriver.from_price_ranges(
                {ETHP: "ETH/USD"},
                "1min",
                client,
                start_timestamp=pd.Timestamp("2023-04-18"),
                duration="1m",
                price_ranges={ETHP: (Decimal("1_000"), Decimal("1_000"))},
            ),
            client,
        ) as tick_generator:
            # Alice deposits 1_000 DDX.
            alice = AccountMatcher(Account(store.wallet.account_for_index(0)))
            await alice.should_deposit_ddx(Decimal("1_000"))

            # Alice intends to withdraw 500 DDX.
            await alice.should_intend_withdraw_ddx(Decimal("500"))

            current = await default_match_all_up_to(
                store.advance_until_next_epoch(
                    tick_generator,
                )
            )
            current.default_match_rest()

            # Alice submits a valid checkpoint and gets a merkle proof for DDX withdrawal.
            _, trader_proof = await alice.should_submit_valid_checkpoint()

            # Alice claims her withdrawal.
            await alice.should_claim_withdrawal_ddx(Decimal("500"), trader_proof)


# TODO 3644: this requires the sequencer to be tested in the harness
# @pytest.mark.asyncio
# async def test_insufficient_balance_withdraw_ddx():
#     async with aiohttp.ClientSession() as client:
#         store: Store = Store.the_store()
#         async with store.run_scenario(
#             CustomMarketDataDriver.from_price_ranges(
#                 {ETHP: "ETH/USD"},
#                 "1min",
#                 client,
#                 start_timestamp=pd.Timestamp("2023-04-18"),
#                 duration="1m",
#                 price_ranges={ETHP: (Decimal("1_000"), Decimal("1_000"))},
#             ),
#             client,
#         ):
#             # Alice deposits 1_000 DDX.
#             alice = AccountMatcher(Account(store.wallet.account_for_index(0)))
#             await alice.should_deposit_ddx(Decimal("1_000"))
#
#             # Alice intends to withdraw 1_500 DDX, which is more than her balance.
#             await alice.should_reject_withdrawal_ddx_intent_insufficient_balance(
#                 Decimal("1_500")
#             )


@pytest.mark.asyncio
async def test_intent_and_claim_withdraw_ddx_do_not_match():
    async with aiohttp.ClientSession() as client:
        store: Store = Store.the_store()
        async with store.run_scenario(
            CustomMarketDataDriver.from_price_ranges(
                {ETHP: "ETH/USD"},
                "1min",
                client,
                start_timestamp=pd.Timestamp("2023-04-18"),
                duration="1m",
                price_ranges={ETHP: (Decimal("1_000"), Decimal("1_000"))},
            ),
            client,
        ) as tick_generator:
            # Alice deposits 1_000 DDX.
            alice = AccountMatcher(Account(store.wallet.account_for_index(0)))
            await alice.should_deposit_ddx(Decimal("1_000"))
            # Alice intends to withdraw 500 DDX.
            await alice.should_intend_withdraw_ddx(Decimal("500"))
            current = await default_match_all_up_to(
                store.advance_until_next_epoch(
                    tick_generator,
                )
            )
            current.default_match_rest()
            # Alice submits a valid checkpoint.
            _, trader_proof = await alice.should_submit_valid_checkpoint()
            # Alice's DDX withdrawal claim is rejected because the amount she is claiming does not match the amount she intended to withdraw.
            await alice.should_reject_withdrawal_ddx_claim(Decimal("600"), trader_proof)


@pytest.mark.asyncio
async def test_trade_with_ddx_fees():
    async with aiohttp.ClientSession() as client:
        store: Store = Store.the_store()
        async with store.run_scenario(
            CustomMarketDataDriver.from_price_ranges(
                {ETHP: "ETH/USD", DDXP: "DDX/USD"},
                "1min",
                client,
                start_timestamp=pd.Timestamp("2023-04-18"),
                duration="1m",
                price_ranges={
                    ETHP: (Decimal("1_000"), Decimal("1_000")),
                    DDXP: (Decimal("1"), Decimal("1")),
                },
            ),
            client,
        ) as tick_generator:
            # Alice deposits 10_000 USDC
            alice = AccountMatcher(Account(store.wallet.account_for_index(0)))
            await alice.should_deposit("main", Decimal("10_000"))

            # Bob deposits 10_000 USDC
            bob = AccountMatcher(Account(store.wallet.account_for_index(1)))
            await bob.should_deposit("main", Decimal("10_000"))

            await bob.should_deposit_ddx(Decimal("1_000"))
            await bob.should_update_profile(True)

            # Alice submits her first order (post).
            await alice.should_post_order(
                ETHP,
                "main",
                OrderSide.Bid,
                OrderType.Limit,
                Decimal("1"),
                Decimal("237"),
                Decimal("0"),
            )

            current = await default_match_all_up_to(
                store.advance_until_next_price_checkpoint(
                    tick_generator,
                )
            )
            current.should_mint_price_checkpoint()
            current.default_match_rest()

            # Bob submit his first order (match - complete fill)
            await bob.should_complete_fill_order(
                ETHP,
                "main",
                OrderSide.Ask,
                OrderType.Limit,
                Decimal("1"),
                Decimal("237"),
                Decimal("0"),
            )


# Issue #4141: https://gitlab.com/dexlabs/derivadex/-/issues/4141
@pytest.mark.asyncio
async def test_trade_with_zero_rounded_ddx_fee():
    async with aiohttp.ClientSession() as client:
        store: Store = Store.the_store()
        async with store.run_scenario(
            CustomMarketDataDriver.from_price_ranges(
                {ETHP: "ETH/USD", DDXP: "DDX/USD"},
                "1min",
                client,
                start_timestamp=pd.Timestamp("2023-04-18"),
                duration="1m",
                price_ranges={
                    ETHP: (Decimal("100"), Decimal("100")),
                    DDXP: (Decimal("1_000"), Decimal("1_000")),
                },
            ),
            client,
        ) as tick_generator:
            # Alice deposits 10_000 USDC
            alice = AccountMatcher(Account(store.wallet.account_for_index(0)))
            await alice.should_deposit("main", Decimal("10_000"))

            # Bob deposits 10_000 USDC
            bob = AccountMatcher(Account(store.wallet.account_for_index(1)))
            await bob.should_deposit("main", Decimal("10_000"))

            await bob.should_deposit_ddx(Decimal("1_000"))
            await bob.should_update_profile(True)

            # Alice submits her first order (post).
            await alice.should_post_order(
                ETHP,
                "main",
                OrderSide.Bid,
                OrderType.Limit,
                Decimal("0.001"),
                Decimal("100"),
                Decimal("0"),
            )

            current = await default_match_all_up_to(
                store.advance_until_next_price_checkpoint(
                    tick_generator,
                )
            )
            current.should_mint_price_checkpoint()
            current.default_match_rest()

            # Bob submit his first order (match - complete fill)
            await bob.should_complete_fill_order(
                ETHP,
                "main",
                OrderSide.Ask,
                OrderType.Limit,
                Decimal("0.001"),
                Decimal("100"),
                Decimal("0"),
            )


# TODO 3825: this requires the sequencer to be tested in the harness
# @pytest.mark.asyncio
# async def test_trade_with_ddx_fees_not_enough_ddx():
#     async with aiohttp.ClientSession() as client:
#         store: Store = Store.the_store()
#         async with store.run_scenario(
#             CustomMarketDataDriver.from_price_ranges(
#                 {ETHP: "ETH/USD", DDXP: "DDX/USD"},
#                 "1min",
#                 client,
#                 start_timestamp=pd.Timestamp("2023-04-18"),
#                 duration="1m",
#                 price_ranges={
#                     ETHP: (Decimal("1_000"), Decimal("1_000")),
#                     DDXP: (Decimal("1"), Decimal("1")),
#                 },
#             ),
#             client,
#         ):
#             # Alice deposits 10_000 USDC
#             alice = AccountMatcher(Account(store.wallet.account_for_index(0)))
#             await alice.should_deposit("main", Decimal("10_000"))
#
#             # Bob deposits 10_000 USDC
#             bob = AccountMatcher(Account(store.wallet.account_for_index(1)))
#             await bob.should_deposit("main", Decimal("10_000"))
#
#             await bob.should_update_profile(True)
#
#             # Alice submits her first order (post).
#             await alice.should_post_order(
#                 ETHP,
#                 "main",
#                 OrderSide.Ask,
#                 OrderType.Limit,
#                 Decimal("1"),
#                 Decimal("237"),
#                 Decimal("0"),
#             )
#
#             # Bob submit his first order (match - complete fill)
#             await bob.should_complete_fill_order(
#                 ETHP,
#                 "main",
#                 OrderSide.Bid,
#                 OrderType.Limit,
#                 Decimal("1"),
#                 Decimal("237"),
#                 Decimal("0"),
#             )
