"""
Service registration module for boto3 integration.

This module handles the registration of the community-devops-agent service
with botocore, enabling boto3.client('community-devops-agent') usage.
"""

import logging
from pathlib import Path

from botocore.session import get_session

from .loaders.service_loader import ServiceLoader

logger = logging.getLogger(__name__)

# Service configuration
SERVICE_NAME = "community-devops-agent"
API_VERSION = "2025-12-09"

# Package data directory
PACKAGE_DIR = Path(__file__).parent
DATA_DIR = PACKAGE_DIR / "data" / SERVICE_NAME


def _register_service_with_botocore() -> None:
    """
    Register the community-devops-agent service with botocore.

    This function monkey-patches the default botocore loader to include our service.
    """
    try:
        from botocore.loaders import Loader

        # Create our custom service loader
        service_loader = ServiceLoader(
            data_path=str(DATA_DIR),
            service_name=SERVICE_NAME,
            api_version=API_VERSION,
        )

        # Store our loader as a module-level variable so it persists
        global _custom_loader
        _custom_loader = service_loader

        # Monkey-patch the Loader.load_data method to check our loader first
        original_load_data = Loader.load_data

        def patched_load_data(self, name):
            # Try our custom loader first
            try:
                result = _custom_loader.load_data(name)
                if result is not None:
                    return result
            except Exception:
                pass
            # Fall back to original method
            return original_load_data(self, name)

        Loader.load_data = patched_load_data

        # Also monkey-patch list_available_services
        original_list_services = Loader.list_available_services

        def patched_list_services(self, type_name="service-2"):
            services = original_list_services(self, type_name)
            if SERVICE_NAME not in services:
                services.append(SERVICE_NAME)
            return services

        Loader.list_available_services = patched_list_services

        # Monkey-patch list_api_versions to return our API version
        original_list_api_versions = Loader.list_api_versions

        def patched_list_api_versions(
            self, service_name, type_name="service-2"
        ):
            if service_name == SERVICE_NAME:
                return [API_VERSION]
            return original_list_api_versions(self, service_name, type_name)

        Loader.list_api_versions = patched_list_api_versions

        # Set the service model path globally
        import botocore.session

        session = botocore.session.Session()
        session.set_config_variable(
            f"service_model_{SERVICE_NAME}",
            f"{SERVICE_NAME}/{API_VERSION}/service-2",
        )

        logger.debug(
            f"Successfully monkey-patched botocore loader for {SERVICE_NAME}"
        )

    except Exception as e:
        logger.error(f"Failed to register {SERVICE_NAME} service: {e}")
        raise


# Global variable to hold our custom loader
_custom_loader = None


def _register_credential_provider() -> None:
    """
    Register the custom DevOpsAgentCredentialProvider with botocore.

    This enables automatic authentication flow for the service.
    """
    try:
        from .auth import DevOpsAgentCredentialProvider

        # Get the botocore session
        session = get_session()

        # Create our credential provider
        provider = DevOpsAgentCredentialProvider()

        # Get the credential resolver
        resolver = session.get_component("credential_provider")

        # Insert our provider at the beginning of the chain so it gets priority
        # This ensures it's checked before environment variables
        if hasattr(resolver, "providers"):
            resolver.providers.insert(0, provider)
            logger.debug(
                "Successfully registered DevOpsAgentCredentialProvider at start of chain"
            )
        else:
            # Fallback to component registration
            session.register_component("credential_provider", provider)
            logger.debug(
                "Successfully registered DevOpsAgentCredentialProvider as component"
            )

    except ImportError:
        logger.warning(
            "DevOpsAgentCredentialProvider not available yet - auth.py not implemented"
        )
    except Exception as e:
        logger.error(f"Failed to register credential provider: {e}")
        raise


def register_service() -> None:
    """
    Main registration function called on package import.

    This function registers both the service loader and credential provider
    with botocore, enabling full boto3 integration.
    """
    logger.info(f"Registering {SERVICE_NAME} service with boto3")

    # Register the service model loader
    _register_service_with_botocore()

    # Register the credential provider (if available)
    _register_credential_provider()

    logger.info(f"{SERVICE_NAME} service successfully registered with boto3")


# Register the service when this module is imported
register_service()
