import stat
import requests
import os
import subprocess
import logging
import re
import base64
from devsecops_engine_tools.engine_sca.engine_container.src.infrastructure.helpers.images_scanned import (
    ImagesScanned,
)
from devsecops_engine_tools.engine_sca.engine_container.src.domain.model.gateways.tool_gateway import (
    ToolGateway,
)
from devsecops_engine_tools.engine_utilities.utils.logger_info import MyLogger
from devsecops_engine_tools.engine_utilities import settings

logger = MyLogger.__call__(**settings.SETTING_LOGGER).get_logger()


class PrismaCloudManagerScan(ToolGateway):
    def download_twistcli(
        self,
        file_path,
        prisma_access_key,
        prisma_secret_key,
        prisma_console_url,
        prisma_api_version,
    ):
        url = f"{prisma_console_url}/api/{prisma_api_version}/util/twistcli"
        credentials = base64.b64encode(
            f"{prisma_access_key}:{prisma_secret_key}".encode()
        ).decode()
        headers = {"Authorization": f"Basic {credentials}"}
        try:
            response = requests.get(url, headers=headers)
            response.raise_for_status()

            with open(file_path, "wb") as file:
                file.write(response.content)

            os.chmod(file_path, stat.S_IRWXU)
            logging.info(f"twistcli downloaded and saved to: {file_path}")
            return 0

        except Exception as e:
            raise ValueError(f"Error downloading twistcli: {e}")

    def scan_image(self, file_path, image, remoteconfig, prisma_secret_key, build_id):
        file_name = "scanned_images.txt"
        image_name = f"{image.tags[0]}"
        result_file = f"{image_name}" + "_scan_result.json"
        images_scanned = []

        if (result_file) in ImagesScanned.get_images_already_scanned(file_name):
            print(f"The image {image_name} has already been scanned previously.")
        else:
            if build_id in image_name:
                command = (
                    file_path,
                    "images",
                    "scan",
                    "--address",
                    remoteconfig["PRISMA_CLOUD"]["PRISMA_CONSOLE_URL"],
                    "--user",
                    remoteconfig["PRISMA_CLOUD"]["PRISMA_ACCESS_KEY"],
                    "--password",
                    prisma_secret_key,
                    "--output-file",
                    result_file,
                    "--details",
                    image_name,
                )
                try:
                    subprocess.run(
                        command,
                        check=True,
                        stdout=subprocess.PIPE,
                        stderr=subprocess.PIPE,
                        text=True,
                    )
                    images_scanned.append(result_file)
                    print(f"The image {image_name} was scanned")
                    with open(file_name, "a") as file:
                        file.write(result_file + "\n")
                except subprocess.CalledProcessError as e:
                    logger.error(f"Error during image scan of {image_name}: {e.stderr}")

        return images_scanned

    def run_tool_container_sca(
        self, remoteconfig, prisma_secret_key, image, build_id, skip_flag
    ):
        images_scanned = []
        if not (skip_flag):
            try:
                file_path = os.path.join(
                    os.getcwd(), remoteconfig["PRISMA_CLOUD"]["TWISTCLI_PATH"]
                )

                if not os.path.exists(file_path):
                    self.download_twistcli(
                        file_path,
                        remoteconfig["PRISMA_CLOUD"]["PRISMA_ACCESS_KEY"],
                        prisma_secret_key,
                        remoteconfig["PRISMA_CLOUD"]["PRISMA_CONSOLE_URL"],
                        remoteconfig["PRISMA_CLOUD"]["PRISMA_API_VERSION"],
                    )
                images_scanned.extend(
                    self.scan_image(
                        file_path,
                        image,
                        remoteconfig,
                        prisma_secret_key,
                        build_id,
                    )
                )

                return images_scanned

            except Exception as ex:
                logger.error(f"An overall error occurred: {ex}")

        return images_scanned
