#
#   MIT License
#   
#   Copyright (c) 2024, Mattias Aabmets
#   
#   The contents of this file are subject to the terms and conditions defined in the License.
#   You may not use, modify, or distribute this file except in compliance with the License.
#   
#   SPDX-License-Identifier: MIT
#
import sys
import subprocess
from typer import Typer
from pathlib import Path
from rich.panel import Panel
from rich.prompt import Confirm
from rich.console import Console
from devtools_cli.utils import *

app = Typer(name="config", help="Project management configuration.")
console = Console(soft_wrap=True)


@app.command(name="init", epilog="Example: devtools config init")
def cmd_init():
    """
    Initializes the current working directory with a .devtools config file, if it doesn't exist.
    """
    config_file = find_local_config_file(init_cwd=False)
    cwd = Path.cwd()

    if config_file:
        if cwd == config_file.parent:
            console.print("The current working directory already has a '.devtools' config file.")
        elif cwd.is_relative_to(config_file.parent):
            console.print("Cannot create nested devtool configurations.")
        raise SystemExit()

    if not config_file:
        if Confirm.ask("Initialize current working directory as project root?"):
            find_local_config_file(init_cwd=True)
            console.print("Devtools initialized!")
        else:
            console.print("Did nothing.")


@app.command(name="edit", epilog="Example: devtools config edit")
def cmd_edit():
    """
    Opens the nearest .devtools config file relative to the current working directory for editing.
    """
    config_file = find_local_config_file(init_cwd=False)
    if not config_file:
        console.print(f"Config file not found, unable to continue.")
        raise SystemExit()

    warning = "[bold grey78]Waiver of Responsibility"
    console.print(Panel.fit(warning, border_style="red"))
    edit = Confirm.ask(
        "You are asking to edit an autogenerated file.\n"
        "This can break devtools if done incorrectly.\n"
        "Are you sure you want to do this?"
    )
    if edit:
        cf = str(config_file)
        if sys.platform == "win32":
            subprocess.run(["cmd", "/c", "start", "", cf], stdout=subprocess.DEVNULL, stderr=subprocess.DEVNULL)
        elif sys.platform == "darwin":
            subprocess.run(["open", cf], stdout=subprocess.DEVNULL, stderr=subprocess.DEVNULL)
        elif sys.platform.startswith("linux"):
            subprocess.run(["xdg-open", cf], stdout=subprocess.DEVNULL, stderr=subprocess.DEVNULL)
        else:
            print(f"Platform {sys.platform} not supported.")
            raise SystemExit()
