"""Configuration source adapters for DFM nowcasting.

This module provides adapters for loading DFMConfig/DDFMConfig from Hydra/OmegaConf sources:
- YAML files (with Hydra/OmegaConf support)
- Hydra DictConfig objects

All adapters implement the ConfigSource protocol and return DFMConfig or DDFMConfig objects.
"""

import warnings
from typing import Protocol, Optional, Dict, Any, Union, TYPE_CHECKING
from pathlib import Path

from .schema import DFMConfig, SeriesConfig
from ..logger import get_logger

try:
    from .schema import DDFMConfig
except ImportError:
    DDFMConfig = None  # Fallback if not available

if TYPE_CHECKING:
    from .schema import DDFMConfig
    ModelConfig = Union[DFMConfig, DDFMConfig]
else:
    ModelConfig = Union[DFMConfig, Any]  # Runtime fallback

_logger = get_logger(__name__)


class ConfigSource(Protocol):
    """Protocol for configuration sources.
    
    Any object that implements a `load()` method returning a DFMConfig or DDFMConfig
    can be used as a configuration source.
    """
    def load(self) -> ModelConfig:
        """Load and return a DFMConfig or DDFMConfig object (auto-detected)."""
        ...


class YamlSource:
    """Load configuration from a YAML file.
    
    Supports Hydra-style configs. Simply converts YAML to dict and delegates to from_dict().
    """
    def __init__(self, yaml_path: Union[str, Path]):
        """Initialize YAML source.
        
        Parameters
        ----------
        yaml_path : str or Path
            Path to YAML configuration file
        """
        self.yaml_path = Path(yaml_path)
    
    def load(self) -> ModelConfig:
        """Load configuration from YAML file using Hydra compose.
        
        Uses Hydra's compose system to properly handle defaults, config groups, and overrides.
        This ensures that defaults like /model: dfm are automatically resolved.
        
        For temporary files (e.g., in tests), falls back to direct YAML loading.
        
        Returns
        -------
        DFMConfig or DDFMConfig
            Configuration object. Type is automatically detected based on config content.
            Returns DDFMConfig if DDFM-specific parameters are present, otherwise DFMConfig.
        """
        try:
            import hydra
            import yaml
            from omegaconf import OmegaConf, DictConfig
        except ImportError:
            raise ImportError("hydra and omegaconf are required for YAML config loading. Install with: pip install hydra-core omegaconf")
        
        configfile = Path(self.yaml_path)
        if not configfile.exists():
            raise FileNotFoundError(f"Configuration file not found: {configfile}")
        
        # Check if this is a temporary file (e.g., in tests) or a real config directory
        # If parent directory doesn't look like a config structure, use direct YAML loading
        is_temp_file = '/tmp' in str(configfile) or '/pytest' in str(configfile) or configfile.parent.name.startswith('test_')
        
        # Extract config_dir and config_name from file path
        # Example: config/experiment/consumption_kowrccnse_report.yaml
        #   -> config_dir: config/
        #   -> config_name: experiment/consumption_kowrccnse_report
        config_dir = configfile.parent.parent  # Go up from experiment/ to config/
        config_path = str(config_dir)  # For loading series config files
        config_name = configfile.parent.name + '/' + configfile.stem  # experiment/consumption_kowrccnse_report
        
        if is_temp_file:
            # For temporary files, load YAML directly and use from_hydra()
            with open(configfile, 'r') as f:
                cfg_dict = yaml.safe_load(f) or {}
            return DFMConfig.from_hydra(cfg_dict, config_path=config_path)
        
        # Use Hydra compose to properly handle defaults
        try:
            with hydra.initialize_config_dir(config_dir=str(config_dir), version_base="1.3"):
                cfg = hydra.compose(config_name=config_name)
                # Convert to dict (not struct) to allow processing
                if OmegaConf.is_struct(cfg):
                    OmegaConf.set_struct(cfg, False)
                cfg_dict = OmegaConf.to_container(cfg, resolve=True)
            
            # Delegate all parsing logic to from_hydra() which handles Hydra format
            # from_hydra() internally calls from_dict() with type detection
            # Pass config_path for loading individual series config files when series is a list of strings
            return DFMConfig.from_hydra(cfg_dict, config_path=config_path)
        except Exception as e:
            # Fallback to direct YAML loading if Hydra compose fails
            _logger.warning(f"Failed to load config via Hydra compose: {e}. Falling back to direct YAML loading.")
            import yaml
            with open(configfile, 'r') as f:
                cfg_dict = yaml.safe_load(f) or {}
            return DFMConfig.from_hydra(cfg_dict, config_path=config_path)


class HydraSource:
    """Load configuration from a Hydra DictConfig or dict.
    
    This adapter handles Hydra's composed configuration objects,
    converting them to DFMConfig format.
    """
    def __init__(self, cfg: Union[Dict[str, Any], 'DictConfig'], config_path: Optional[str] = None):
        """Initialize Hydra source.
        
        Parameters
        ----------
        cfg : DictConfig or dict
            Hydra configuration object or dictionary in Hydra format
        config_path : Optional[str]
            Path to config directory (for loading individual series files when series is a list of strings).
            If None, attempts to extract from cfg metadata (_config_path_ or _name_).
        """
        self.cfg = cfg
        self.config_path = config_path
    
    def load(self) -> ModelConfig:
        """Load configuration from Hydra DictConfig/dict.
        
        Returns DFMConfig or DDFMConfig based on config content (auto-detected).
        """
        # Try to extract config_path from cfg metadata if not provided
        config_path = self.config_path
        if config_path is None:
            try:
                from omegaconf import DictConfig
                if isinstance(self.cfg, DictConfig):
                    # Try to extract from Hydra metadata
                    config_path_attr = getattr(self.cfg, '_config_path_', None)
                    if config_path_attr:
                        config_path = str(config_path_attr)
                    else:
                        # Try to infer from _name_ (e.g., "experiment/consumption_kowrccnse_report")
                        name_attr = getattr(self.cfg, '_name_', None)
                        if name_attr and '/' in str(name_attr):
                            from pathlib import Path
                            # Extract directory path (e.g., "experiment" from "experiment/consumption_kowrccnse_report")
                            # But we need the full config directory, not just the subdirectory
                            config_path = str(Path(str(name_attr)).parent.parent)
            except Exception:
                pass  # Fallback to None
        
        return DFMConfig.from_hydra(self.cfg, config_path=config_path)



def make_config_source(
    source: Optional[Union[str, Path, ConfigSource]] = None,
    *,
    yaml: Optional[Union[str, Path]] = None,
    hydra: Optional[Union[Dict[str, Any], 'DictConfig']] = None,
    mapping: Optional[Dict[str, Any]] = None,
) -> ConfigSource:
    """Create a ConfigSource adapter from Hydra/OmegaConf sources.
    
    This factory function automatically selects the appropriate adapter
    based on the input type or explicit keyword arguments.
    
    Parameters
    ----------
    source : str, Path, or ConfigSource, optional
        Configuration source. If a ConfigSource, returned as-is.
        If str/Path, treated as YAML file path.
    yaml : str or Path, optional
        Explicit YAML file path
    hydra : DictConfig or dict, optional
        Explicit Hydra config
        
    Returns
    -------
    ConfigSource
        Appropriate adapter for the input (YamlSource or HydraSource)
        
    Examples
    --------
    >>> # From YAML file
    >>> source = make_config_source('config/default.yaml')
    >>> 
    >>> # Explicit keyword arguments
    >>> source = make_config_source(yaml='config/default.yaml')
    >>> 
    >>> # From Hydra DictConfig
    >>> source = make_config_source(hydra=cfg)
    """
    # Check for explicit keyword arguments (only one allowed)
    explicit_kwargs = [k for k, v in [('yaml', yaml), ('hydra', hydra), ('mapping', mapping)] if v is not None]
    if len(explicit_kwargs) > 1:
        raise ValueError(
            f"Only one of yaml or hydra can be specified. "
            f"Got: {', '.join(explicit_kwargs)}."
        )
    
    # Handle explicit keyword arguments
    if yaml is not None:
        return YamlSource(yaml)
    if hydra is not None:
        return HydraSource(hydra)
    if mapping is not None:
        return HydraSource(mapping)
    
    # Infer from source argument
    if source is None:
        raise ValueError(
            "No configuration source provided. "
            "Specify source, yaml, or hydra."
        )
    
    # If already a ConfigSource, return as-is
    if hasattr(source, 'load') and callable(getattr(source, 'load')):
        return source  # type: ignore
    
    # Infer type from source
    if isinstance(source, (str, Path)):
        path = Path(source)
        suffix = path.suffix.lower()
        if suffix in ['.yaml', '.yml']:
            return YamlSource(path)
        else:
            # Default to YAML if extension unclear
            return YamlSource(path)
    
    # Support dict directly (for backward compatibility and testing)
    if isinstance(source, dict):
        # Create a HydraSource for dict
        return HydraSource(source)
    
    raise TypeError(
        f"Unsupported source type: {type(source)}. "
        f"Expected str, Path, dict, or ConfigSource."
    )




# ============================================================================
# Hydra ConfigStore Registration (optional - only if Hydra is available)
# ============================================================================

try:
    from hydra.core.config_store import ConfigStore
    HYDRA_AVAILABLE = True
except ImportError:
    HYDRA_AVAILABLE = False
    ConfigStore = None

if HYDRA_AVAILABLE and ConfigStore is not None:
    try:
        cs = ConfigStore.instance()
        if cs is not None:
            from dataclasses import dataclass as schema_dataclass
            from typing import List as ListType
            
            @schema_dataclass
            class SeriesConfigSchema:
                """Schema for SeriesConfig validation in Hydra."""
                series_id: str
                series_name: str
                frequency: str
                transformation: str
                units: Optional[str] = None  # Optional, for display only
            
            @schema_dataclass
            class DFMConfigSchema:
                """Schema for unified DFMConfig validation in Hydra."""
                series: ListType[SeriesConfigSchema]
                block_names: ListType[str]
                factors_per_block: Optional[ListType[int]] = None
                ar_lag: int = 1
                threshold: float = 1e-5
                max_iter: int = 5000
                nan_method: int = 2
                nan_k: int = 3
                clock: str = 'm'
            
            @schema_dataclass
            class DDFMConfigSchema:
                """Schema for unified DDFMConfig validation in Hydra."""
                series: ListType[SeriesConfigSchema]
                encoder_layers: Optional[ListType[int]] = None
                num_factors: Optional[int] = None
                activation: str = 'relu'
                use_batch_norm: bool = True
                learning_rate: float = 0.001
                epochs: int = 100
                batch_size: int = 100
                factor_order: int = 1
                use_idiosyncratic: bool = True
                min_obs_idio: int = 5
                max_iter: int = 200
                tolerance: float = 0.0005
                disp: int = 10
                seed: Optional[int] = None
                nan_method: int = 2
                nan_k: int = 3
                clock: str = 'm'
                scaler: Optional[str] = 'standard'
            
            # Register schemas
            cs.store(group="dfm", name="base_dfm_config", node=DFMConfigSchema)
            cs.store(group="ddfm", name="base_ddfm_config", node=DDFMConfigSchema)
            cs.store(group="model", name="base_model_config", node=DFMConfigSchema)
            cs.store(name="dfm_config_schema", node=DFMConfigSchema)
            cs.store(name="ddfm_config_schema", node=DDFMConfigSchema)
            cs.store(name="model_config_schema", node=DFMConfigSchema)
            
    except Exception as e:
        warnings.warn(f"Could not register Hydra structured config schemas: {e}. "
                     f"Configs will still work via from_dict() but without schema validation.")

