"""NumPy-based Kalman filter and smoother.

This module provides KalmanFilter, a pure NumPy implementation for Kalman filtering and
fixed-interval smoothing operations, following the MATLAB Nowcasting reference.

Numerical Stability:
    NumPy matrix operations can fail on near-singular matrices. This module implements
    robust error handling with progressive fallbacks:
    1. Standard operations (fastest)
    2. Regularized operations (handles near-singular matrices)
    3. Pseudo-inverse fallback (most robust, but slower)
    
    All covariance matrices are regularized to ensure positive definiteness,
    preventing LinAlgError exceptions.

Reference:
    MATLAB Nowcasting (FRBNY) implementation - runKF(), SKF(), FIS() functions.
"""

from typing import Tuple, Optional
import numpy as np
from dataclasses import dataclass

from ..logger import get_logger
from .utils import (
    check_finite,
    ensure_real,
    ensure_symmetric,
    ensure_real_and_symmetric,
    ensure_covariance_stable,
    ensure_positive_definite,
    clean_matrix,
    safe_inverse,
    safe_determinant,
    cap_max_eigenval,
)

_logger = get_logger(__name__)


@dataclass
class KalmanFilterState:
    """Kalman filter state structure using NumPy arrays.
    
    This dataclass stores the complete state of the Kalman filter after forward
    and backward passes, including prior/posterior estimates and covariances.
    
    Attributes
    ----------
    Zm : np.ndarray
        Prior (predicted) factor state estimates, shape (m x nobs).
        Zm[:, t] is the predicted state at time t given observations up to t-1.
    Vm : np.ndarray
        Prior covariance matrices, shape (m x m x nobs).
        Vm[:, :, t] is the covariance of Zm[:, t].
    ZmU : np.ndarray
        Posterior (updated) factor state estimates, shape (m x (nobs+1)).
        ZmU[:, t] is the updated state at time t given observations up to t.
        Includes initial state at t=0.
    VmU : np.ndarray
        Posterior covariance matrices, shape (m x m x (nobs+1)).
        VmU[:, :, t] is the covariance of ZmU[:, t].
    loglik : float
        Log-likelihood of the data under the current model parameters.
        Computed as sum of log-likelihoods at each time step.
    k_t : np.ndarray
        Kalman gain matrix, shape (m x k) where k is number of observed series.
        Used to update state estimates with new observations.
    ZmT : Optional[np.ndarray]
        Smoothed factor state estimates (added by smoother_backward), shape (m x (nobs+1)).
    VmT : Optional[np.ndarray]
        Smoothed factor covariance (added by smoother_backward), shape (m x m x (nobs+1)).
    VmT_1 : Optional[np.ndarray]
        Lag-1 smoothed factor covariance (added by smoother_backward), shape (m x m x nobs).
    """
    Zm: np.ndarray      # Prior/predicted factor state (m x nobs)
    Vm: np.ndarray      # Prior covariance (m x m x nobs)
    ZmU: np.ndarray     # Posterior/updated state (m x (nobs+1))
    VmU: np.ndarray     # Posterior covariance (m x m x (nobs+1))
    loglik: float         # Log-likelihood
    k_t: np.ndarray     # Kalman gain
    ZmT: Optional[np.ndarray] = None  # Smoothed state (added by smoother)
    VmT: Optional[np.ndarray] = None  # Smoothed covariance (added by smoother)
    VmT_1: Optional[np.ndarray] = None  # Lag-1 smoothed covariance (added by smoother)


class KalmanFilter:
    """NumPy-based Kalman filter and smoother.
    
    This class provides Kalman filter (forward pass) and fixed-interval smoother
    (backward pass) operations using pure NumPy, following MATLAB Nowcasting reference.
    
    Parameters
    ----------
    min_eigenval : float, default 1e-8
        Minimum eigenvalue for covariance matrices
    min_diagonal_variance : float, default 1e-6
        Minimum diagonal variance for regularization
    default_variance_fallback : float, default 1.0
        Fallback variance when matrix operations fail
    min_variance_covariance : float, default 1e-10
        Minimum variance for covariance estimation
    inv_regularization : float, default 1e-6
        Regularization added before matrix inversion
    cholesky_regularization : float, default 1e-8
        Regularization for Cholesky decomposition
    chunk_size : int, default 1000
        Chunk size for memory optimization
    chunk_threshold : int, default 10000
        Threshold for chunking large datasets
    use_cpu : bool, default True
        Always True for NumPy implementation (kept for backward compatibility)
    """
    
    def __init__(
        self,
        min_eigenval: float = 1e-8,
        min_diagonal_variance: float = 1e-6,
        default_variance_fallback: float = 1.0,
        min_variance_covariance: float = 1e-10,
        inv_regularization: float = 1e-6,
        cholesky_regularization: float = 1e-8,
        chunk_size: int = 1000,
        chunk_threshold: int = 10000,
        use_cpu: bool = True  # Always True for NumPy implementation
    ):
        # Store numerical stability parameters as plain attributes
        self.min_eigenval = min_eigenval
        self.min_diagonal_variance = min_diagonal_variance
        self.default_variance_fallback = default_variance_fallback
        self.min_variance_covariance = min_variance_covariance
        self.inv_regularization = inv_regularization
        self.cholesky_regularization = cholesky_regularization
        # Memory optimization parameters
        self.chunk_size = chunk_size
        self.chunk_threshold = chunk_threshold
        self.use_cpu = True  # Always True for NumPy
    
    def forward(
        self,
        Y: np.ndarray,
        A: np.ndarray,
        C: np.ndarray,
        Q: np.ndarray,
        R: np.ndarray,
        Z_0: np.ndarray,
        V_0: np.ndarray
    ) -> Tuple[np.ndarray, np.ndarray, np.ndarray, float]:
        """Apply Kalman filter and smoother. Main entry point.
        
        This is a pure NumPy implementation following the MATLAB Nowcasting reference.
        All inputs must be NumPy arrays. Callers should convert PyTorch tensors to NumPy
        before calling this method.
        
        Parameters
        ----------
        Y : np.ndarray
            Input data (k x nobs), where k is number of series, nobs is time periods
        A : np.ndarray
            Transition matrix (m x m)
        C : np.ndarray
            Observation matrix (k x m)
        Q : np.ndarray
            Covariance for transition residuals (m x m)
        R : np.ndarray
            Covariance for observation residuals (k x k)
        Z_0 : np.ndarray
            Initial state (m,)
        V_0 : np.ndarray
            Initial covariance (m x m)
            
        Returns
        -------
        zsmooth : np.ndarray
            Smoothed factor estimates (m x (nobs+1)), zsmooth[:, t+1] = Z_t|T
        Vsmooth : np.ndarray
            Smoothed factor covariance (m x m x (nobs+1)), Vsmooth[:, :, t+1] = Cov(Z_t|T)
        VVsmooth : np.ndarray
            Lag 1 factor covariance (m x m x nobs), Cov(Z_t, Z_t-1|T)
        loglik : float
            Log-likelihood
        """
        
        # Kalman filter (forward pass)
        S = self.filter_forward(Y, A, C, Q, R, Z_0, V_0)
        
        # Fixed-interval smoother (backward pass)
        S = self.smoother_backward(A, S)
        
        # Organize output
        zsmooth = S.ZmT
        Vsmooth = S.VmT
        VVsmooth = S.VmT_1
        loglik = S.loglik
        
        # Ensure outputs are not None (shouldn't happen, but type checker needs this)
        if zsmooth is None or Vsmooth is None or VVsmooth is None:
            raise RuntimeError("Kalman smoother did not produce valid outputs")
        
        # Delete intermediate state to free memory
        del S
        
        # Ensure loglik is real and finite
        if not np.isfinite(loglik):
            loglik = float('-inf')
        
        return zsmooth, Vsmooth, VVsmooth, loglik
    
    def filter_forward(
        self,
        Y: np.ndarray,
        A: np.ndarray,
        C: np.ndarray,
        Q: np.ndarray,
        R: np.ndarray,
        Z_0: np.ndarray,
        V_0: np.ndarray
    ) -> KalmanFilterState:
        """Apply Kalman filter (forward pass)."""
        return self._filter_forward_impl(Y, A, C, Q, R, Z_0, V_0)
    
    def _filter_forward_impl(
        self,
        Y: np.ndarray,
        A: np.ndarray,
        C: np.ndarray,
        Q: np.ndarray,
        R: np.ndarray,
        Z_0: np.ndarray,
        V_0: np.ndarray
    ) -> KalmanFilterState:
        """Apply Kalman filter (forward pass).
        
        Follows MATLAB SKF() function structure (lines 874-992 in dfm.m).
        
        Parameters
        ----------
        Y : np.ndarray
            Input data (k x nobs), where k = number of series, nobs = time periods
        A : np.ndarray
            Transition matrix (m x m)
        C : np.ndarray
            Observation matrix (k x m)
        Q : np.ndarray
            Covariance for transition equation residuals (m x m)
        R : np.ndarray
            Covariance for observation matrix residuals (k x k)
        Z_0 : np.ndarray
            Initial state vector (m,)
        V_0 : np.ndarray
            Initial state covariance matrix (m x m)
            
        Returns
        -------
        KalmanFilterState
            Filter state with prior and posterior estimates
        """
        # Dimensions
        k, nobs = Y.shape  # k series, nobs time periods
        m = C.shape[1]     # m factors
        
        dtype = Y.dtype
        
        # Replace Inf values with NaN (handled by missing data logic)
        if np.any(np.isinf(Y)):
            inf_count = np.sum(np.isinf(Y))
            _logger.warning(f"kalman_filter_forward: Input Y contains {inf_count} Inf values, replacing with NaN")
            Y = np.nan_to_num(Y, nan=float('nan'), posinf=float('nan'), neginf=float('nan'))
        
        # Pre-stabilize input matrices
        # Q: process noise covariance - must be positive definite
        Q = self._ensure_cov_stable(Q, min_eigenval=self.min_eigenval, ensure_real=True)
        Q = self._clean_matrix(Q, 'covariance', default_nan=1e-6, default_inf=1e6)
        
        # R: observation noise covariance - must be positive definite (diagonal)
        # R is typically diagonal, but ensure it's stable
        if R.ndim == 2:
            # Ensure R is diagonal and positive
            diag_R = np.diag(R)
            diag_R = np.clip(diag_R, a_min=self.min_diagonal_variance, a_max=None)
            diag_R = np.nan_to_num(diag_R, nan=self.min_diagonal_variance, posinf=1e4, neginf=self.min_diagonal_variance)
            R = np.diag(diag_R)
        else:
            R = self._clean_matrix(R, 'diagonal', default_nan=self.min_diagonal_variance, default_inf=1e4)
        
        # A: transition matrix - ensure it's real and finite
        A = self._clean_matrix(A, 'general', default_nan=0.0, default_inf=1.0)
        # Clip A to prevent instability
        A = np.clip(A, a_min=-0.99, a_max=0.99)
        
        # C: observation matrix - ensure it's real and finite
        C = self._clean_matrix(C, 'loading', default_nan=0.0, default_inf=1.0)
        
        # V_0: initial covariance - must be positive definite
        V_0 = self._ensure_cov_stable(V_0, min_eigenval=self.min_eigenval, ensure_real=True)
        V_0 = self._clean_matrix(V_0, 'covariance', default_nan=1e-6, default_inf=1e6)
        
        # Z_0: initial state - ensure finite
        Z_0 = np.nan_to_num(Z_0, nan=0.0, posinf=1.0, neginf=-1.0)
        
        # Initialize output
        Zm = np.full((m, nobs), float('nan'), dtype=dtype)  # Z_t | t-1 (prior)
        Vm = np.full((m, m, nobs), float('nan'), dtype=dtype)  # V_t | t-1 (prior)
        ZmU = np.full((m, nobs + 1), float('nan'), dtype=dtype)  # Z_t | t (posterior/updated)
        VmU = np.full((m, m, nobs + 1), float('nan'), dtype=dtype)  # V_t | t (posterior/updated)
        loglik = 0.0
        
        # Set initial values
        Zu = Z_0.copy()  # Z_0|0 (In loop, Zu gives Z_t | t)
        Vu = V_0.copy()  # V_0|0 (In loop, Vu gives V_t | t)
        
        # Validate dimensions match
        if Zu.shape[0] != m:
            raise ValueError(
                f"Dimension mismatch: Z_0 has shape {Zu.shape[0]}, but C has {m} columns. "
                f"This usually indicates a mismatch between init_conditions and em_step. "
                f"Z_0 should have dimension {m} to match C.shape[1]."
            )
        if Vu.shape[0] != m or Vu.shape[1] != m:
            raise ValueError(
                f"Dimension mismatch: V_0 has shape {Vu.shape}, but expected ({m}, {m}). "
                f"This usually indicates a mismatch between init_conditions and em_step."
            )
        
        # Store initial values
        ZmU[:, 0] = Zu
        VmU[:, :, 0] = Vu
        
        # Initialize variables for final iteration (used after loop)
        Y_t = np.array([], dtype=dtype)  # Initialize Y_t to empty array
        C_t = None
        VCF = None
        Z = None  # Initialize Z and V for potential use after loop
        V = None
        
        # Kalman filter procedure
        for t in range(nobs):
            # Calculate prior distribution
            # Use transition equation to create prior estimate for factor
            # i.e. Z = Z_t|t-1
            # Check for NaN/Inf in inputs
            if not self._check_finite(Zu, f"Zu at t={t}"):
                _logger.warning(f"kalman_filter_forward: Zu contains NaN/Inf at t={t}, resetting to zeros")
                Zu = np.zeros_like(Zu)
            
            Z = A @ Zu
            
            # Check for NaN/Inf in Z
            if not self._check_finite(Z, f"Z at t={t}"):
                _logger.warning(f"kalman_filter_forward: Z contains NaN/Inf at t={t}, using previous Zu")
                Z = Zu.copy()
            
            # Prior covariance matrix of Z (i.e. V = V_t|t-1)
            # Var(Z) = Var(A*Z + u_t) = Var(A*Z) + Var(u) = A*Vu*A' + Q
            # PRE-REGULARIZATION: Check Vu condition number before V calculation to prevent NaN propagation
            if not self._check_finite(Vu, f"Vu before V calculation at t={t}"):
                Vu = self._ensure_cov_stable(Vu, min_eigenval=self.min_eigenval, ensure_real=True)
            else:
                # Pre-regularization: Check condition number and apply adaptive regularization if ill-conditioned
                try:
                    eigenvals = np.linalg.eigvalsh(Vu)
                    eigenvals = eigenvals[eigenvals > 1e-12]
                    if len(eigenvals) > 0:
                        max_eig = np.max(eigenvals)
                        min_eig = np.min(eigenvals)
                        cond_num = max_eig / min_eig if min_eig > 1e-12 else float('inf')
                        
                        # If ill-conditioned, apply pre-regularization (MATLAB-style stability)
                        if cond_num > 1e8:
                            reg_scale = self.min_eigenval * (cond_num / 1e8)
                            Vu = Vu + np.eye(Vu.shape[0], dtype=dtype) * reg_scale
                            _logger.debug(f"Pre-regularized Vu at t={t}: cond={cond_num:.2e}, reg={reg_scale:.2e}")
                except (RuntimeError, ValueError, np.linalg.LinAlgError):
                    # Fallback: apply default regularization if eigendecomposition fails
                    Vu = Vu + np.eye(Vu.shape[0], dtype=dtype) * self.min_eigenval
            
            # Ensure Vu is symmetric before matrix multiplication (MATLAB: V = 0.5 * (V+V'))
            Vu = ensure_real_and_symmetric(Vu)
            
            # Now safely compute V
            V = A @ Vu @ A.T + Q
            
            # Check for NaN/Inf before stabilization
            if not self._check_finite(V, f"V at t={t}"):
                # Fallback: use previous covariance with regularization
                V = Vu + np.eye(V.shape[0], dtype=dtype) * 1e-6
            
            # Cap maximum eigenvalue to prevent explosion
            V = self._cap_max_eigenval(V, max_eigenval=1e6)
            # Ensure V is real, symmetric, and positive semi-definite (MATLAB: V = 0.5 * (V+V'))
            V = ensure_real_and_symmetric(V)
            V = self._ensure_cov_stable(V, min_eigenval=self.min_eigenval, ensure_real=True)
            
            # Calculate posterior distribution
            # Remove missing series: These are removed from Y, C, and R
            Y_t, C_t, R_t, _ = self.handle_missing_data(Y[:, t], C, R)
            
            # Check if y_t contains no data
            if len(Y_t) == 0:
                Zu = Z
                Vu = V
            else:
                # Adaptive regularization based on missing data ratio
                missing_ratio = 1.0 - (len(Y_t) / k)
                adaptive_reg = self.inv_regularization * (1.0 + missing_ratio * 10.0)
                
                # Additional regularization for large n_obs
                n_obs = len(Y_t)
                if n_obs >= 30:
                    n_obs_factor = 1.0 + (n_obs - 29) * 0.1
                    adaptive_reg = adaptive_reg * n_obs_factor
                    if t == 0:
                        _logger.debug(f"kalman_filter_forward: n_obs={n_obs} >= 30, n_obs_factor={n_obs_factor:.2f}, adaptive_reg={adaptive_reg:.2e}")
                
                # Steps for variance and population regression coefficients:
                # Var(c_t*Z_t + e_t) = c_t Var(Z) c_t' + Var(e) = c_t*V*c_t' + R
                VC = V @ C_t.T
                
                # Compute innovation covariance F = C_t @ V @ C_t.T + R_t
                F = C_t @ VC + R_t
                
                # Apply adaptive regularization to R_t diagonal
                if R_t.ndim == 2 and R_t.shape[0] == R_t.shape[1]:
                    diag_R_t = np.diag(R_t)
                    diag_R_t = np.clip(diag_R_t, a_min=self.min_diagonal_variance * (1.0 + missing_ratio * 5.0), a_max=None)
                    R_t = np.diag(diag_R_t)
                
                # Log F matrix condition number for diagnostics
                try:
                    eigenvals_F = np.linalg.eigvalsh(F)
                    eigenvals_F = eigenvals_F[eigenvals_F > 1e-12]
                    if len(eigenvals_F) > 0:
                        max_eig_F = np.max(eigenvals_F)
                        min_eig_F = np.min(eigenvals_F)
                        cond_F = max_eig_F / min_eig_F if min_eig_F > 1e-12 else float('inf')
                        if cond_F > 1e10 or t < 5:
                            _logger.debug(
                                f"kalman_filter_forward: t={t}, n_obs={n_obs}, F condition={cond_F:.2e}, "
                                f"eigenvals=[{min_eig_F:.2e}, {max_eig_F:.2e}], adaptive_reg={adaptive_reg:.2e}"
                            )
                except (RuntimeError, ValueError, np.linalg.LinAlgError):
                    pass
                
                # Cap maximum eigenvalue to prevent condition number explosion
                F = self._cap_max_eigenval(F, max_eigenval=1e6)
                # Ensure F is real, symmetric, and positive semi-definite
                # Use adaptive minimum eigenvalue based on missing data ratio
                min_eig_adaptive = self.min_eigenval * (1.0 + missing_ratio * 5.0)
                F = self._ensure_cov_stable(F, min_eigenval=min_eig_adaptive, ensure_real=True)
                
                # Check for NaN/Inf before inversion
                if not self._check_finite(F, f"F at t={t}"):
                    # Fallback: use identity with large variance, scaled by missing ratio
                    fallback_var = 1e6 * (1.0 + missing_ratio * 5.0)
                    F = np.eye(F.shape[0], dtype=dtype) * fallback_var
                    _logger.warning(f"kalman_filter_forward: F matrix contains NaN/Inf at t={t}, using fallback (missing_ratio={missing_ratio:.2f})")
                
                # Safe inverse with adaptive regularization
                iF = self._safe_inv(F, regularization=adaptive_reg, use_pinv_fallback=True)
                
                if not self._check_finite(iF, f"iF (inverse of F) at t={t}") and t < 5:
                    _logger.warning(
                        f"kalman_filter_forward: iF contains NaN/Inf at t={t}, n_obs={n_obs}, "
                        f"adaptive_reg={adaptive_reg:.2e}, missing_ratio={missing_ratio:.2f}"
                    )
                
                # Matrix of population regression coefficients (Kalman gain)
                VCF = VC @ iF
                
                # Difference between actual and predicted observation matrix values
                innov = Y_t - C_t @ Z
                
                # Check for NaN/Inf in innovation
                if not self._check_finite(innov, f"innovation at t={t}"):
                    _logger.warning(f"kalman_filter_forward: Innovation contains NaN/Inf at t={t}, skipping update")
                    Zu = Z
                    Vu = V
                else:
                    # Check VCF before matrix multiplication
                    if not self._check_finite(VCF, f"VCF (Kalman gain) at t={t}"):
                        _logger.warning(f"kalman_filter_forward: VCF contains NaN/Inf at t={t}, using zero matrix")
                        VCF = np.zeros_like(VCF)
                    
                    # Update estimate of factor values (posterior)
                    Zu = Z + VCF @ innov
                    
                    # Clean NaN/Inf immediately after update
                    if not self._check_finite(Zu, f"Zu at t={t}"):
                        Zu = Z.copy()
                        Zu = self._clean_matrix(Zu, 'general', default_nan=0.0, default_inf=0.0)
                        _logger.warning(f"kalman_filter_forward: Zu update produced NaN/Inf at t={t}, using prior Z as fallback")
                    
                    # Update covariance matrix (posterior)
                    if not self._check_finite(VC, f"VC at t={t}"):
                        _logger.warning(f"kalman_filter_forward: VC contains NaN/Inf at t={t}, using zero matrix")
                        VC = np.zeros_like(VC)
                    
                    Vu = V - VCF @ VC.T
                    
                    # MATLAB-style: Ensure symmetry immediately after update (V = 0.5 * (V+V'))
                    Vu = ensure_real_and_symmetric(Vu)
                    # Cap maximum eigenvalue to prevent explosion
                    Vu = self._cap_max_eigenval(Vu, max_eigenval=1e6)
                    
                    # Clean NaN/Inf before stabilization
                    if not self._check_finite(Vu, f"Vu at t={t}"):
                        Vu = self._clean_matrix(Vu, 'general', default_nan=1e-8, default_inf=1e6)
                        Vu = ensure_real_and_symmetric(Vu)  # Re-apply symmetry after cleaning
                    
                    # Check for NaN/Inf after cleaning
                    if not self._check_finite(Vu, f"Vu at t={t}"):
                        _logger.warning(f"kalman_filter_forward: Vu contains NaN/Inf at t={t}, using V as fallback")
                        Vu = V.copy()
                        Vu = ensure_real_and_symmetric(Vu)  # Ensure symmetry of fallback
                        # Ensure fallback is also stable
                        Vu = self._ensure_cov_stable(Vu, min_eigenval=self.min_eigenval, ensure_real=True)
                    
                    # Ensure Vu is real, symmetric, and positive semi-definite (MATLAB: Vu = 0.5 * (Vu+Vu'))
                    Vu = ensure_real_and_symmetric(Vu)
                    Vu = self._ensure_cov_stable(Vu, min_eigenval=self.min_eigenval, ensure_real=True)
                    
                    # Update log-likelihood (MATLAB-style: always compute when observations exist)
                    # MATLAB line 970: S.loglik = S.loglik + 0.5*(log(det(iF)) - innov'*iF*innov);
                    # MATLAB computes directly without checking det(iF) > 0, relying on inv() to handle singular matrices
                    # We match this behavior by computing log(det(iF)) directly, using safe_det for numerical stability
                    try:
                        # Compute det(iF) - MATLAB uses det(inv(F)) = 1/det(F), but we compute det(iF) directly
                        # MATLAB's inv() may return a pseudo-inverse for singular matrices, but we use safe_det
                        det_iF = self._safe_det(iF, use_logdet=True)
                        
                        # MATLAB computes log(det(iF)) directly - if det <= 0, log will be NaN/Inf, but MATLAB still computes
                        # We use logdet for numerical stability, but match MATLAB's behavior of always computing
                        if det_iF > 0:
                            log_det = np.log(det_iF)
                        else:
                            # If det(iF) <= 0, MATLAB's inv() would still compute, but result may be unreliable
                            # We use logdet of regularized F to match MATLAB's behavior while maintaining stability
                            F_reg = C_t @ VC + R_t
                            # Add regularization to ensure positive definite (MATLAB's inv() handles this internally)
                            F_reg = F_reg + np.eye(F_reg.shape[0], dtype=dtype) * adaptive_reg
                            det_F_reg = self._safe_det(F_reg, use_logdet=True)
                            if det_F_reg > 0:
                                # log(det(iF)) = log(det(inv(F))) = -log(det(F)) when F is invertible
                                log_det = -np.log(det_F_reg)
                            else:
                                # Fallback: use small positive value to avoid NaN
                                # Clip to reasonable range to prevent extreme loglik values
                                # MATLAB's inv() handles singular matrices, but we need bounded fallback
                                log_det_raw = -n_obs * np.log(max(adaptive_reg, 1e-6))
                                log_det = np.clip(log_det_raw, -100.0, 100.0)
                                if t < 5:
                                    _logger.warning(
                                        f"kalman_filter_forward: det(iF) <= 0 at t={t}, using regularized logdet. "
                                        f"n_obs={n_obs}, missing_ratio={missing_ratio:.2f}, "
                                        f"log_det_raw={log_det_raw:.2f}, log_det_clipped={log_det:.2f}"
                                    )
                        
                        # MATLAB: innov'*iF*innov where innov is column vector (n_obs x 1)
                        # Our innov is (n_obs,) - need to ensure it's column vector for transpose
                        innov_col = innov.reshape(-1, 1) if innov.ndim == 1 else innov  # (n_obs, 1)
                        # innov'*iF*innov = (1 x n_obs) @ (n_obs x n_obs) @ (n_obs x 1) = scalar
                        innov_term = (innov_col.T @ iF @ innov_col).item() if innov_col.ndim == 2 else (innov_col.T @ iF @ innov_col)
                        if isinstance(innov_term, np.ndarray):
                            innov_term = innov_term.item() if innov_term.size == 1 else float(innov_term)
                        
                        # Clip innov_term to prevent extreme values (especially in first iteration)
                        # This prevents loglik from becoming extremely negative due to unstable initial parameters
                        # MATLAB's inv() may handle this automatically, but we need explicit clipping
                        innov_term_clipped = np.clip(innov_term, a_min=-1e6, a_max=1e6)
                        
                        # MATLAB computes: 0.5*(log(det(iF)) - innov'*iF*innov)
                        # Both terms are scalars, so we compute directly
                        if np.isfinite(innov_term_clipped) and np.isfinite(log_det):
                            loglik_increment = 0.5 * (log_det - innov_term_clipped)
                            loglik += loglik_increment
                            # Warn if clipping occurred
                            if abs(innov_term - innov_term_clipped) > 1e-3 and t < 5:
                                _logger.warning(
                                    f"kalman_filter_forward: innov_term clipped at t={t}: "
                                    f"original={innov_term:.2e}, clipped={innov_term_clipped:.2e}"
                                )
                            # Debug first few iterations
                            if t < 3:
                                _logger.debug(
                                    f"kalman_filter_forward: t={t}, n_obs={n_obs}, "
                                    f"det_iF={det_iF:.6e}, log_det={log_det:.6f}, "
                                    f"innov_term={innov_term:.6f}, loglik_inc={loglik_increment:.6f}, "
                                    f"loglik_total={loglik:.6f}"
                                )
                        else:
                            if t < 5:
                                _logger.warning(
                                    f"kalman_filter_forward: innov_term or log_det not finite at t={t}, "
                                    f"innov_term={innov_term}, log_det={log_det}, skipping loglik update"
                                )
                    except (RuntimeError, ValueError, OverflowError, np.linalg.LinAlgError) as e:
                        if t < 5:
                            _logger.warning(f"kalman_filter_forward: Log-likelihood calculation failed at t={t}: {e}")
                        # MATLAB would error here too, so we skip the update
            
            # Store output
            # Store covariance and observation values for t (priors)
            # Ensure Z and V are real and finite before storing
            Z = self._ensure_real(Z)
            if not self._check_finite(Z, f"Z (prior) at t={t}"):
                Z = self._clean_array_nan_inf(Z, default_value=0.0)
                _logger.warning(f"kalman_filter_forward: Z (prior) contains NaN/Inf at t={t}, cleaned to zero")
            V = ensure_real_and_symmetric(V)
            if not self._check_finite(V, f"V (prior) at t={t}"):
                V = self._get_regularized_identity(V.shape[0], dtype, scale=10.0)
                _logger.warning(f"kalman_filter_forward: V (prior) contains NaN/Inf at t={t}, using regularized identity")
            Zm[:, t] = Z
            Vm[:, :, t] = V
            
            # Store covariance and state values for t (posteriors)
            # i.e. Zu = Z_t|t   & Vu = V_t|t
            # Ensure Zu and Vu are finite before storing
            Zu = self._ensure_real(Zu)
            if not self._check_finite(Zu, f"Zu (posterior) at t={t}"):
                Zu = self._clean_array_nan_inf(Zu, default_value=0.0)
                _logger.warning(f"kalman_filter_forward: Zu (posterior) contains NaN/Inf at t={t}, cleaned to zero")
            Vu = ensure_real_and_symmetric(Vu)
            if not self._check_finite(Vu, f"Vu (posterior) at t={t}"):
                Vu = self._get_regularized_identity(Vu.shape[0], dtype, scale=10.0)
                Vu = self._ensure_cov_stable(Vu, min_eigenval=self.min_eigenval, ensure_real=True)
                _logger.warning(f"kalman_filter_forward: Vu (posterior) contains NaN/Inf at t={t}, using regularized identity")
            ZmU[:, t + 1] = Zu
            VmU[:, :, t + 1] = Vu
        
        # Store Kalman gain k_t (from final iteration)
        # k_t should be m x n_obs where n_obs is number of observed series at final time
        # VCF is m x n_obs, C_t is n_obs x m, so VCF @ C_t gives m x m
        # However, if no observations at final time, use zeros
        if len(Y_t) == 0 or VCF is None or C_t is None:
            k_t = np.zeros((m, m), dtype=dtype)
        else:
            # VCF is m x n_obs, C_t is n_obs x m, so k_t = VCF @ C_t is m x m
            k_t = VCF @ C_t
        
        # Delete intermediate variables to free memory
        # These are no longer needed after creating KalmanFilterState
        del Y_t, C_t, VCF, Zu, Vu, Z, V
        
        state = KalmanFilterState(Zm=Zm, Vm=Vm, ZmU=ZmU, VmU=VmU, loglik=loglik, k_t=k_t)
        
        return state
    
    def smoother_backward(
        self,
        A: np.ndarray,
        S: KalmanFilterState
    ) -> KalmanFilterState:
        """Apply fixed-interval smoother (backward pass).
        
        Follows MATLAB FIS() function structure (lines 996-1071 in dfm.m).
        
        Parameters
        ----------
        A : np.ndarray
            Transition matrix (m x m)
        S : KalmanFilterState
            State from Kalman filter (forward pass)
            
        Returns
        -------
        KalmanFilterState
            State with smoothed estimates added (ZmT, VmT, VmT_1)
        """
        return self._smoother_backward_impl(A, S)
    
    def _smoother_backward_impl(
        self,
        A: np.ndarray,
        S: KalmanFilterState
    ) -> KalmanFilterState:
        """Internal implementation of smoother backward pass.
        
        Parameters
        ----------
        A : np.ndarray
            Transition matrix (m x m). Must be NumPy array.
        S : KalmanFilterState
            State from Kalman filter (forward pass)
            
        Returns
        -------
        KalmanFilterState
            State with smoothed estimates added (ZmT, VmT, VmT_1)
        """
        m, nobs = S.Zm.shape
        
        dtype = S.Zm.dtype
        
        # MEMORY OPTIMIZATION: Extract needed values for initialization
        # We need: ZmU[:, nobs], VmU[:, :, nobs], VmU[:, :, nobs-1], Vm[:, :, nobs-1], k_t
        ZmU_final = S.ZmU[:, nobs].copy()  # (m,)
        VmU_final = S.VmU[:, :, nobs].copy()  # (m x m)
        if nobs > 0:
            VmU_prev = S.VmU[:, :, nobs - 1].copy()  # (m x m)
            Vm_prev = S.Vm[:, :, nobs - 1].copy()  # (m x m)
        else:
            VmU_prev = VmU_final.copy()
            Vm_prev = np.eye(m, dtype=dtype)
        k_t = S.k_t  # Keep reference (already NumPy array from filter_forward)
        
        # Initialize output matrices
        ZmT = np.empty((m, nobs + 1), dtype=dtype)
        VmT = np.empty((m, m, nobs + 1), dtype=dtype)
        
        # Fill the final period of ZmT, VmT with SKF posterior values (using extracted values)
        # Check and clean initial values
        ZmT_init = ZmU_final.copy()
        if not self._check_finite(ZmT_init, f"ZmU[:, nobs] (initial smoother state)"):
            # If initial state contains NaN, use zero as fallback
            ZmT_init = self._clean_array_nan_inf(ZmT_init, default_value=0.0)
            _logger.warning("smoother_backward: Initial ZmU[:, nobs] contains NaN/Inf, using zero as fallback")
        ZmT[:, nobs] = ZmT_init
        
        VmT_init = VmU_final.copy()
        if not self._check_finite(VmT_init, f"VmU[:, :, nobs] (initial smoother covariance)"):
            # If initial covariance contains NaN, use identity with small regularization as fallback
            VmT_init = self._get_regularized_identity(m, dtype, scale=10.0)
            _logger.warning("smoother_backward: Initial VmU[:, :, nobs] contains NaN/Inf, using regularized identity as fallback")
        else:
            # Ensure initial covariance is PSD even if finite
            VmT_init = self._ensure_cov_stable(VmT_init, min_eigenval=self.min_eigenval, ensure_real=True)
        VmT[:, :, nobs] = VmT_init
        
        # Early termination if initial state contains NaN
        # This prevents NaN from propagating through entire backward pass
        initial_state_has_nan = not self._check_finite(ZmT[:, nobs], f"ZmT[:, nobs] (initial)") or not self._check_finite(VmT[:, :, nobs], f"VmT[:, :, nobs] (initial)")
        if initial_state_has_nan:
            _logger.warning("smoother_backward: Initial smoothed state contains NaN/Inf after cleaning. Using forward-only estimates (skipping backward pass) to prevent NaN propagation.")
            # Fill all ZmT with forward estimates (using extracted final value for all)
            for t in range(nobs + 1):
                ZmT_forward = ZmU_final.copy()
                if not self._check_finite(ZmT_forward, f"ZmU_final (forward fallback)"):
                    ZmT_forward = self._clean_array_nan_inf(ZmT_forward, default_value=0.0)
                ZmT[:, t] = ZmT_forward
                
                VmT_forward = VmU_final.copy()
                if not self._check_finite(VmT_forward, f"VmU_final (forward fallback)"):
                    VmT_forward = self._get_regularized_identity(m, dtype, scale=10.0)
                    VmT_forward = self._ensure_cov_stable(VmT_forward, min_eigenval=self.min_eigenval, ensure_real=True)
                else:
                    VmT_forward = self._ensure_cov_stable(VmT_forward, min_eigenval=self.min_eigenval, ensure_real=True)
                VmT[:, :, t] = VmT_forward
            
            # Initialize VmT_1 with zeros (no lag-1 covariance available)
            VmT_1 = np.zeros((m, m, nobs), dtype=dtype)
            
            # Add smoothed estimates as attributes (using forward-only estimates)
            S.ZmT = ZmT
            S.VmT = VmT
            S.VmT_1 = VmT_1
            
            return S
        
        # Initialize VmT_1 lag 1 covariance matrix for final period
        VmT_1 = np.empty((m, m, nobs), dtype=dtype)
        # Check VmU_prev (extracted earlier) before using it
        VmU_nobs_minus_1 = VmU_prev.copy()
        if not self._check_finite(VmU_nobs_minus_1, f"VmU_prev (for VmT_1 initialization)"):
            # Use regularized identity as fallback
            VmU_nobs_minus_1 = self._get_regularized_identity(m, dtype, scale=10.0)
            _logger.warning("smoother_backward: VmU_prev contains NaN/Inf, using regularized identity for VmT_1")
        VmT_1_temp = (np.eye(m, dtype=dtype) - k_t) @ A @ VmU_nobs_minus_1
        VmT_1[:, :, nobs - 1] = ensure_real_and_symmetric(VmT_1_temp)
        
        # Used for recursion process
        # Check forward pass values (extracted earlier) before computing J_2
        VmU_for_J2 = VmU_prev.copy()
        Vm_for_J2 = Vm_prev.copy()
        if not self._check_finite(VmU_for_J2, f"VmU_prev (for J_2)") or not self._check_finite(Vm_for_J2, f"Vm_prev (for J_2)"):
            # If forward pass values contain NaN, use zero matrix as fallback
            J_2 = np.zeros((m, m), dtype=dtype)
            _logger.warning("smoother_backward: Forward pass values contain NaN/Inf, using zero matrix for J_2")
        else:
            try:
                J_2 = VmU_for_J2 @ A.T @ np.linalg.pinv(Vm_for_J2)
                # Check if J_2 contains NaN/Inf
                if not self._check_finite(J_2, f"J_2 (initial)"):
                    J_2 = np.zeros((m, m), dtype=dtype)
                    _logger.warning("smoother_backward: J_2 contains NaN/Inf, using zero matrix as fallback")
            except (RuntimeError, ValueError, np.linalg.LinAlgError):
                # Fallback if pinv fails
                J_2 = np.zeros((m, m), dtype=dtype)
                _logger.warning("smoother_backward: pinv failed for J_2, using zero matrix as fallback")
        
        # Run smoothing algorithm
        # Loop through time reverse-chronologically (starting at final period nobs-1)
        # Follows MATLAB FIS() function structure
        cleanup_interval = max(100, nobs // 100) if nobs > self.chunk_threshold else nobs + 1
        for t in range(nobs - 1, -1, -1):
            # Store posterior and prior factor covariance values
            VmU = S.VmU[:, :, t].copy()
            Vm1 = S.Vm[:, :, t].copy()
            
            # Store previous period smoothed factor covariance and lag-1 covariance
            V_T = VmT[:, :, t + 1].copy()
            V_T1 = VmT_1[:, :, t].copy() if t < nobs - 1 else np.zeros((m, m), dtype=dtype)
            
            # Check and clean V_T before using in recursion to prevent NaN propagation
            if not self._check_finite(V_T, f"V_T (VmT[:, :, t+1]) at t={t}"):
                # If previous smoothed covariance is invalid, use posterior covariance as fallback
                V_T = VmU.copy()
                V_T = self._ensure_cov_stable(V_T, min_eigenval=self.min_eigenval, ensure_real=True)
                _logger.warning(f"smoother_backward: V_T contains NaN/Inf at t={t}, using VmU as fallback")
            
            # Check and clean ZmT[:, t+1] before using in recursion
            ZmT_next = ZmT[:, t + 1].copy()
            if not self._check_finite(ZmT_next, f"ZmT[:, t+1] at t={t}"):
                # If next smoothed state is invalid, use posterior state as fallback
                ZmT_next = S.ZmU[:, t].copy()
                ZmT_next = self._clean_array_nan_inf(ZmT_next, default_value=0.0)
                _logger.warning(f"smoother_backward: ZmT[:, t+1] contains NaN/Inf at t={t}, using ZmU as fallback")
            
            J_1 = J_2.copy()
            
            # Check J_1 before using in recursion
            if not self._check_finite(J_1, f"J_1 at t={t}"):
                # If J_1 contains NaN, use zero matrix as fallback
                J_1 = np.zeros((m, m), dtype=dtype)
                _logger.warning(f"smoother_backward: J_1 contains NaN/Inf at t={t}, using zero matrix as fallback")
            
            # Check forward pass values before using in recursion
            ZmU_t = S.ZmU[:, t].copy()
            if not self._check_finite(ZmU_t, f"ZmU[:, t] at t={t}"):
                ZmU_t = self._clean_array_nan_inf(ZmU_t, default_value=0.0)
                _logger.warning(f"smoother_backward: ZmU[:, t] contains NaN/Inf at t={t}, using cleaned values")
            
            # Update smoothed factor estimate
            # Check all inputs before recursion to prevent NaN propagation
            if not self._check_finite(ZmT_next, f"ZmT[:, t+1] at t={t}") or not self._check_finite(ZmU_t, f"ZmU[:, t] at t={t}") or not self._check_finite(J_1, f"J_1 at t={t}"):
                # If any input contains NaN, use forward estimate as fallback
                ZmT[:, t] = self._clean_array_nan_inf(ZmU_t.copy(), default_value=0.0)
                _logger.warning(f"smoother_backward: Input contains NaN/Inf at t={t}, using ZmU as fallback (skipping recursion)")
            else:
                # Safe to perform recursion
                try:
                    ZmT[:, t] = ZmU_t + J_1 @ (ZmT_next - A @ ZmU_t)
                except (RuntimeError, ValueError, np.linalg.LinAlgError) as e:
                    _logger.warning(f"smoother_backward: Recursion failed at t={t}: {e}, using ZmU as fallback")
                    ZmT[:, t] = self._clean_array_nan_inf(ZmU_t.copy(), default_value=0.0)
            
            # Clean NaN/Inf after recursion
            if not self._check_finite(ZmT[:, t], f"ZmT[:, t] at t={t}"):
                # If recursion produces NaN, fall back to forward pass value
                ZmT[:, t] = self._clean_array_nan_inf(ZmU_t.copy(), default_value=0.0)
                _logger.warning(f"smoother_backward: ZmT[:, t] recursion produced NaN/Inf at t={t}, using ZmU as fallback")
            
            # Check forward pass covariance values before using in recursion
            if not self._check_finite(VmU, f"VmU[:, :, t] at t={t}"):
                VmU = self._get_regularized_identity(m, dtype, scale=10.0)
                _logger.warning(f"smoother_backward: VmU[:, :, t] contains NaN/Inf at t={t}, using regularized identity as fallback")
            if not self._check_finite(Vm1, f"Vm[:, :, t] at t={t}"):
                Vm1 = self._get_regularized_identity(m, dtype, scale=10.0)
                _logger.warning(f"smoother_backward: Vm[:, :, t] contains NaN/Inf at t={t}, using regularized identity as fallback")
            
            # Update smoothed factor covariance matrix
            # Check all inputs before recursion to prevent NaN propagation
            if not self._check_finite(V_T, f"V_T (VmT[:, :, t+1]) at t={t}") or not self._check_finite(VmU, f"VmU[:, :, t] at t={t}") or not self._check_finite(Vm1, f"Vm[:, :, t] at t={t}") or not self._check_finite(J_1, f"J_1 at t={t}"):
                # If any input contains NaN, use forward estimate as fallback
                VmT[:, :, t] = VmU.copy()
                VmT[:, :, t] = self._ensure_cov_stable(VmT[:, :, t], min_eigenval=self.min_eigenval, ensure_real=True)
                _logger.warning(f"smoother_backward: Input contains NaN/Inf at t={t}, using VmU as fallback (skipping recursion)")
            else:
                # Safe to perform recursion
                try:
                    VmT_temp = VmU + J_1 @ (V_T - Vm1) @ J_1.T
                    VmT[:, :, t] = ensure_real_and_symmetric(VmT_temp)
                except (RuntimeError, ValueError, np.linalg.LinAlgError) as e:
                    _logger.warning(f"smoother_backward: Covariance recursion failed at t={t}: {e}, using VmU as fallback")
                    VmT[:, :, t] = VmU.copy()
                    VmT[:, :, t] = self._ensure_cov_stable(VmT[:, :, t], min_eigenval=self.min_eigenval, ensure_real=True)
            
            # Clean NaN/Inf and ensure PSD
            if not self._check_finite(VmT[:, :, t], f"VmT[:, :, t] at t={t}"):
                # If recursion produces NaN, fall back to forward pass value
                VmT[:, :, t] = VmU.copy()
                VmT[:, :, t] = self._ensure_cov_stable(VmT[:, :, t], min_eigenval=self.min_eigenval, ensure_real=True)
                _logger.warning(f"smoother_backward: VmT[:, :, t] recursion produced NaN/Inf at t={t}, using VmU as fallback")
            
            if t > 0:
                # Update weight
                # Check forward pass values before computing J_2
                VmU_t_minus_1 = S.VmU[:, :, t - 1].copy()
                Vm_t_minus_1 = S.Vm[:, :, t - 1].copy()
                if not self._check_finite(VmU_t_minus_1, f"VmU[:, :, t - 1] at t={t}") or not self._check_finite(Vm_t_minus_1, f"Vm[:, :, t - 1] at t={t}"):
                    # If forward pass values contain NaN, use zero matrix as fallback
                    J_2 = np.zeros((m, m), dtype=dtype)
                    _logger.warning(f"smoother_backward: Forward pass values contain NaN/Inf at t={t}, using zero matrix for J_2")
                else:
                    try:
                        J_2 = VmU_t_minus_1 @ A.T @ np.linalg.pinv(Vm_t_minus_1)
                        # Check if J_2 contains NaN/Inf
                        if not self._check_finite(J_2, f"J_2 at t={t}"):
                            J_2 = np.zeros((m, m), dtype=dtype)
                            _logger.warning(f"smoother_backward: J_2 contains NaN/Inf at t={t}, using zero matrix as fallback")
                    except (RuntimeError, ValueError, np.linalg.LinAlgError):
                        J_2 = np.zeros((m, m), dtype=dtype)
                        _logger.warning(f"smoother_backward: pinv failed for J_2 at t={t}, using zero matrix as fallback")
                
                # Update lag 1 factor covariance matrix
                # Check V_T1 before using in recursion
                if not self._check_finite(V_T1, f"VmT_1[:, :, t] at t={t}"):
                    V_T1 = np.zeros((m, m), dtype=dtype)
                    _logger.warning(f"smoother_backward: VmT_1[:, :, t] contains NaN/Inf at t={t}, using zero matrix as fallback")
                
                VmT_1_temp = VmU @ J_2.T + J_1 @ (V_T1 - A @ VmU) @ J_2.T
                VmT_1[:, :, t - 1] = ensure_real_and_symmetric(VmT_1_temp)
                
                # Check VmT_1 result
                if not self._check_finite(VmT_1[:, :, t - 1], f"VmT_1[:, :, t - 1] at t={t}"):
                    VmT_1[:, :, t - 1] = np.zeros((m, m), dtype=dtype)
                    _logger.warning(f"smoother_backward: VmT_1[:, :, t - 1] recursion produced NaN/Inf at t={t}, using zero matrix as fallback")
                
                # Periodic memory cleanup for large datasets
                if nobs > self.chunk_threshold and (nobs - t) % cleanup_interval == 0:
                    # Delete temporary variables to free memory
                    del VmU, Vm1, ZmU_t, VmU_t_minus_1, Vm_t_minus_1
                    # Force Python garbage collection
                    import gc
                    gc.collect()
            else:
                # Periodic memory cleanup for large datasets (when t == 0)
                if nobs > self.chunk_threshold and (nobs - t) % cleanup_interval == 0:
                    # Delete temporary variables to free memory
                    del VmU, Vm1, ZmU_t
                    # Force Python garbage collection
                    import gc
                    gc.collect()
        
        # Add smoothed estimates as attributes
        S.ZmT = ZmT
        S.VmT = VmT
        S.VmT_1 = VmT_1
        
        # Note: We keep Zm, Vm, ZmU, VmU in the state for potential debugging
        # but they're no longer needed after backward pass
        # Python garbage collector will free memory when references are cleared
        
        return S
    
    def handle_missing_data(
        self,
        y: np.ndarray, 
        C: np.ndarray, 
        R: np.ndarray
    ) -> Tuple[np.ndarray, np.ndarray, np.ndarray, np.ndarray]:
        """Handle missing data by removing NaN observations from the Kalman filter equations.
        
        Follows MATLAB MissData() function structure (lines 1074-1108 in dfm.m).
        
        Parameters
        ----------
        y : np.ndarray
            Vector of observations at time t, shape (k,) where k is number of series.
            Missing values should be NaN.
        C : np.ndarray
            Observation/loading matrix, shape (k x m) where m is state dimension.
            Each row corresponds to a series in y.
        R : np.ndarray
            Covariance matrix for observation residuals, shape (k x k).
            Typically diagonal (idiosyncratic variances).
            
        Returns
        -------
        y_clean : np.ndarray
            Reduced observation vector with NaN values removed, shape (k_obs,)
            where k_obs is number of non-missing observations.
        C_clean : np.ndarray
            Reduced observation matrix, shape (k_obs x m).
            Rows corresponding to missing observations are removed.
        R_clean : np.ndarray
            Reduced covariance matrix, shape (k_obs x k_obs).
            Rows and columns corresponding to missing observations are removed.
        L : np.ndarray
            Selection matrix, shape (k x k_obs), used to restore standard dimensions.
            L @ y_clean gives y with zeros for missing values.
        """
        # Returns True for nonmissing series
        ix = ~np.isnan(y)
        
        # Index for columns with nonmissing variables
        k = len(y)
        e = np.eye(k, dtype=y.dtype)
        L = e[:, ix]
        
        # Remove missing series
        y = y[ix]
        
        # Remove missing series from observation matrix
        C = C[ix, :]
        
        # Remove missing series from covariance matrix
        R = R[np.ix_(ix, ix)]
        
        return y, C, R, L
    
    # Private helper methods (delegate to utils module)
    def _clean_array_nan_inf(self, arr: np.ndarray, default_value: float = 0.0) -> np.ndarray:
        """Clean array by removing NaN/Inf values.
        
        Parameters
        ----------
        arr : np.ndarray
            Array to clean
        default_value : float
            Value to replace NaN/Inf with
            
        Returns
        -------
        np.ndarray
            Cleaned array
        """
        return np.nan_to_num(arr, nan=default_value, posinf=default_value, neginf=default_value)
    
    def _get_regularized_identity(self, size: int, dtype: np.dtype, 
                                   scale: float = 10.0) -> np.ndarray:
        """Get regularized identity matrix for fallback values.
        
        Parameters
        ----------
        size : int
            Size of identity matrix
        dtype : np.dtype
            Dtype for array
        scale : float
            Scaling factor for regularization
            
        Returns
        -------
        np.ndarray
            Regularized identity matrix
        """
        return np.eye(size, dtype=dtype) * (self.min_eigenval * scale)
    
    def _check_finite(self, arr: np.ndarray, name: str = "array") -> bool:
        """Check if array contains only finite values."""
        return check_finite(arr, name)
    
    def _ensure_real(self, arr: np.ndarray) -> np.ndarray:
        """Ensure array is real by extracting real part if complex."""
        return ensure_real(arr)
    
    def _ensure_symmetric(self, arr: np.ndarray) -> np.ndarray:
        """Ensure matrix is symmetric by averaging with its transpose."""
        return ensure_symmetric(arr)
    
    def _ensure_cov_stable(
        self,
        M: np.ndarray, 
        min_eigenval: Optional[float] = None,
        ensure_real: bool = True
    ) -> np.ndarray:
        """Ensure covariance matrix is real, symmetric, and positive semi-definite."""
        if min_eigenval is None:
            min_eigenval = self.min_eigenval
        return ensure_covariance_stable(M, min_eigenval=min_eigenval, ensure_real_flag=ensure_real)
    
    def _cap_max_eigenval(self, M: np.ndarray, max_eigenval: float = 1e6) -> np.ndarray:
        """Cap maximum eigenvalue of matrix to prevent numerical explosion."""
        return cap_max_eigenval(M, max_eigenval=max_eigenval, warn=False)
    
    def _clean_matrix(
        self,
        M: np.ndarray, 
        matrix_type: str = 'general', 
        default_nan: float = 0.0, 
        default_inf: Optional[float] = None
    ) -> np.ndarray:
        """Clean matrix by removing NaN/Inf values and ensuring numerical stability."""
        # Convert None to a default value for default_inf
        default_inf_val = default_inf if default_inf is not None else 1e6
        return clean_matrix(
            M, 
            matrix_type=matrix_type,
            default_nan=default_nan,
            default_inf=default_inf_val,
            min_eigenval=self.min_eigenval,
            min_diagonal_variance=self.min_diagonal_variance
        )
    
    def _safe_inv(
        self,
        M: np.ndarray,
        regularization: Optional[float] = None,
        use_pinv_fallback: bool = True
    ) -> np.ndarray:
        """Safely compute matrix inverse with robust error handling."""
        if regularization is None:
            regularization = self.inv_regularization
        return safe_inverse(M, regularization=regularization, use_pinv_fallback=use_pinv_fallback)
    
    def _safe_det(self, M: np.ndarray, use_logdet: bool = True) -> float:
        """Compute determinant safely to avoid overflow warnings."""
        return safe_determinant(M, use_logdet=use_logdet)

