import glob
from typing import Literal, Optional
from django.template import Library
from .. import NODE_MODULES_DIR, utils

register = Library()


@register.inclusion_tag('django_node_modules/includes/hrenpack/css.html')
def hrenpack_js_css(version: Optional[str] = None):
    """CSS для hrenpack_js"""
    package_name = 'hrenpack_js'
    is_local = utils.get_local_check(package_name, version)
    base_url = utils.get_hrenpack_base_context(package_name, is_local, version)

    if is_local:
        with utils.change_dir(NODE_MODULES_DIR / package_name):
            files = glob.glob('*.css')
    else:
        files = utils.get_hrenpack_js_files(version or None, files_extension='.css')

    return dict(base_url=base_url, files=files)


@register.inclusion_tag('django_node_modules/includes/hrenpack_js.html')
def hrenpack_js_scripts(type: Literal['include', 'exclude', 'all'] = 'all', *args, version: Optional[str] = None):
    """JS скрипты для hrenpack_js"""
    package_name = 'hrenpack_js'
    is_local = utils.get_local_check(package_name, version)
    base_url = utils.get_hrenpack_base_context(package_name, is_local, version)
    scripts = [arg if arg.endswith('.js') else arg + '.js' for arg in args]
    if is_local:
        with utils.change_dir(NODE_MODULES_DIR / package_name):
            files = glob.glob('*.js')
    else:
        files = utils.get_hrenpack_js_files(version or None)

    return dict(
        scripts=scripts,
        type=type,
        base_url=base_url,
        files=files
    )


@register.inclusion_tag('django_node_modules/includes/hrenpack/hts-css.html')
def hrenpack_theme_style_css(config: str = 'full', version: Optional[str] = None):
    """CSS для hrenpack-theme-style"""
    package_name = 'hrenpack-theme-style'
    is_local = utils.get_local_check(package_name, version)
    base_url = utils.get_hrenpack_base_context(package_name, is_local, version)
    files = utils.get_hrenpack_theme_style_files(config, 'css')
    return dict(base_url=base_url, files=files, bootstrap_url=utils.get_bootstrap_url())


@register.inclusion_tag('django_node_modules/includes/hrenpack/hrenpack-theme-style.html')
def hrenpack_theme_style_scripts(config: str = 'full', version: Optional[str] = None, no_deps: bool = False):
    """JS скрипты для hrenpack-theme-style"""
    package_name = 'hrenpack-theme-style'
    is_local = utils.get_local_check(package_name, version)
    base_url = utils.get_hrenpack_base_context(package_name, is_local, version)
    files = utils.get_hrenpack_theme_style_files(config, 'js')
    dependencies = utils.get_hrenpack_theme_style_dependencies(config) if not no_deps else []
    return dict(base_url=base_url, files=files, dependencies=dependencies)
