import json
import sys
from typing import Any, Literal, TypedDict, cast
from uuid import uuid4

import yaml
from pydantic import (
    BaseModel,
    Field,
    field_serializer,
    field_validator,
    model_validator,
)

from docent.data_models._tiktoken_util import get_token_count, group_messages_into_ranges
from docent.data_models.transcript import (
    Transcript,
    TranscriptGroup,
    TranscriptWithoutMetadataValidator,
    fake_model_dump,
)


class FilterableField(TypedDict):
    name: str
    type: Literal["str", "bool", "int", "float"]


class AgentRun(BaseModel):
    """Represents a complete run of an agent with transcripts and metadata.

    An AgentRun encapsulates the execution of an agent, storing all communication
    transcripts and associated metadata. It must contain at least one transcript.

    Attributes:
        id: Unique identifier for the agent run, auto-generated by default.
        name: Optional human-readable name for the agent run.
        description: Optional description of the agent run.
        transcripts: Dict mapping transcript IDs to Transcript objects.
        transcript_groups: Dict mapping transcript group IDs to TranscriptGroup objects.
        metadata: Additional structured metadata about the agent run as a JSON-serializable dictionary.
    """

    id: str = Field(default_factory=lambda: str(uuid4()))
    name: str | None = None
    description: str | None = None

    transcripts: dict[str, Transcript]
    transcript_groups: dict[str, TranscriptGroup] = Field(default_factory=dict)
    metadata: dict[str, Any] = Field(default_factory=dict)

    @field_serializer("metadata")
    def serialize_metadata(self, metadata: dict[str, Any], _info: Any) -> dict[str, Any]:
        """
        Custom serializer for the metadata field - returns the dict as-is since it's already serializable.
        """
        return fake_model_dump(metadata)

    @field_validator("metadata", mode="before")
    @classmethod
    def _validate_metadata_json_serializable(cls, v: Any) -> dict[str, Any]:
        """
        Validates that metadata is a dictionary and is JSON-serializable.
        """
        if v is None:
            return {}

        if not isinstance(v, dict):
            raise ValueError(f"metadata must be a dictionary, got {type(v).__name__}")

        # Check that the metadata is JSON serializable
        try:
            json.dumps(fake_model_dump(cast(dict[str, Any], v)))
        except (TypeError, ValueError) as e:
            raise ValueError(f"metadata must be JSON-serializable: {e}")

        return cast(dict[str, Any], v)

    @model_validator(mode="after")
    def _validate_transcripts_not_empty(self):
        """Validates that the agent run contains at least one transcript.

        Raises:
            ValueError: If the transcripts list is empty.

        Returns:
            AgentRun: The validated AgentRun instance.
        """
        if len(self.transcripts) == 0:
            raise ValueError("AgentRun must have at least one transcript")
        return self

    def _to_text_impl(self, token_limit: int = sys.maxsize, use_blocks: bool = False) -> list[str]:
        """
        Core implementation for converting agent run to text representation.

        Args:
            token_limit: Maximum tokens per returned string under the GPT-4 tokenization scheme
            use_blocks: If True, use individual message blocks. If False, use action units.

        Returns:
            List of strings, each at most token_limit tokens
        """
        # Generate transcript strings using appropriate method
        transcript_strs: list[str] = []
        for i, (t_key, t) in enumerate(self.transcripts.items()):
            if use_blocks:
                transcript_content = t.to_str_blocks_with_token_limit(
                    token_limit=sys.maxsize,
                    transcript_idx=i,
                    agent_run_idx=None,
                )[0]
            else:
                transcript_content = t.to_str_with_token_limit(
                    token_limit=sys.maxsize,
                    transcript_idx=i,
                    agent_run_idx=None,
                )[0]
            transcript_strs.append(
                f"<transcript {t_key}>\n{transcript_content}\n</transcript {t_key}>"
            )

        transcripts_str = "\n\n".join(transcript_strs)

        # Gather metadata
        metadata_obj = fake_model_dump(self.metadata)
        if self.name is not None:
            metadata_obj["name"] = self.name
        if self.description is not None:
            metadata_obj["description"] = self.description

        yaml_width = float("inf")
        transcripts_str = (
            f"Here is a complete agent run for analysis purposes only:\n{transcripts_str}\n\n"
        )
        metadata_str = f"Metadata about the complete agent run:\n<agent run metadata>\n{yaml.dump(metadata_obj, width=yaml_width)}\n</agent run metadata>"

        if token_limit == sys.maxsize:
            return [f"{transcripts_str}" f"{metadata_str}"]

        # Compute message length; if fits, return the full transcript and metadata
        transcript_str_tokens = get_token_count(transcripts_str)
        metadata_str_tokens = get_token_count(metadata_str)
        if transcript_str_tokens + metadata_str_tokens <= token_limit:
            return [f"{transcripts_str}" f"{metadata_str}"]

        # Otherwise, split up the transcript and metadata into chunks
        else:
            results: list[str] = []
            transcript_token_counts = [get_token_count(t) for t in transcript_strs]
            ranges = group_messages_into_ranges(
                transcript_token_counts, metadata_str_tokens, token_limit - 50
            )
            for msg_range in ranges:
                if msg_range.include_metadata:
                    cur_transcript_str = "\n\n".join(
                        transcript_strs[msg_range.start : msg_range.end]
                    )
                    results.append(
                        f"Here is a partial agent run for analysis purposes only:\n{cur_transcript_str}"
                        f"{metadata_str}"
                    )
                else:
                    assert (
                        msg_range.end == msg_range.start + 1
                    ), "Ranges without metadata should be a single message"
                    t_id, t = list(self.transcripts.items())[msg_range.start]
                    if msg_range.num_tokens < token_limit - 50:
                        if use_blocks:
                            transcript = f"<transcript {t_id}>\n{t.to_str_blocks_with_token_limit(token_limit=sys.maxsize)[0]}\n</transcript {t_id}>"
                        else:
                            transcript = f"<transcript {t_id}>\n{t.to_str_with_token_limit(token_limit=sys.maxsize)[0]}\n</transcript {t_id}>"
                        result = (
                            f"Here is a partial agent run for analysis purposes only:\n{transcript}"
                        )
                        results.append(result)
                    else:
                        if use_blocks:
                            transcript_fragments = t.to_str_blocks_with_token_limit(
                                token_limit=token_limit - 50,
                            )
                        else:
                            transcript_fragments = t.to_str_with_token_limit(
                                token_limit=token_limit - 50,
                            )
                        for fragment in transcript_fragments:
                            result = f"<transcript {t_id}>\n{fragment}\n</transcript {t_id}>"
                            result = (
                                f"Here is a partial agent run for analysis purposes only:\n{result}"
                            )
                            results.append(result)
            return results

    def to_text(self, token_limit: int = sys.maxsize) -> list[str]:
        """
        Represents an agent run as a list of strings, each of which is at most token_limit tokens
        under the GPT-4 tokenization scheme.

        We'll try to split up long AgentRuns along transcript boundaries and include metadata.
        For very long transcripts, we'll have to split them up further and remove metadata.
        """
        return self._to_text_impl(token_limit=token_limit, use_blocks=False)

    def to_text_blocks(self, token_limit: int = sys.maxsize) -> list[str]:
        """
        Represents an agent run as a list of strings using individual message blocks,
        each of which is at most token_limit tokens under the GPT-4 tokenization scheme.

        Unlike to_text() which uses action units, this method formats each message
        as an individual block.
        """
        return self._to_text_impl(token_limit=token_limit, use_blocks=True)

    @property
    def text(self) -> str:
        """Concatenates all transcript texts with double newlines as separators.

        Returns:
            str: A string representation of all transcripts.
        """
        return self._to_text_impl(token_limit=sys.maxsize, use_blocks=False)[0]

    @property
    def text_blocks(self) -> str:
        """Concatenates all transcript texts using individual blocks format.

        Returns:
            str: A string representation of all transcripts using individual message blocks.
        """
        return self._to_text_impl(token_limit=sys.maxsize, use_blocks=True)[0]

    def model_dump(self, *args: Any, **kwargs: Any) -> dict[str, Any]:
        """Extends the parent model_dump method to include the text property.

        Args:
            *args: Variable length argument list passed to parent method.
            **kwargs: Arbitrary keyword arguments passed to parent method.

        Returns:
            dict[str, Any]: Dictionary representation of the model including the text property.
        """
        return super().model_dump(*args, **kwargs) | {"text": self.text}

    def get_filterable_fields(self, max_depth: int = 1) -> list[FilterableField]:
        """Returns a list of all fields that can be used to filter the agent run,
        by recursively exploring the model_dump() for singleton types in dictionaries.

        Returns:
            list[FilterableField]: A list of filterable fields, where each field is a
                                   dictionary containing its 'name' (path) and 'type'.
        """

        result: list[FilterableField] = []

        def _explore_dict(d: dict[str, Any], prefix: str, depth: int):
            nonlocal result

            if depth > max_depth:
                return

            for k, v in d.items():
                if isinstance(v, (str, int, float, bool)):
                    result.append(
                        {
                            "name": f"{prefix}.{k}",
                            "type": cast(Literal["str", "bool", "int", "float"], type(v).__name__),
                        }
                    )
                elif isinstance(v, dict):
                    _explore_dict(cast(dict[str, Any], v), f"{prefix}.{k}", depth + 1)

        # Look at the agent run metadata
        _explore_dict(fake_model_dump(self.metadata), "metadata", 0)
        # Look at the transcript metadata
        # TODO(mengk): restore this later when we have the ability to integrate with SQL.
        # for t_id, t in self.transcripts.items():
        #     _explore_dict(
        #         t.metadata.model_dump(strip_internal_fields=True), f"transcript.{t_id}.metadata", 0
        #     )

        # Append the text field
        result.append({"name": "text", "type": "str"})

        return result


class AgentRunWithoutMetadataValidator(AgentRun):
    """
    A version of AgentRun that doesn't have the model_validator on metadata.
    Needed for sending/receiving agent runs via JSON, since they incorrectly trip the existing model_validator.
    """

    transcripts: dict[str, TranscriptWithoutMetadataValidator]  # type: ignore

    @field_validator("metadata", mode="before")
    @classmethod
    def _validate_metadata_type(cls, v: Any) -> Any:
        # Bypass the model_validator
        return v
