# Copyright 2025 BBDevs
# Licensed under the Apache License, Version 2.0

"""Base command class for all CLI commands.

Author: A M (am@bbdevs.com)

Created At: 08 Nov 2025
"""

from __future__ import annotations

import asyncio
import traceback
from collections.abc import Awaitable, Coroutine
from typing import TYPE_CHECKING, Any

from rich.console import Console

from dockpycli.config import CLIConfig
from dockpycli.exceptions import CLIError
from dockpycli.output import OutputFormatter, get_formatter
from dockpycli.utils.error_formatter import format_error
from dockpycore.exceptions import (
    APIError,
    ConflictError,
    DockerError,
    DockerSDKError,
    NotFound,
    ValidationError,
)
from dockpycore.exceptions import (
    ConnectionError as DockerConnectionError,
)
from dockpycore.logging import get_logger
from dockpysdk import AsyncDockerClient
from dockpysdk.compose_parser import ComposeParserError


if TYPE_CHECKING:
    from rich.progress import Progress

__all__ = ["BaseCommand"]

logger = get_logger(__name__)


class BaseCommand:
    """Base class for all CLI commands - context manager.

    Provides common functionality:
    - AsyncDockerClient context management
    - Error handling and logging
    - Output formatting
    - Progress indicators

    Usage:
        cmd = BaseCommand(config)
        cmd.run(cmd._some_impl_method(...))
    """

    def __init__(self, config: CLIConfig) -> None:
        """Initialize base command with config.

        Args:
            config: CLI configuration
        """
        self.config = config
        self.logger = get_logger(self.__class__.__name__)
        self._client: AsyncDockerClient | None = None
        self._console: Console | None = None
        self._formatter: OutputFormatter | None = None

    async def __aenter__(self) -> BaseCommand:
        """Enter context manager - initialize all resources."""
        # Initialize Docker client
        self._client = AsyncDockerClient(
            socket_path=self.config.docker_socket,
            api_version=self.config.api_version,
            timeout=self.config.timeout,
        )
        await self._client.__aenter__()

        # Initialize console
        self._console = Console(no_color=self.config.no_color)

        # Initialize formatter
        self._formatter = get_formatter(self.config)

        return self

    async def __aexit__(self, exc_type: type[BaseException] | None, exc_val: BaseException | None, exc_tb: Any) -> bool:
        """Exit context manager - cleanup resources."""
        if self._client:
            await self._client.__aexit__(exc_type, exc_val, exc_tb)

        # Handle errors
        if exc_type and exc_type is not KeyboardInterrupt:
            if isinstance(exc_val, DockerSDKError):
                self.logger.error(
                    "docker_operation_failed",
                    error=str(exc_val),
                    operation=self.__class__.__name__,
                )
            else:
                self.logger.error(
                    "unexpected_error",
                    error=str(exc_val),
                    operation=self.__class__.__name__,
                )

        return False  # Don't suppress exceptions

    @property
    def client(self) -> AsyncDockerClient:
        """Get Docker client."""
        if self._client is None:
            raise RuntimeError("Command context not entered. Use 'async with' statement or 'run()' method.")
        return self._client

    @property
    def console(self) -> Console:
        """Get Rich console."""
        if self._console is None:
            raise RuntimeError("Command context not entered. Use 'async with' statement or 'run()' method.")
        return self._console

    @property
    def formatter(self) -> OutputFormatter:
        """Get output formatter."""
        if self._formatter is None:
            raise RuntimeError("Command context not entered. Use 'async with' statement or 'run()' method.")
        return self._formatter

    def run(self, coro: Coroutine[Any, Any, Any] | Awaitable[Any]) -> None:  # noqa: PLR0915
        """Run async coroutine with context manager and exception handling.

        Wraps the coroutine in 'async with self:' context and runs it with asyncio.run().
        Catches core exceptions and converts them to user-friendly CLI messages.

        Args:
            coro: Async coroutine to run

        Example:
            cmd = ContainerCommand(config)
            cmd.run(cmd._list_containers_impl(all=True, limit=None, filters=None))
        """
        console = Console(no_color=self.config.no_color)

        async def wrapper() -> None:
            async with self:
                await coro

        try:
            asyncio.run(wrapper())

        except CLIError as e:
            # CLI-specific errors are already user-friendly
            console.print(format_error(e.message))
            raise SystemExit(e.exit_code) from e

        except NotFound as e:
            # Convert core NotFound to user-friendly message
            resource_type = e.context.get("resource_type", "resource")
            resource_id = e.context.get("resource_id", "unknown")
            console.print(format_error(f"No such {resource_type}: {resource_id}"))
            raise SystemExit(1) from e

        except ValidationError as e:
            # Convert core ValidationError to user-friendly message
            field = e.details.get("field", "input")
            console.print(format_error(f"Invalid {field}: {e.message}"))
            raise SystemExit(2) from e

        except DockerConnectionError as e:
            # Convert core ConnectionError to user-friendly message
            endpoint = e.details.get("endpoint", "Docker daemon")
            reason = e.details.get("reason") if self.config.debug else None
            console.print(format_error(f"Cannot connect to Docker daemon at {endpoint}", details=reason))
            raise SystemExit(1) from e

        except ConflictError as e:
            # Convert core ConflictError to user-friendly message
            details = str(e.response) if self.config.debug and hasattr(e, "response") and e.response else None
            console.print(format_error(f"Conflict: {e.message}", details=details))
            raise SystemExit(1) from e

        except APIError as e:
            # Convert core APIError to user-friendly message
            status_code = getattr(e, "status_code", "unknown")
            details = str(e.response) if self.config.debug and hasattr(e, "response") and e.response else None
            console.print(format_error(f"Docker API error ({status_code}): {e.message}", details=details))
            raise SystemExit(1) from e

        except DockerError as e:
            # Generic core Docker error
            details = str(e.details) if self.config.debug and e.details else None
            console.print(format_error(e.message, details=details))
            raise SystemExit(1) from e

        except ComposeParserError as e:
            # Compose file parsing error
            hint = (
                "Make sure you have a docker-compose.yml file in the current directory" if self.config.debug else None
            )
            console.print(format_error(str(e), hint=hint))
            raise SystemExit(1) from e

        except KeyboardInterrupt:
            console.print("\n[yellow]Interrupted by user[/yellow]")
            raise SystemExit(130) from None

        except Exception as e:
            # Unexpected error
            if self.config.debug:
                console.print_exception()
            else:
                console.print(format_error(str(e)))
            raise SystemExit(1) from e

    def handle_error(self, error: Exception) -> None:
        """Handle error with user-friendly message.

        Args:
            error: Exception to handle
        """
        if self._console is None:
            self._console = Console(no_color=self.config.no_color)

        if isinstance(error, DockerSDKError):
            if self.config.debug:
                self.console.print(f"[red]Error:[/red] {error}", style="red")
                self.console.print(f"[dim]Details:[/dim] {error.__class__.__name__}")
            else:
                self.console.print(f"[red]Error:[/red] {error!s}", style="red")
        elif self.config.debug:
            self.console.print(f"[red]Unexpected error:[/red] {error}", style="red")
            self.console.print(traceback.format_exc())
        else:
            self.console.print(
                "[red]An unexpected error occurred. Use --debug for details.[/red]",
                style="red",
            )

    def show_progress(self, description: str = "Processing...") -> Progress:  # noqa: ARG002
        """Show progress indicator for long operations.

        Args:
            description: Progress description (not used here, but available for custom progress)

        Returns:
            Rich Progress instance

        Example:
            with cmd.show_progress("Pulling image...") as progress:
                task = progress.add_task(description, total=100)
                # update progress
        """
        from rich.progress import Progress, SpinnerColumn, TextColumn  # noqa: PLC0415

        if self._console is None:
            self._console = Console(no_color=self.config.no_color)

        return Progress(
            SpinnerColumn(),
            TextColumn("[progress.description]{task.description}"),
            console=self._console,
            disable=self.config.no_color,
        )
