# Copyright 2025 BBDevs
# Licensed under the Apache License, Version 2.0

"""Compose command Click entry points.

Author: A M (am@bbdevs.com)

Created At: 08 Nov 2025
"""

from __future__ import annotations

import sys

import click

from dockpycli.commands.compose.command import ComposeCommand
from dockpycli.config import CLIConfig
from dockpycore.exceptions import DockerSDKError


__all__ = [
    "compose_build",
    "compose_config",
    "compose_down",
    "compose_kill",
    "compose_logs",
    "compose_ls",
    "compose_pause",
    "compose_ps",
    "compose_pull",
    "compose_restart",
    "compose_rm",
    "compose_start",
    "compose_stop",
    "compose_unpause",
    "compose_up",
]


@click.command()
@click.argument("services", nargs=-1, required=False)
@click.option("--build", is_flag=True, help="Build images before starting")
@click.option("--detach", "-d", is_flag=True, help="Run in detached mode")
@click.option("--force-recreate", is_flag=True, help="Recreate containers even if unchanged")
@click.option("--env", help="Environment name (e.g., dev, prod)")
@click.option("-f", "--file", "explicit_files", multiple=True, help="Compose file paths")
@click.option("-p", "--project-name", help="Project name")
@click.pass_context
def compose_up(
    ctx: click.Context,
    services: tuple[str, ...],
    build: bool,
    detach: bool,
    force_recreate: bool,
    env: str | None,
    explicit_files: tuple[str, ...],
    project_name: str | None,
) -> None:
    """Start services."""
    config: CLIConfig = ctx.obj.config
    cmd = ComposeCommand(
        config=config,
        environment=env,
        explicit_files=list(explicit_files) if explicit_files else None,
        project_name=project_name,
        service_filter=list(services) if services else None,
    )

    try:
        cmd.run(
            cmd._up_impl(
                services=list(services) if services else None,
                build=build,
                detach=detach,
                force_recreate=force_recreate,
            )
        )
    except DockerSDKError as e:
        cmd.console.print(f"[red]Error starting services:[/red] {e}", err=True)
        sys.exit(1)


@click.command()
@click.argument("services", nargs=-1, required=False)
@click.option("--volumes", "-v", is_flag=True, help="Remove volumes")
@click.option("--remove-orphans", is_flag=True, help="Remove orphaned containers")
@click.option("--env", help="Environment name (e.g., dev, prod)")
@click.option("-f", "--file", "explicit_files", multiple=True, help="Compose file paths")
@click.option("-p", "--project-name", help="Project name")
@click.pass_context
def compose_down(
    ctx: click.Context,
    services: tuple[str, ...],
    volumes: bool,
    remove_orphans: bool,
    env: str | None,
    explicit_files: tuple[str, ...],
    project_name: str | None,
) -> None:
    """Stop and remove services."""
    config: CLIConfig = ctx.obj.config
    cmd = ComposeCommand(
        config=config,
        environment=env,
        explicit_files=list(explicit_files) if explicit_files else None,
        project_name=project_name,
        service_filter=list(services) if services else None,
    )

    try:
        cmd.run(
            cmd._down_impl(
                services=list(services) if services else None,
                volumes=volumes,
                remove_orphans=remove_orphans,
            )
        )
    except DockerSDKError as e:
        cmd.console.print(f"[red]Error stopping services:[/red] {e}", err=True)
        sys.exit(1)


@click.command()
@click.argument("services", nargs=-1, required=False)
@click.option("--env", help="Environment name (e.g., dev, prod)")
@click.option("-f", "--file", "explicit_files", multiple=True, help="Compose file paths")
@click.option("-p", "--project-name", help="Project name")
@click.pass_context
def compose_ps(
    ctx: click.Context,
    services: tuple[str, ...],
    env: str | None,
    explicit_files: tuple[str, ...],
    project_name: str | None,
) -> None:
    """List services."""
    config: CLIConfig = ctx.obj.config
    cmd = ComposeCommand(
        config=config,
        environment=env,
        explicit_files=list(explicit_files) if explicit_files else None,
        project_name=project_name,
        service_filter=list(services) if services else None,
    )
    cmd.run(cmd._ps_impl(services=list(services) if services else None))


@click.command()
@click.option("--all", "-a", is_flag=True, help="Show all projects (default: only running)")
@click.option("--format", "format_template", help="Format output using a template")
@click.pass_context
def compose_ls(
    ctx: click.Context,
    all: bool,
    format_template: str | None,
) -> None:
    """List compose projects."""
    config: CLIConfig = ctx.obj.config
    cmd = ComposeCommand(config)
    cmd.run(cmd._ls_impl(all_projects=all, format_template=format_template))


@click.command()
@click.option("--env", help="Environment name (e.g., dev, prod)")
@click.option("-f", "--file", "explicit_files", multiple=True, help="Compose file paths")
@click.option("--resolve-image-digests", is_flag=True, help="Resolve image tags to digests")
@click.pass_context
def compose_config(
    ctx: click.Context,
    env: str | None,
    explicit_files: tuple[str, ...],
    resolve_image_digests: bool,
) -> None:
    """Show merged compose file."""
    config: CLIConfig = ctx.obj.config
    cmd = ComposeCommand(
        config=config,
        environment=env,
        explicit_files=list(explicit_files) if explicit_files else None,
    )
    cmd.run(cmd._config_impl(resolve_image_digests=resolve_image_digests))


@click.command()
@click.argument("services", nargs=-1, required=False)
@click.option("--follow", "-f", is_flag=True, help="Follow log output")
@click.option("--tail", type=int, help="Number of lines to show from end")
@click.option("--since", help="Show logs since timestamp")
@click.option("--until", help="Show logs before timestamp")
@click.option("--env", help="Environment name (e.g., dev, prod)")
@click.option("-f", "--file", "explicit_files", multiple=True, help="Compose file paths")
@click.option("-p", "--project-name", help="Project name")
@click.pass_context
def compose_logs(
    ctx: click.Context,
    services: tuple[str, ...],
    follow: bool,
    tail: int | None,
    since: str | None,
    until: str | None,
    env: str | None,
    explicit_files: tuple[str, ...],
    project_name: str | None,
) -> None:
    """View service logs."""
    config: CLIConfig = ctx.obj.config
    cmd = ComposeCommand(
        config=config,
        environment=env,
        explicit_files=list(explicit_files) if explicit_files else None,
        project_name=project_name,
        service_filter=list(services) if services else None,
    )
    cmd.run(
        cmd._logs_impl(
            services=list(services) if services else None,
            follow=follow,
            tail=tail,
            since=since,
            until=until,
        )
    )


@click.command()
@click.argument("services", nargs=-1, required=False)
@click.option("--env", help="Environment name (e.g., dev, prod)")
@click.option("-f", "--file", "explicit_files", multiple=True, help="Compose file paths")
@click.option("-p", "--project-name", help="Project name")
@click.pass_context
def compose_build(
    ctx: click.Context,
    services: tuple[str, ...],
    env: str | None,
    explicit_files: tuple[str, ...],
    project_name: str | None,
) -> None:
    """Build service images."""
    config: CLIConfig = ctx.obj.config
    cmd = ComposeCommand(
        config=config,
        environment=env,
        explicit_files=list(explicit_files) if explicit_files else None,
        project_name=project_name,
        service_filter=list(services) if services else None,
    )

    try:
        cmd.run(cmd._build_impl(services=list(services) if services else None))
    except DockerSDKError as e:
        cmd.console.print(f"[red]Error building services:[/red] {e}", err=True)
        sys.exit(1)


@click.command()
@click.argument("services", nargs=-1, required=False)
@click.option("--env", help="Environment name (e.g., dev, prod)")
@click.option("-f", "--file", "explicit_files", multiple=True, help="Compose file paths")
@click.option("-p", "--project-name", help="Project name")
@click.pass_context
def compose_start(
    ctx: click.Context,
    services: tuple[str, ...],
    env: str | None,
    explicit_files: tuple[str, ...],
    project_name: str | None,
) -> None:
    """Start services."""
    config: CLIConfig = ctx.obj.config
    cmd = ComposeCommand(
        config=config,
        environment=env,
        explicit_files=list(explicit_files) if explicit_files else None,
        project_name=project_name,
        service_filter=list(services) if services else None,
    )

    try:
        cmd.run(cmd._start_impl(services=list(services) if services else None))
    except DockerSDKError as e:
        cmd.console.print(f"[red]Error starting services:[/red] {e}", err=True)
        sys.exit(1)


@click.command()
@click.argument("services", nargs=-1, required=False)
@click.option("--timeout", "-t", type=int, default=10, help="Seconds to wait before killing")
@click.option("--env", help="Environment name (e.g., dev, prod)")
@click.option("-f", "--file", "explicit_files", multiple=True, help="Compose file paths")
@click.option("-p", "--project-name", help="Project name")
@click.pass_context
def compose_stop(
    ctx: click.Context,
    services: tuple[str, ...],
    timeout: int,
    env: str | None,
    explicit_files: tuple[str, ...],
    project_name: str | None,
) -> None:
    """Stop services."""
    config: CLIConfig = ctx.obj.config
    cmd = ComposeCommand(
        config=config,
        environment=env,
        explicit_files=list(explicit_files) if explicit_files else None,
        project_name=project_name,
        service_filter=list(services) if services else None,
    )

    try:
        cmd.run(cmd._stop_impl(services=list(services) if services else None, timeout=timeout))
    except DockerSDKError as e:
        cmd.console.print(f"[red]Error stopping services:[/red] {e}", err=True)
        sys.exit(1)


@click.command()
@click.argument("services", nargs=-1, required=False)
@click.option("--timeout", "-t", type=int, default=10, help="Seconds to wait before killing")
@click.option("--env", help="Environment name (e.g., dev, prod)")
@click.option("-f", "--file", "explicit_files", multiple=True, help="Compose file paths")
@click.option("-p", "--project-name", help="Project name")
@click.pass_context
def compose_restart(
    ctx: click.Context,
    services: tuple[str, ...],
    timeout: int,
    env: str | None,
    explicit_files: tuple[str, ...],
    project_name: str | None,
) -> None:
    """Restart services."""
    config: CLIConfig = ctx.obj.config
    cmd = ComposeCommand(
        config=config,
        environment=env,
        explicit_files=list(explicit_files) if explicit_files else None,
        project_name=project_name,
        service_filter=list(services) if services else None,
    )

    try:
        cmd.run(cmd._restart_impl(services=list(services) if services else None, timeout=timeout))
    except DockerSDKError as e:
        cmd.console.print(f"[red]Error restarting services:[/red] {e}", err=True)
        sys.exit(1)


@click.command()
@click.argument("services", nargs=-1, required=False)
@click.option("--env", help="Environment name (e.g., dev, prod)")
@click.option("-f", "--file", "explicit_files", multiple=True, help="Compose file paths")
@click.option("-p", "--project-name", help="Project name")
@click.pass_context
def compose_pull(
    ctx: click.Context,
    services: tuple[str, ...],
    env: str | None,
    explicit_files: tuple[str, ...],
    project_name: str | None,
) -> None:
    """Pull service images."""
    config: CLIConfig = ctx.obj.config
    cmd = ComposeCommand(
        config=config,
        environment=env,
        explicit_files=list(explicit_files) if explicit_files else None,
        project_name=project_name,
        service_filter=list(services) if services else None,
    )

    try:
        cmd.run(cmd._pull_impl(services=list(services) if services else None))
    except DockerSDKError as e:
        cmd.console.print(f"[red]Error pulling images:[/red] {e}", err=True)
        sys.exit(1)


@click.command()
@click.argument("services", nargs=-1, required=False)
@click.option("--env", help="Environment name (e.g., dev, prod)")
@click.option("-f", "--file", "explicit_files", multiple=True, help="Compose file paths")
@click.option("-p", "--project-name", help="Project name")
@click.pass_context
def compose_pause(
    ctx: click.Context,
    services: tuple[str, ...],
    env: str | None,
    explicit_files: tuple[str, ...],
    project_name: str | None,
) -> None:
    """Pause services."""
    config: CLIConfig = ctx.obj.config
    cmd = ComposeCommand(
        config=config,
        environment=env,
        explicit_files=list(explicit_files) if explicit_files else None,
        project_name=project_name,
        service_filter=list(services) if services else None,
    )

    try:
        cmd.run(cmd._pause_impl(services=list(services) if services else None))
    except DockerSDKError as e:
        cmd.console.print(f"[red]Error pausing services:[/red] {e}", err=True)
        sys.exit(1)


@click.command()
@click.argument("services", nargs=-1, required=False)
@click.option("--env", help="Environment name (e.g., dev, prod)")
@click.option("-f", "--file", "explicit_files", multiple=True, help="Compose file paths")
@click.option("-p", "--project-name", help="Project name")
@click.pass_context
def compose_unpause(
    ctx: click.Context,
    services: tuple[str, ...],
    env: str | None,
    explicit_files: tuple[str, ...],
    project_name: str | None,
) -> None:
    """Unpause services."""
    config: CLIConfig = ctx.obj.config
    cmd = ComposeCommand(
        config=config,
        environment=env,
        explicit_files=list(explicit_files) if explicit_files else None,
        project_name=project_name,
        service_filter=list(services) if services else None,
    )

    try:
        cmd.run(cmd._unpause_impl(services=list(services) if services else None))
    except DockerSDKError as e:
        cmd.console.print(f"[red]Error unpausing services:[/red] {e}", err=True)
        sys.exit(1)


@click.command()
@click.argument("services", nargs=-1, required=False)
@click.option("--signal", "-s", default="SIGKILL", help="Signal to send (default: SIGKILL)")
@click.option("--env", help="Environment name (e.g., dev, prod)")
@click.option("-f", "--file", "explicit_files", multiple=True, help="Compose file paths")
@click.option("-p", "--project-name", help="Project name")
@click.pass_context
def compose_kill(
    ctx: click.Context,
    services: tuple[str, ...],
    signal: str,
    env: str | None,
    explicit_files: tuple[str, ...],
    project_name: str | None,
) -> None:
    """Force stop services."""
    config: CLIConfig = ctx.obj.config
    cmd = ComposeCommand(
        config=config,
        environment=env,
        explicit_files=list(explicit_files) if explicit_files else None,
        project_name=project_name,
        service_filter=list(services) if services else None,
    )

    try:
        cmd.run(cmd._kill_impl(services=list(services) if services else None, signal=signal))
    except DockerSDKError as e:
        cmd.console.print(f"[red]Error killing services:[/red] {e}", err=True)
        sys.exit(1)


@click.command()
@click.argument("services", nargs=-1, required=False)
@click.option("--volumes", "-v", is_flag=True, help="Remove associated volumes")
@click.option("--env", help="Environment name (e.g., dev, prod)")
@click.option("-f", "--file", "explicit_files", multiple=True, help="Compose file paths")
@click.option("-p", "--project-name", help="Project name")
@click.pass_context
def compose_rm(
    ctx: click.Context,
    services: tuple[str, ...],
    volumes: bool,
    env: str | None,
    explicit_files: tuple[str, ...],
    project_name: str | None,
) -> None:
    """Remove stopped service containers."""
    config: CLIConfig = ctx.obj.config
    cmd = ComposeCommand(
        config=config,
        environment=env,
        explicit_files=list(explicit_files) if explicit_files else None,
        project_name=project_name,
        service_filter=list(services) if services else None,
    )

    try:
        cmd.run(cmd._rm_impl(services=list(services) if services else None, volumes=volumes))
    except DockerSDKError as e:
        cmd.console.print(f"[red]Error removing services:[/red] {e}", err=True)
        sys.exit(1)
