# Copyright 2025 BBDevs
# Licensed under the Apache License, Version 2.0

"""Compose command handler - business logic only.

Author: A M (am@bbdevs.com)

Created At: 08 Nov 2025
"""

from __future__ import annotations

import asyncio
import sys
from pathlib import Path
from typing import Any

from rich.table import Table

from dockpycli.commands.base import BaseCommand
from dockpycli.commands.compose.helpers import parse_compose_file, resolve_project_name
from dockpycli.commands.container.helpers import format_ports
from dockpycli.config import CLIConfig
from dockpycli.output.compose_formatters import format_compose_config, format_compose_ps_table
from dockpysdk import ComposeOrchestrator


__all__ = ["ComposeCommand"]


class ComposeCommand(BaseCommand):
    """Compose command handler - business logic only."""

    def __init__(
        self,
        config: CLIConfig,
        environment: str | None = None,
        explicit_files: list[str] | None = None,
        project_name: str | None = None,
        service_filter: list[str] | None = None,
    ) -> None:
        """Initialize compose command.

        Args:
            config: CLI configuration
            environment: Environment name (e.g., dev, prod)
            explicit_files: Explicit compose file paths
            project_name: Project name override
            service_filter: Service filter for operations
        """
        super().__init__(config)
        self.environment = environment
        self.explicit_files = explicit_files
        self.project_name = project_name
        self.service_filter = service_filter
        self._orchestrator: ComposeOrchestrator | None = None
        self._compose_file: Any | None = None
        self._project: str | None = None
        self._work_dir: Path | None = None

    async def __aenter__(self) -> ComposeCommand:
        """Enter context - initialize compose resources."""
        await super().__aenter__()

        # Get working directory
        self._work_dir = Path.cwd()

        # Parse compose file
        self._compose_file = parse_compose_file(
            environment=self.environment,
            explicit_files=self.explicit_files,
            service_filter=self.service_filter,
            directory=self._work_dir,
        )

        # Resolve project name
        self._project = resolve_project_name(
            directory=self._work_dir,
            project_name=self.project_name,
            compose_file=self._compose_file,
        )

        # Create orchestrator
        self._orchestrator = ComposeOrchestrator(
            self.client,
            self._compose_file,
            self._project,
        )

        return self

    @property
    def orchestrator(self) -> ComposeOrchestrator:
        """Get compose orchestrator."""
        if self._orchestrator is None:
            raise RuntimeError("Compose context not entered. Use 'async with' statement or 'run()' method.")
        return self._orchestrator

    @property
    def project(self) -> str:
        """Get project name."""
        if self._project is None:
            raise RuntimeError("Compose context not entered. Use 'async with' statement or 'run()' method.")
        return self._project

    @property
    def compose_file(self) -> Any:
        """Get compose file."""
        if self._compose_file is None:
            raise RuntimeError("Compose context not entered. Use 'async with' statement or 'run()' method.")
        return self._compose_file

    @property
    def work_dir(self) -> Path:
        """Get working directory."""
        if self._work_dir is None:
            raise RuntimeError("Compose context not entered. Use 'async with' statement or 'run()' method.")
        return self._work_dir

    async def _up_impl(
        self,
        services: list[str] | None,
        build: bool,
        detach: bool,
        force_recreate: bool,
    ) -> None:
        """Start services implementation."""
        self.console.print(f"[green]Starting services for project '{self.project}'...[/green]")

        result = await self.orchestrator.up(
            services=services,
            build=build,
            detach=detach,
            force_recreate=force_recreate,
        )

        if result["started"]:
            self.console.print(f"[green]Started {len(result['started'])} service(s)[/green]")
            for service in result["started"]:
                self.console.print(f"  - {service}")

        if result["failed"]:
            self.console.print(f"[red]Failed to start {len(result['failed'])} service(s)[/red]", err=True)
            for service in result["failed"]:
                self.console.print(f"  - {service}", err=True)
            sys.exit(1)

    async def _down_impl(
        self,
        services: list[str] | None,
        volumes: bool,
        remove_orphans: bool,
    ) -> None:
        """Stop and remove services implementation."""
        self.console.print(f"[yellow]Stopping services for project '{self.project}'...[/yellow]")

        result = await self.orchestrator.down(
            services=services,
            volumes=volumes,
            remove_orphans=remove_orphans,
        )

        if result["stopped"]:
            self.console.print(f"[green]Stopped {len(result['stopped'])} service(s)[/green]")
            for service in result["stopped"]:
                self.console.print(f"  - {service}")

        if result["failed"]:
            self.console.print(f"[red]Failed to stop {len(result['failed'])} service(s)[/red]", err=True)
            for service in result["failed"]:
                self.console.print(f"  - {service}", err=True)
            sys.exit(1)

    async def _ps_impl(self, services: list[str] | None) -> None:
        """List services implementation."""
        # List containers with project label
        filters = {"label": [f"com.docker.compose.project={self.project}"]}
        containers = await self.client.containers.list(all=True, filters=filters)

        # Format as compose ps
        service_list: list[dict[str, Any]] = []
        for container in containers:
            service_name = container.name.replace(f"{self.project}_", "")
            # Filter by service names if provided
            if services and service_name not in services:
                continue

            service_list.append(
                {
                    "name": container.name,
                    "service": service_name,
                    "status": container.status,
                    "ports": format_ports(container.ports),
                    "command": container.command,
                    "image": container.image,
                }
            )

        if self.config.output_format == "table":
            table = format_compose_ps_table(service_list)
            self.console.print(table)
        else:
            self.formatter.format(service_list)

    async def _ls_impl(self, all_projects: bool, format_template: str | None) -> None:  # noqa: ARG002
        """List compose projects implementation."""
        # List all containers with compose project label
        filters: dict[str, Any] = {}
        if not all_projects:
            filters["status"] = ["running"]

        containers = await self.client.containers.list(all=all_projects, filters=filters)

        # Group containers by project name
        projects: dict[str, dict[str, Any]] = {}
        for container in containers:
            project_name = container.labels.get("com.docker.compose.project", "")
            if not project_name:
                continue

            if project_name not in projects:
                projects[project_name] = {
                    "name": project_name,
                    "status": container.status,
                    "config": container.labels.get("com.docker.compose.project.config_files", ""),
                    "services": [],
                }

            # Extract service name from container name
            service_name = container.name.replace(f"{project_name}_", "")
            projects[project_name]["services"].append(service_name)

        # Format output
        if self.config.output_format == "table":
            table = Table(
                show_header=True,
                header_style="bold magenta",
                box=None,
                padding=(0, 1),
            )
            table.add_column("NAME", style="blue")
            table.add_column("STATUS", style="yellow")
            table.add_column("CONFIG", style="white")
            table.add_column("SERVICES", style="cyan")

            for project in sorted(projects.values(), key=lambda x: x["name"]):
                services_str = ", ".join(project["services"][:3])
                if len(project["services"]) > 3:
                    services_str += f" (+{len(project['services']) - 3} more)"

                table.add_row(
                    project["name"],
                    project["status"],
                    project["config"] or "-",
                    services_str,
                )

            self.console.print(table)
        else:
            # JSON/YAML/Plain format
            project_list = list(projects.values())
            self.formatter.format(project_list)

    async def _config_impl(self, resolve_image_digests: bool) -> None:
        """Show merged compose file implementation."""
        # Convert to dict and format
        merged_config = self.compose_file.to_dict()

        if resolve_image_digests:
            # Resolve image digests for each service
            services = merged_config.get("services", {})
            for _service_name, service_config in services.items():
                image = service_config.get("image")
                if image:
                    try:
                        # Get image details to find digest
                        image_details = await self.client.images.inspect_detailed(image)
                        # Use first digest if available
                        if image_details.repo_digests:
                            # Format: image@sha256:digest
                            digest = image_details.repo_digests[0]
                            service_config["image"] = digest
                    except Exception:
                        # If image not found, keep original image name
                        # Log warning but continue with other services
                        pass

        yaml_str = format_compose_config(merged_config)
        self.console.print(yaml_str)

    async def _logs_impl(
        self,
        services: list[str] | None,
        follow: bool,
        tail: int | None,
        since: str | None,
        until: str | None,
    ) -> None:
        """View service logs implementation."""
        # List containers with project label
        filters = {"label": [f"com.docker.compose.project={self.project}"]}
        containers = await self.client.containers.list(all=True, filters=filters)

        # Get logs for each container
        async def get_and_format_logs(container: Any, service_name: str) -> None:  # Container type
            """Get and format logs for a single container."""
            try:
                logs = self.client.containers.logs(
                    container.id,
                    follow=follow,
                    tail=tail,
                    timestamps=False,
                    since=since,
                    until=until,
                )

                # Format logs with service name prefix
                async for line in logs:
                    # Format: [service-name] log line
                    formatted_line = f"[bold blue]{service_name}[/bold blue] | {line.rstrip()}"
                    self.console.print(formatted_line)
            except Exception as e:
                self.console.print(f"[red]Error getting logs for {service_name}:[/red] {e}", err=True)

        # Collect tasks for all services
        tasks = []
        for container in containers:
            service_name = container.name.replace(f"{self.project}_", "")
            if services and service_name not in services:
                continue

            if follow:
                # For follow mode, run concurrently
                tasks.append(get_and_format_logs(container, service_name))
            else:
                # For non-follow mode, run sequentially
                await get_and_format_logs(container, service_name)

        # Run all follow tasks concurrently
        if tasks:
            await asyncio.gather(*tasks)

    async def _start_impl(self, services: list[str] | None) -> None:
        """Start services implementation."""
        result = await self.orchestrator.start(services=services)

        if self.config.output_format == "table":
            if result["started"]:
                self.console.print(f"[green]Started:[/green] {', '.join(result['started'])}")
            if result["failed"]:
                self.console.print(f"[red]Failed:[/red] {', '.join(result['failed'])}")
        else:
            self.formatter.format(result)

    async def _stop_impl(self, services: list[str] | None, timeout: int) -> None:
        """Stop services implementation."""
        result = await self.orchestrator.stop(services=services, timeout=timeout)

        if self.config.output_format == "table":
            if result["stopped"]:
                self.console.print(f"[green]Stopped:[/green] {', '.join(result['stopped'])}")
            if result["failed"]:
                self.console.print(f"[red]Failed:[/red] {', '.join(result['failed'])}")
        else:
            self.formatter.format(result)

    async def _restart_impl(self, services: list[str] | None, timeout: int) -> None:
        """Restart services implementation."""
        result = await self.orchestrator.restart(services=services, timeout=timeout)

        if self.config.output_format == "table":
            if result["restarted"]:
                self.console.print(f"[green]Restarted:[/green] {', '.join(result['restarted'])}")
            if result["failed"]:
                self.console.print(f"[red]Failed:[/red] {', '.join(result['failed'])}")
        else:
            self.formatter.format(result)

    async def _kill_impl(self, services: list[str] | None, signal: str) -> None:
        """Kill services implementation."""
        result = await self.orchestrator.kill(services=services, signal=signal)

        if self.config.output_format == "table":
            if result["killed"]:
                self.console.print(f"[green]Killed:[/green] {', '.join(result['killed'])}")
            if result["failed"]:
                self.console.print(f"[red]Failed:[/red] {', '.join(result['failed'])}")
        else:
            self.formatter.format(result)

    async def _pause_impl(self, services: list[str] | None) -> None:
        """Pause services implementation."""
        result = await self.orchestrator.pause(services=services)

        if self.config.output_format == "table":
            if result["paused"]:
                self.console.print(f"[green]Paused:[/green] {', '.join(result['paused'])}")
            if result["failed"]:
                self.console.print(f"[red]Failed:[/red] {', '.join(result['failed'])}")
        else:
            self.formatter.format(result)

    async def _unpause_impl(self, services: list[str] | None) -> None:
        """Unpause services implementation."""
        result = await self.orchestrator.unpause(services=services)

        if self.config.output_format == "table":
            if result["unpaused"]:
                self.console.print(f"[green]Unpaused:[/green] {', '.join(result['unpaused'])}")
            if result["failed"]:
                self.console.print(f"[red]Failed:[/red] {', '.join(result['failed'])}")
        else:
            self.formatter.format(result)

    async def _rm_impl(self, services: list[str] | None, volumes: bool) -> None:
        """Remove services implementation."""
        result = await self.orchestrator.rm(services=services, volumes=volumes)

        if self.config.output_format == "table":
            if result["removed"]:
                self.console.print(f"[green]Removed:[/green] {', '.join(result['removed'])}")
            if result["failed"]:
                self.console.print(f"[red]Failed:[/red] {', '.join(result['failed'])}")
        else:
            self.formatter.format(result)

    async def _pull_impl(self, services: list[str] | None) -> None:
        """Pull service images implementation."""
        result = await self.orchestrator.pull(services=services)

        if self.config.output_format == "table":
            if result["pulled"]:
                self.console.print(f"[green]Pulled:[/green] {', '.join(result['pulled'])}")
            if result["failed"]:
                self.console.print(f"[red]Failed:[/red] {', '.join(result['failed'])}")
        else:
            self.formatter.format(result)

    async def _build_impl(self, services: list[str] | None) -> None:
        """Build service images implementation."""
        result = await self.orchestrator.build(services=services)

        if self.config.output_format == "table":
            if result["built"]:
                self.console.print(f"[green]Built:[/green] {', '.join(result['built'])}")
            if result["failed"]:
                self.console.print(f"[red]Failed:[/red] {', '.join(result['failed'])}")
        else:
            self.formatter.format(result)
