# Copyright 2025 BBDevs
# Licensed under the Apache License, Version 2.0

"""Helper functions for compose commands.

Author: A M (am@bbdevs.com)
Created At: 08 Nov 2025
"""

from __future__ import annotations

import os
from pathlib import Path
from typing import Any

from dockpycore.logging import get_logger
from dockpysdk import ComposeParser
from dockpysdk.compose_parser import discover_compose_files as sdk_discover_compose_files


__all__ = [
    "discover_compose_files",
    "find_compose_file",
    "format_service_status",
    "get_environment_files",
    "get_image_files",
    "parse_compose_file",
    "resolve_compose_files",
    "resolve_project_name",
]

logger = get_logger(__name__)


def find_compose_file(directory: str | Path | None = None) -> Path | None:
    """Find base compose file in directory.

    Looks for: compose.yml, docker-compose.yml, compose.yaml, docker-compose.yaml

    Args:
        directory: Directory to search (default: current directory)

    Returns:
        Path to compose file or None if not found
    """
    if directory is None:
        directory = Path.cwd()
    else:
        directory = Path(directory).resolve()

    base_patterns = ["compose.yml", "docker-compose.yml", "compose.yaml", "docker-compose.yaml"]

    for pattern in base_patterns:
        file_path = directory / pattern
        if file_path.exists():
            logger.debug("compose_file_found", file=str(file_path))
            return file_path

    return None


def resolve_project_name(
    directory: str | Path | None = None,
    project_name: str | None = None,
    compose_file: Any = None,  # ComposeFile type
) -> str:
    """Resolve compose project name.

    Priority:
    1. Explicit project_name parameter
    2. COMPOSE_PROJECT_NAME environment variable
    3. name field from compose file
    4. Directory name

    Args:
        directory: Directory containing compose files
        project_name: Explicit project name
        compose_file: Parsed compose file

    Returns:
        Project name
    """
    # 1. Explicit project name
    if project_name:
        return project_name

    # 2. Environment variable
    env_name = os.environ.get("COMPOSE_PROJECT_NAME")
    if env_name:
        return env_name

    # 3. From compose file
    if compose_file and compose_file.name:
        return compose_file.name

    # 4. Directory name
    if directory is None:
        directory = Path.cwd()
    else:
        directory = Path(directory).resolve()

    return directory.name.lower().replace(" ", "-").replace("_", "-")


def get_environment_files(
    directory: str | Path,
    base_name: str | None = None,
    environment: str | None = None,
) -> list[Path]:
    """Get environment-specific compose files.

    Args:
        directory: Directory to search
        base_name: Base name for files (e.g., "jenkins")
        environment: Environment name (e.g., "dev", "prod")

    Returns:
        List of environment file paths
    """
    directory = Path(directory).resolve()
    files: list[Path] = []

    if not environment:
        return files

    if base_name:
        patterns = [
            f"{base_name}.{environment}.yml",
            f"{base_name}.{environment}.yaml",
        ]
    else:
        patterns = [
            f"*.{environment}.yml",
            f"*.{environment}.yaml",
        ]

    for pattern in patterns:
        if "*" in pattern:
            matches = list(directory.glob(pattern))
            files.extend(matches)
        else:
            file_path = directory / pattern
            if file_path.exists():
                files.append(file_path)

    return files


def get_image_files(
    directory: str | Path,
    base_name: str | None = None,
) -> list[Path]:
    """Get image-related compose files.

    Args:
        directory: Directory to search
        base_name: Base name for files (e.g., "jenkins")

    Returns:
        List of image file paths
    """
    directory = Path(directory).resolve()
    files: list[Path] = []

    if base_name:
        patterns = [
            f"{base_name}.image.yml",
            f"{base_name}.image.yaml",
        ]
    else:
        patterns = [
            "*.image.yml",
            "*.image.yaml",
        ]

    for pattern in patterns:
        if "*" in pattern:
            matches = list(directory.glob(pattern))
            files.extend(matches)
        else:
            file_path = directory / pattern
            if file_path.exists():
                files.append(file_path)

    return files


def format_service_status(status: str) -> str:
    """Format service status for display.

    Args:
        status: Service status string

    Returns:
        Formatted status string
    """
    status_map = {
        "running": "Running",
        "exited": "Exited",
        "created": "Created",
        "restarting": "Restarting",
        "removing": "Removing",
        "paused": "Paused",
        "dead": "Dead",
    }

    return status_map.get(status.lower(), status)


# Re-export SDK functions for convenience
def discover_compose_files(
    directory: str | Path,
    base_name: str | None = None,
    environment: str | None = None,
    explicit_files: list[str] | None = None,
) -> list[Path]:
    """Discover compose files (re-export from SDK).

    Args:
        directory: Directory to search
        base_name: Base name for files
        environment: Environment name
        explicit_files: Explicit file paths

    Returns:
        List of compose file paths
    """
    return sdk_discover_compose_files(directory, base_name, environment, explicit_files)


def parse_compose_file(
    base_name: str | None = None,
    environment: str | None = None,
    explicit_files: list[str] | None = None,
    service_filter: list[str] | None = None,
    directory: str | Path | None = None,
) -> Any:  # ComposeFile type
    """Parse compose file using ComposeParser.

    Args:
        base_name: Base name for files
        environment: Environment name
        explicit_files: Explicit file paths
        service_filter: List of service names to include
        directory: Working directory

    Returns:
        Parsed ComposeFile instance
    """
    if directory is None:
        directory = Path.cwd()
    else:
        directory = Path(directory).resolve()

    parser = ComposeParser(directory)
    return parser.parse(
        base_name=base_name,
        environment=environment,
        explicit_files=explicit_files,
        service_filter=service_filter,
    )


def resolve_compose_files(
    directory: str | Path | None = None,
    base_name: str | None = None,
    environment: str | None = None,
    explicit_files: list[str] | None = None,
) -> list[Path]:
    """Resolve compose files to use.

    Args:
        directory: Directory to search
        base_name: Base name for files
        environment: Environment name
        explicit_files: Explicit file paths

    Returns:
        List of compose file paths in merge order
    """
    if directory is None:
        directory = Path.cwd()
    else:
        directory = Path(directory).resolve()

    return sdk_discover_compose_files(
        directory, base_name=base_name, environment=environment, explicit_files=explicit_files
    )
