# Copyright 2025 BBDevs
# Licensed under the Apache License, Version 2.0

"""Container command Click entry points.

Author: A M (am@bbdevs.com)

Created At: 08 Nov 2025
"""

from __future__ import annotations

from typing import Any

import click

from dockpycli.commands.container.command import ContainerCommand
from dockpycli.commands.container.helpers import (
    parse_environment,
    parse_port_mapping,
    parse_volume_mapping,
)
from dockpycli.config import CLIConfig


__all__ = [
    "container_logs",
    "container_port",
    "container_prune",
    "container_stats",
    "container_top",
    "create_container",
    "exec_container",
    "inspect_container",
    "list_containers",
    "pause_container",
    "remove_container",
    "rename_container",
    "restart_container",
    "run_container",
    "start_container",
    "stop_container",
    "unpause_container",
    "wait_container",
]


@click.command()
@click.option("--all", "-a", is_flag=True, help="Show all containers (default: only running)")
@click.option("--limit", type=int, help="Limit number of containers")
@click.option("--filter", "-f", multiple=True, help="Filter containers (e.g., status=running)")
@click.option("--no-trunc", is_flag=True, help="Don't truncate output")
@click.pass_context
def list_containers(
    ctx: click.Context,
    all: bool,
    limit: int | None,
    filter: tuple[str, ...],
    no_trunc: bool,
) -> None:
    """List containers."""
    config: CLIConfig = ctx.obj.config
    cmd = ContainerCommand(config)

    # Parse filters
    filters: dict[str, Any] | None = None
    if filter:
        filters = {}
        for f in filter:
            if "=" in f:
                key, value = f.split("=", 1)
                if key not in filters:
                    filters[key] = []
                filters[key].append(value)

    cmd.run(cmd._list_containers_impl(all, limit, filters, no_trunc))


@click.command()
@click.argument("image", required=True)
@click.option("--name", "-n", help="Container name")
@click.option("--command", "-c", multiple=True, help="Command to run")
@click.option("--entrypoint", multiple=True, help="Entrypoint override")
@click.option("--env", "-e", multiple=True, help="Environment variable (KEY=value or KEY)")
@click.option("--workdir", "-w", help="Working directory")
@click.option("--user", "-u", help="Username or UID")
@click.option("--hostname", "-h", help="Container hostname")
@click.option("--label", "-l", multiple=True, help="Label (key=value)")
@click.option("--volume", "-v", multiple=True, help="Volume mapping (/host:/container)")
@click.option("--port", "-p", multiple=True, help="Port mapping (8080:80)")
@click.option("--network", help="Network to connect to")
@click.option("--detach", "-d", is_flag=True, help="Run container in background")
@click.option("--init", is_flag=True, help="Run an init inside the container")
@click.pass_context
def create_container(
    ctx: click.Context,
    image: str,
    name: str | None,
    command: tuple[str, ...],
    entrypoint: tuple[str, ...],
    env: tuple[str, ...],
    workdir: str | None,
    user: str | None,
    hostname: str | None,
    label: tuple[str, ...],
    volume: tuple[str, ...],
    port: tuple[str, ...],
    network: str | None,
    detach: bool,
    init: bool,
) -> None:
    """Create a new container."""
    config: CLIConfig = ctx.obj.config
    cmd = ContainerCommand(config)

    cmd_list = list(command) if command else None
    entrypoint_list = list(entrypoint) if entrypoint else None

    cmd.run(
        cmd._create_container_impl(
            image,
            name,
            cmd_list,
            entrypoint_list,
            env,
            workdir,
            user,
            hostname,
            label,
            volume,
            port,
            network,
            detach,
            init,
        )
    )


@click.command()
@click.argument("image", required=True)
@click.argument("command", required=False, nargs=-1)
@click.option("--name", "-n", help="Container name")
@click.option("--rm", is_flag=True, help="Automatically remove container when it exits")
@click.option("--detach", "-d", is_flag=True, help="Run container in background")
@click.option("--interactive", "-i", is_flag=True, help="Keep STDIN open")
@click.option("--tty", "-t", is_flag=True, help="Allocate a pseudo-TTY")
@click.option("--env", "-e", multiple=True, help="Environment variable")
@click.option("--volume", "-v", multiple=True, help="Volume mapping")
@click.option("--port", "-p", multiple=True, help="Port mapping")
@click.option("--init", is_flag=True, help="Run an init inside the container")
@click.pass_context
def run_container(
    ctx: click.Context,
    image: str,
    command: tuple[str, ...],
    name: str | None,
    rm: bool,
    detach: bool,
    interactive: bool,
    tty: bool,
    env: tuple[str, ...],
    volume: tuple[str, ...],
    port: tuple[str, ...],
    init: bool,
) -> None:
    """Run a container (create + start)."""
    config: CLIConfig = ctx.obj.config
    cmd = ContainerCommand(config)

    cmd_list = list(command) if command else None
    environment = parse_environment(list(env)) if env else None

    volumes: dict[str, dict[str, str]] | None = None
    if volume:
        volumes = {}
        for v in volume:
            volumes.update(parse_volume_mapping(v))

    ports: dict[str, list[dict[str, str]]] | None = None
    if port:
        ports = {}
        for p in port:
            ports.update(parse_port_mapping(p))

    cmd.run(
        cmd._run_container_impl(
            image,
            cmd_list,
            name,
            rm,
            detach,
            interactive,
            tty,
            command=cmd_list,
            environment=environment,
            volumes=volumes,
            ports=ports,
            init=init,
        )
    )


@click.command()
@click.argument("container", required=True)
@click.pass_context
def start_container(ctx: click.Context, container: str) -> None:
    """Start a container."""
    config: CLIConfig = ctx.obj.config
    cmd = ContainerCommand(config)
    cmd.run(cmd._start_container_impl(container))


@click.command()
@click.argument("container", required=True)
@click.option("--timeout", "-t", type=int, default=10, help="Seconds to wait before killing")
@click.option("--signal", "-s", help="Signal to send to container (e.g., SIGTERM, SIGKILL)")
@click.pass_context
def stop_container(ctx: click.Context, container: str, timeout: int, signal: str | None) -> None:
    """Stop a container."""
    config: CLIConfig = ctx.obj.config
    cmd = ContainerCommand(config)
    cmd.run(cmd._stop_container_impl(container, timeout, signal))


@click.command()
@click.argument("container", required=True)
@click.option("--timeout", "-t", type=int, default=10, help="Seconds to wait before killing")
@click.pass_context
def restart_container(ctx: click.Context, container: str, timeout: int) -> None:
    """Restart a container."""
    config: CLIConfig = ctx.obj.config
    cmd = ContainerCommand(config)
    cmd.run(cmd._restart_container_impl(container, timeout))


@click.command()
@click.argument("container", required=True)
@click.pass_context
def pause_container(ctx: click.Context, container: str) -> None:
    """Pause a container."""
    config: CLIConfig = ctx.obj.config
    cmd = ContainerCommand(config)
    cmd.run(cmd._pause_container_impl(container))


@click.command()
@click.argument("container", required=True)
@click.pass_context
def unpause_container(ctx: click.Context, container: str) -> None:
    """Unpause a container."""
    config: CLIConfig = ctx.obj.config
    cmd = ContainerCommand(config)
    cmd.run(cmd._unpause_container_impl(container))


@click.command()
@click.argument("containers", required=True, nargs=-1)
@click.option("--force", "-f", is_flag=True, help="Force removal")
@click.option("--volumes", "-v", is_flag=True, help="Remove associated volumes")
@click.pass_context
def remove_container(ctx: click.Context, containers: tuple[str, ...], force: bool, volumes: bool) -> None:
    """Remove one or more containers."""
    config: CLIConfig = ctx.obj.config
    cmd = ContainerCommand(config)
    cmd.run(cmd._remove_container_impl(list(containers), force, volumes))


@click.command()
@click.argument("container", required=True)
@click.option("--size", "-s", is_flag=True, help="Display total file sizes")
@click.option(
    "--format",
    "format_template",
    help="Format output using a template (JSON/YAML/Plain)",
)
@click.pass_context
def inspect_container(ctx: click.Context, container: str, size: bool, format_template: str | None) -> None:
    """Inspect a container."""
    config: CLIConfig = ctx.obj.config
    cmd = ContainerCommand(config)
    cmd.run(cmd._inspect_container_impl(container, size, format_template))


@click.command()
@click.argument("container", required=True)
@click.option("--follow", "-f", is_flag=True, help="Follow log output")
@click.option("--tail", type=int, help="Number of lines to show from end")
@click.option("--timestamps", "-t", is_flag=True, help="Show timestamps")
@click.option("--since", help="Show logs since timestamp (e.g., 2013-01-02T13:23:37, 1376077697, or 2h)")
@click.option("--until", help="Show logs before timestamp (e.g., 2013-01-02T13:23:37, 1376077697, or 2h)")
@click.option("--details", is_flag=True, help="Show extra details provided to logs")
@click.pass_context
def container_logs(
    ctx: click.Context,
    container: str,
    follow: bool,
    tail: int | None,
    timestamps: bool,
    since: str | None,
    until: str | None,
    details: bool,
) -> None:
    """Show container logs."""
    config: CLIConfig = ctx.obj.config
    cmd = ContainerCommand(config)
    cmd.run(cmd._container_logs_impl(container, follow, tail, timestamps, since, until, details))


@click.command()
@click.argument("containers", required=False, nargs=-1)
@click.option("--no-stream", is_flag=True, help="Disable streaming stats")
@click.option(
    "--format",
    "format_template",
    help="Format output using a template (JSON/YAML/Plain)",
)
@click.pass_context
def container_stats(
    ctx: click.Context,
    containers: tuple[str, ...],
    no_stream: bool,
    format_template: str | None,
) -> None:
    """Show container resource usage statistics."""
    config: CLIConfig = ctx.obj.config
    container_list = list(containers) if containers else None
    cmd = ContainerCommand(config)
    cmd.run(cmd._container_stats_impl(container_list, no_stream, format_template))


@click.command()
@click.argument("container", required=True)
@click.argument("command", required=True, nargs=-1)
@click.option("--workdir", "-w", help="Working directory")
@click.option("--env", "-e", multiple=True, help="Environment variable")
@click.option("--user", "-u", help="User to run as")
@click.option("--privileged", is_flag=True, help="Give extended privileges to the command")
@click.pass_context
def exec_container(
    ctx: click.Context,
    container: str,
    command: tuple[str, ...],
    workdir: str | None,
    env: tuple[str, ...],
    user: str | None,
    privileged: bool,
) -> None:
    """Execute a command in a running container."""
    config: CLIConfig = ctx.obj.config
    cmd = ContainerCommand(config)
    cmd.run(cmd._exec_container_impl(container, list(command), workdir, env, user, privileged))


@click.command()
@click.argument("container", required=True)
@click.option(
    "--condition",
    type=click.Choice(["not-running", "next-exit", "removed"], case_sensitive=False),
    help="Wait condition: not-running (default), next-exit, or removed",
)
@click.pass_context
def wait_container(ctx: click.Context, container: str, condition: str | None) -> None:
    """Wait for a container to stop and print exit code."""
    config: CLIConfig = ctx.obj.config
    cmd = ContainerCommand(config)
    cmd.run(cmd._wait_container_impl(container, condition))


@click.command()
@click.argument("container", required=True)
@click.argument("new_name", required=True)
@click.pass_context
def rename_container(ctx: click.Context, container: str, new_name: str) -> None:
    """Rename a container."""
    config: CLIConfig = ctx.obj.config
    cmd = ContainerCommand(config)
    cmd.run(cmd._rename_container_impl(container, new_name))


@click.command()
@click.argument("container", required=True)
@click.option(
    "--format",
    "format_template",
    help="Format output using a template (JSON/YAML/Plain)",
)
@click.pass_context
def container_port(ctx: click.Context, container: str, format_template: str | None) -> None:
    """Show port mappings for a container."""
    config: CLIConfig = ctx.obj.config
    cmd = ContainerCommand(config)
    cmd.run(cmd._port_impl(container, format_template))


@click.command()
@click.option("--filter", "-f", multiple=True, help="Filter containers (e.g., until=24h)")
@click.pass_context
def container_prune(ctx: click.Context, filter: tuple[str, ...]) -> None:
    """Remove all stopped containers."""
    config: CLIConfig = ctx.obj.config
    cmd = ContainerCommand(config)

    # Parse filters
    filters: dict[str, Any] | None = None
    if filter:
        filters = {}
        for f in filter:
            if "=" in f:
                key, value = f.split("=", 1)
                if key not in filters:
                    filters[key] = []
                filters[key].append(value)

    cmd.run(cmd._prune_impl(filters))


@click.command()
@click.argument("container", required=True)
@click.option("--ps-args", default="-ef", help="Arguments for ps command")
@click.option(
    "--format",
    "format_template",
    help="Format output using a template (JSON/YAML/Plain)",
)
@click.pass_context
def container_top(ctx: click.Context, container: str, ps_args: str, format_template: str | None) -> None:
    """Display the running processes of a container."""
    config: CLIConfig = ctx.obj.config
    cmd = ContainerCommand(config)
    cmd.run(cmd._top_impl(container, ps_args, format_template))
