# Copyright 2025 BBDevs
# Licensed under the Apache License, Version 2.0

"""Container command handler - business logic only.

Author: A M (am@bbdevs.com)

Created At: 08 Nov 2025
"""

from __future__ import annotations

import asyncio
import sys
from typing import Any

from dockpycli.commands.base import BaseCommand
from dockpycli.commands.container.helpers import (
    parse_environment,
    parse_labels,
    parse_port_mapping,
    parse_volume_mapping,
    resolve_container_id,
)
from dockpycli.output.container_formatters import (
    format_container_plain,
    format_container_stats_table,
    format_container_table,
)
from dockpycli.output.formatters import JSONFormatter
from dockpycore.exceptions import NotFound


__all__ = ["ContainerCommand"]


class ContainerCommand(BaseCommand):
    """Container command handler - business logic only."""

    async def _list_containers_impl(
        self,
        all_containers: bool,
        limit: int | None,
        filters: dict[str, Any] | None,
        no_trunc: bool = False,
    ) -> None:
        """List containers implementation."""
        containers = await self.client.containers.list(
            all=all_containers,
            limit=limit,
            filters=filters,
        )

        if self.config.output_format == "table":
            table = format_container_table(containers, no_trunc=no_trunc)
            self.console.print(table)
        elif self.config.output_format == "plain":
            plain_output = format_container_plain(containers)
            self.console.print(plain_output)
        else:
            data = [container.to_dict() for container in containers]
            self.formatter.format(data)

    async def _create_container_impl(
        self,
        image: str,
        name: str | None,
        command: list[str] | None,
        entrypoint: list[str] | None,
        env: tuple[str, ...],
        workdir: str | None,
        user: str | None,
        hostname: str | None,
        label: tuple[str, ...],
        volume: tuple[str, ...],
        port: tuple[str, ...],
        network: str | None,  # noqa: ARG002
        detach: bool,  # noqa: ARG002
        init: bool = False,
    ) -> None:
        """Create container implementation."""
        # Parse options
        environment = parse_environment(list(env)) if env else None
        labels = parse_labels(list(label)) if label else None

        # Parse volumes
        volumes: dict[str, dict[str, str]] | None = None
        if volume:
            volumes = {}
            for v in volume:
                volumes.update(parse_volume_mapping(v))

        # Parse ports
        ports: dict[str, list[dict[str, str]]] | None = None
        if port:
            ports = {}
            for p in port:
                ports.update(parse_port_mapping(p))

        container = await self.client.containers.create(
            image,
            name=name,
            command=command,
            entrypoint=entrypoint,
            environment=environment,
            working_dir=workdir,
            user=user,
            hostname=hostname,
            labels=labels,
            volumes=volumes,
            ports=ports,
            init=init,
        )

        if self.config.output_format == "table":
            self.console.print(f"[green]Container created:[/green] {container.id[:12]}")
            if container.name:
                self.console.print(f"[green]Name:[/green] {container.name}")
        else:
            self.formatter.format(container.to_dict())

    async def _run_container_impl(
        self,
        image: str,
        command: list[str] | None,  # noqa: ARG002
        name: str | None,
        rm: bool,
        detach: bool,
        interactive: bool,  # noqa: ARG002
        tty: bool,  # noqa: ARG002
        init: bool = False,
        **create_kwargs: Any,
    ) -> None:
        """Run container (create + start) implementation."""
        # Create container
        container = await self.client.containers.create(image, name=name, init=init, **create_kwargs)

        # Start container
        await self.client.containers.start(container.id)

        if detach:
            self.console.print(f"[green]Container started:[/green] {container.id[:12]}")
            if container.name:
                self.console.print(f"[green]Name:[/green] {container.name}")
        else:
            # Attach to container (simplified - full TTY support in future)
            self.console.print(f"[green]Container started:[/green] {container.id[:12]}")

        # If --rm, schedule removal on exit
        if rm:
            # Note: In a real implementation, we'd track this and remove on exit
            pass

    async def _start_container_impl(self, container_ref: str) -> None:
        """Start container implementation."""
        container_id = await resolve_container_id(self.client, container_ref)
        await self.client.containers.start(container_id)
        self.console.print(f"[green]Container started:[/green] {container_id[:12]}")

    async def _stop_container_impl(self, container_ref: str, timeout: int, signal: str | None = None) -> None:
        """Stop container implementation."""
        container_id = await resolve_container_id(self.client, container_ref)
        await self.client.containers.stop(container_id, timeout=timeout, signal=signal)
        self.console.print(f"[green]Container stopped:[/green] {container_id[:12]}")

    async def _restart_container_impl(self, container_ref: str, timeout: int) -> None:
        """Restart container implementation."""
        container_id = await resolve_container_id(self.client, container_ref)
        await self.client.containers.restart(container_id, timeout=timeout)
        self.console.print(f"[green]Container restarted:[/green] {container_id[:12]}")

    async def _pause_container_impl(self, container_ref: str) -> None:
        """Pause container implementation."""
        container_id = await resolve_container_id(self.client, container_ref)
        await self.client.containers.pause(container_id)
        self.console.print(f"[green]Container paused:[/green] {container_id[:12]}")

    async def _unpause_container_impl(self, container_ref: str) -> None:
        """Unpause container implementation."""
        container_id = await resolve_container_id(self.client, container_ref)
        await self.client.containers.unpause(container_id)
        self.console.print(f"[green]Container unpaused:[/green] {container_id[:12]}")

    async def _remove_container_impl(self, container_refs: list[str], force: bool, volumes: bool) -> None:
        """Remove container(s) implementation."""
        for container_ref in container_refs:
            try:
                container_id = await resolve_container_id(self.client, container_ref)
                await self.client.containers.remove(container_id, force=force, volumes=volumes)
                self.console.print(f"[green]Container removed:[/green] {container_id[:12]}")
            except NotFound:
                self.console.print(f"[yellow]Container not found:[/yellow] {container_ref}")
                if not force:
                    sys.exit(1)

    async def _inspect_container_impl(self, container_ref: str, size: bool, format_template: str | None) -> None:
        """Inspect container implementation."""
        container_id = await resolve_container_id(self.client, container_ref)
        container = await self.client.containers.inspect_detailed(container_id)

        # Convert to dict for formatting
        inspect_dict = {
            "Id": container.id,
            "Name": container.name,
            "Image": container.image,
            "Created": container.created,
            "State": {
                "Status": container.state.status,
                "Running": container.state.running,
                "Paused": container.state.paused,
                "ExitCode": container.state.exit_code,
            },
            "Config": {
                "Hostname": container.config.hostname,
                "User": container.config.user,
                "WorkingDir": container.config.working_dir,
                "Env": container.config.env,
                "Cmd": container.config.cmd,
            },
        }

        # Add size if requested
        if size:
            # Get size information from ContainerInspect model
            inspect_dict["SizeRw"] = container.size_rw
            inspect_dict["SizeRootFs"] = container.size_root_fs

        # If format template is provided, use it (basic implementation)
        # For MVP, just use JSON formatter if template is provided
        # Full Go template support can be added later
        formatter = JSONFormatter(self.config) if format_template else self.formatter

        formatter.format(inspect_dict)

    async def _container_logs_impl(
        self,
        container_ref: str,
        follow: bool,
        tail: int | str | None,
        timestamps: bool,
        since: str | None = None,
        until: str | None = None,
        details: bool = False,
    ) -> None:
        """Get container logs implementation."""
        container_id = await resolve_container_id(self.client, container_ref)

        async for line in self.client.containers.logs(
            container_id, follow=follow, tail=tail, timestamps=timestamps, since=since, until=until, details=details
        ):
            print(line, end="")

    async def _container_stats_impl(  # noqa: PLR0912
        self,
        container_refs: list[str] | None,
        no_stream: bool,
        format_template: str | None,
    ) -> None:
        """Get container stats implementation."""
        # If no containers specified, get all running containers
        if not container_refs:
            containers = await self.client.containers.list(all=False)
            container_refs = [c.id for c in containers]

        if not container_refs:
            self.console.print("[yellow]No running containers[/yellow]")
            return

        # Resolve container IDs
        container_ids: list[str] = []
        for ref in container_refs:
            try:
                container_id = await resolve_container_id(self.client, ref)
                container_ids.append(container_id)
            except NotFound:
                self.console.print(f"[yellow]Container not found:[/yellow] {ref}")

        if not container_ids:
            return

        # If format is specified, use formatter instead of table
        use_formatter = format_template is not None or self.config.output_format != "table"

        if no_stream:
            # One-shot stats
            stats_data = []
            for container_id in container_ids:
                stats = await self.client.containers.stats(container_id, stream=False)
                container = await self.client.containers.inspect(container_id)

                if use_formatter:
                    # Convert to dict for formatter
                    stats_dict = {
                        "Id": container_id,
                        "Name": container.name,
                        "CPU": stats.cpu_percent,
                        "Memory": stats.memory_usage,
                        "MemoryLimit": stats.memory_limit,
                        "NetworkRx": stats.network_rx_bytes,
                        "NetworkTx": stats.network_tx_bytes,
                        "BlockRead": stats.blkio_read_bytes,
                        "BlockWrite": stats.blkio_write_bytes,
                        "PIDs": stats.pids,
                    }
                    stats_data.append(stats_dict)
                else:
                    table = format_container_stats_table(container_id, container.name, stats)
                    self.console.print(table)

            if use_formatter and stats_data:
                self.formatter.format(stats_data if len(stats_data) > 1 else stats_data[0])
        else:
            # Stream stats (always use table for streaming)
            self.console.print("[yellow]Press Ctrl+C to stop[/yellow]")
            try:
                # Get stats iterators for all containers
                container_names: dict[str, str] = {}

                # Get initial container names
                for container_id in container_ids:
                    container = await self.client.containers.inspect(container_id)
                    container_names[container_id] = container.name

                # Start streaming stats
                stats_iters = []
                for container_id in container_ids:
                    # stats() returns AsyncIterator when stream=True
                    stats_result = self.client.containers.stats(container_id, stream=True)
                    stats_iters.append((container_id, stats_result))

                # Stream all stats
                while True:
                    stats_list = []
                    for container_id, stats_iter in stats_iters:
                        try:
                            stats = await stats_iter.__anext__()
                            container_name = container_names.get(container_id, "")
                            stats_list.append((container_id, container_name, stats))
                        except StopAsyncIteration:
                            pass

                    if stats_list:
                        # Clear screen and print stats
                        self.console.clear()
                        for container_id, container_name, stats in stats_list:
                            table = format_container_stats_table(container_id, container_name, stats)
                            self.console.print(table)
                    await asyncio.sleep(1)

            except KeyboardInterrupt:
                self.console.print("\n[yellow]Stats stopped[/yellow]")

    async def _exec_container_impl(
        self,
        container_ref: str,
        cmd: list[str],
        workdir: str | None,
        env: tuple[str, ...],
        user: str | None,
        privileged: bool = False,
    ) -> None:
        """Execute command in container implementation."""
        container_id = await resolve_container_id(self.client, container_ref)

        environment: dict[str, str] | None = None
        if env:
            env_list = parse_environment(list(env))
            environment = {}
            for e in env_list:
                if "=" in e:
                    key, value = e.split("=", 1)
                    environment[key] = value

        exit_code, output = await self.client.containers.exec(
            container_id,
            cmd,
            workdir=workdir,
            environment=environment,
            user=user,
            privileged=privileged,
        )

        print(output, end="")
        if exit_code != 0:
            sys.exit(exit_code)

    async def _wait_container_impl(self, container_ref: str, condition: str | None = None) -> None:
        """Wait for container to stop implementation."""
        container_id = await resolve_container_id(self.client, container_ref)
        exit_code = await self.client.containers.wait(container_id, condition=condition)
        self.console.print(f"[green]Container exited with code:[/green] {exit_code}")
        sys.exit(exit_code)

    async def _rename_container_impl(self, container_ref: str, new_name: str) -> None:
        """Rename container implementation."""
        from dockpycli.utils.validators import validate_container_name  # noqa: PLC0415

        # Validate new name
        validate_container_name(new_name)

        container_id = await resolve_container_id(self.client, container_ref)
        await self.client.containers.rename(container_id, new_name)
        self.console.print(f"[green]Container renamed:[/green] {container_id[:12]} -> {new_name}")

    async def _port_impl(self, container_ref: str, format_template: str | None) -> None:
        """Show port mappings for a container."""
        from dockpycli.commands.container.helpers import format_ports  # noqa: PLC0415
        from dockpycli.output.container_formatters import format_port_table  # noqa: PLC0415

        container_id = await resolve_container_id(self.client, container_ref)
        container = await self.client.containers.inspect(container_id)

        # Format ports
        if self.config.output_format == "table" and not format_template:
            table = format_port_table(container.ports)
            self.console.print(table)
        else:
            # JSON/YAML/Plain format
            port_data = {
                "container": container_id[:12],
                "name": container.name,
                "ports": format_ports(container.ports),
            }
            self.formatter.format(port_data)

    async def _prune_impl(self, filters: dict[str, Any] | None) -> None:
        """Prune stopped containers implementation."""
        from dockpycli.utils.helpers import format_size  # noqa: PLC0415

        result = await self.client.containers.prune(filters=filters)

        deleted = result.get("ContainersDeleted", [])
        space_reclaimed = result.get("SpaceReclaimed", 0)

        if self.config.output_format == "table":
            self.console.print(f"[green]Deleted:[/green] {len(deleted)} containers")
            self.console.print(f"[green]Space reclaimed:[/green] {format_size(space_reclaimed)}")
        else:
            # JSON/YAML/Plain format
            prune_data = {
                "containers_deleted": deleted,
                "space_reclaimed": space_reclaimed,
            }
            self.formatter.format(prune_data)

    async def _top_impl(self, container_ref: str, ps_args: str, format_template: str | None) -> None:
        """Show running processes in container implementation."""
        from dockpycli.output.container_formatters import format_top_table  # noqa: PLC0415

        container_id = await resolve_container_id(self.client, container_ref)
        result = await self.client.containers.top(container_id, ps_args=ps_args)

        titles = result.get("Titles", [])
        processes = result.get("Processes", [])

        if self.config.output_format == "table" and not format_template:
            table = format_top_table(titles, processes)
            self.console.print(table)
        else:
            # JSON/YAML/Plain format
            top_data = {
                "container": container_id[:12],
                "titles": titles,
                "processes": processes,
            }
            self.formatter.format(top_data)
