# Copyright 2025 BBDevs
# Licensed under the Apache License, Version 2.0

"""Helper functions for container operations.

Author: A M (am@bbdevs.com)

Created At: 08 Nov 2025
"""

from __future__ import annotations

import os
from typing import Any

from dockpycli.utils.validators import ValidationError, validate_port_mapping


__all__ = [
    "format_command",
    "format_ports",
    "parse_environment",
    "parse_labels",
    "parse_port_mapping",
    "parse_volume_mapping",
    "resolve_container_id",
]


def parse_port_mapping(port_str: str) -> dict[str, list[dict[str, str]]]:
    """Parse port mapping string to SDK format.

    Supports formats:
    - 8080:80
    - 8080:80/tcp
    - 127.0.0.1:8080:80
    - 127.0.0.1:8080:80/tcp

    Args:
        port_str: Port mapping string

    Returns:
        Port mapping in SDK format: {"80/tcp": [{"HostPort": "8080", "HostIp": "127.0.0.1"}]}

    Raises:
        ValidationError: If port mapping is invalid

    Example:
        >>> parse_port_mapping("8080:80")
        {"80/tcp": [{"HostPort": "8080"}]}
        >>> parse_port_mapping("127.0.0.1:8080:80/tcp")
        {"80/tcp": [{"HostPort": "8080", "HostIp": "127.0.0.1"}]}
    """
    # Extract protocol (default: tcp)
    protocol = "tcp"
    port_str_no_protocol = port_str
    if "/" in port_str:
        parts = port_str.split("/")
        if len(parts) == 2:
            protocol = parts[1]
            port_str_no_protocol = parts[0]

    # Extract host IP (optional)
    host_ip = ""
    parts = port_str_no_protocol.split(":")
    if len(parts) == 3:  # IP:host_port:container_port
        host_ip = parts[0]
        host_port_str = f"{parts[1]}:{parts[2]}"
    elif len(parts) == 2:  # host_port:container_port
        host_port_str = port_str_no_protocol
    else:
        msg = f"Invalid port mapping format: {port_str}"
        raise ValidationError(msg)

    # Validate port mapping
    try:
        host_port, container_port = validate_port_mapping(host_port_str)
    except ValidationError as e:
        msg = f"Invalid port mapping: {port_str}. {e!s}"
        raise ValidationError(msg) from e

    # Build port mapping
    port_key = f"{container_port}/{protocol}"
    port_binding: dict[str, str] = {"HostPort": str(host_port)}
    if host_ip:
        port_binding["HostIp"] = host_ip

    return {port_key: [port_binding]}


def parse_volume_mapping(volume_str: str) -> dict[str, dict[str, str]]:
    """Parse volume mapping string to SDK format.

    Supports formats:
    - /host:/container
    - /host:/container:ro
    - /host:/container:rw

    Args:
        volume_str: Volume mapping string

    Returns:
        Volume mapping in SDK format: {"/host": {"bind": "/container", "mode": "ro"}}

    Raises:
        ValidationError: If volume mapping is invalid

    Example:
        >>> parse_volume_mapping("/host:/container")
        {"/host": {"bind": "/container", "mode": "rw"}}
        >>> parse_volume_mapping("/host:/container:ro")
        {"/host": {"bind": "/container", "mode": "ro"}}
    """
    if not volume_str:
        msg = "Volume mapping cannot be empty"
        raise ValidationError(msg)

    # Split by colon
    parts = volume_str.split(":")
    if len(parts) < 2:
        msg = f"Invalid volume mapping format: {volume_str}. Use format: /host:/container[:mode]"
        raise ValidationError(msg)

    host_path = parts[0]
    container_path = parts[1]
    mode = parts[2] if len(parts) > 2 else "rw"

    if not host_path or not container_path:
        msg = "Host path and container path cannot be empty"
        raise ValidationError(msg)

    if mode not in ("ro", "rw"):
        msg = f"Invalid volume mode: {mode}. Use 'ro' or 'rw'"
        raise ValidationError(msg)

    return {host_path: {"bind": container_path, "mode": mode}}


def parse_environment(env_list: list[str]) -> list[str]:
    """Parse environment variables to SDK format.

    Supports formats:
    - KEY=value
    - KEY (reads from environment)

    Args:
        env_list: List of environment variable strings

    Returns:
        List of environment variables in SDK format: ["KEY=value"]

    Example:
        >>> parse_environment(["KEY=value", "HOME"])
        ["KEY=value", "HOME=/home/user"]
    """
    result: list[str] = []

    for env_str in env_list:
        if "=" in env_str:
            # Direct assignment
            result.append(env_str)
        else:
            # Read from environment
            value = os.getenv(env_str, "")
            if value:
                result.append(f"{env_str}={value}")
            else:
                # Include even if not set (Docker behavior)
                result.append(env_str)

    return result


def parse_labels(label_list: list[str]) -> dict[str, str]:
    """Parse labels to SDK format.

    Supports format:
    - key=value

    Args:
        label_list: List of label strings

    Returns:
        Dictionary of labels: {"key": "value"}

    Raises:
        ValidationError: If label format is invalid

    Example:
        >>> parse_labels(["app=web", "env=prod"])
        {"app": "web", "env": "prod"}
    """
    result: dict[str, str] = {}

    for label_str in label_list:
        if "=" not in label_str:
            msg = f"Invalid label format: {label_str}. Use format: key=value"
            raise ValidationError(msg)

        key, value = label_str.split("=", 1)
        if not key:
            msg = "Label key cannot be empty"
            raise ValidationError(msg)

        result[key] = value

    return result


async def resolve_container_id(client: Any, container_ref: str) -> str:  # AsyncDockerClient type
    """Resolve container ID or name to full container ID.

    Args:
        client: AsyncDockerClient instance
        container_ref: Container ID (full or short) or name

    Returns:
        Full container ID

    Raises:
        NotFound: If container not found
    """
    from dockpycore.exceptions import NotFound

    # If it's already a full ID (64 chars), return it
    if len(container_ref) == 64:
        return container_ref

    # Try to find by name or short ID
    containers = await client.containers.list(all=True)

    # Check for exact name match
    for container in containers:
        if container.name == container_ref or container_ref in container.names:
            return container.id

        # Check for short ID match
        if container.id.startswith(container_ref):
            return container.id

    raise NotFound("container", container_ref)


def format_ports(ports: dict[str, list[dict[str, Any]]] | list[dict[str, Any]]) -> str:
    """Format port mappings for display.

    Args:
        ports: Port mappings from Container model (dict or list format)

    Returns:
        Formatted port string (e.g., "0.0.0.0:8080->80/tcp")

    Example:
        >>> format_ports({"80/tcp": [{"HostPort": "8080", "HostIp": "0.0.0.0"}]})
        "0.0.0.0:8080->80/tcp"
        >>> format_ports([{"PrivatePort": 80, "PublicPort": 8080, "Type": "tcp"}])
        "8080->80/tcp"
    """
    if not ports:
        return ""

    port_strings: list[str] = []

    # Handle dict format: {"80/tcp": [{"HostPort": "8080"}]}
    if isinstance(ports, dict):
        for container_port, bindings in ports.items():
            for binding in bindings:
                host_ip = binding.get("HostIp", "0.0.0.0")
                host_port = binding.get("HostPort", "")

                if host_port:
                    if host_ip == "0.0.0.0":
                        port_str = f"{host_port}->{container_port}"
                    else:
                        port_str = f"{host_ip}:{host_port}->{container_port}"
                    port_strings.append(port_str)
    # Handle list format: [{"PrivatePort": 80, "PublicPort": 8080, "Type": "tcp"}]
    elif isinstance(ports, list):
        for port_info in ports:
            private_port = port_info.get("PrivatePort", "")
            public_port = port_info.get("PublicPort", "")
            port_type = port_info.get("Type", "tcp")
            host_ip = port_info.get("IP", "0.0.0.0")

            if private_port and public_port:
                container_port = f"{private_port}/{port_type}"
                if host_ip == "0.0.0.0":
                    port_str = f"{public_port}->{container_port}"
                else:
                    port_str = f"{host_ip}:{public_port}->{container_port}"
                port_strings.append(port_str)

    return ", ".join(port_strings)


def format_command(cmd: list[str] | None) -> str:
    """Format command for display.

    Args:
        cmd: Command list or None

    Returns:
        Formatted command string

    Example:
        >>> format_command(["echo", "hello"])
        "echo hello"
    """
    if not cmd:
        return ""

    # Join command parts, escape spaces if needed
    return " ".join(str(part) for part in cmd)
