# Copyright 2025 BBDevs
# Licensed under the Apache License, Version 2.0

"""Image command Click entry points.

Author: A M (am@bbdevs.com)

Created At: 08 Nov 2025
"""

from __future__ import annotations

from typing import Any

import click

from dockpycli.commands.container.helpers import parse_labels
from dockpycli.commands.image.command import ImageCommand
from dockpycli.config import CLIConfig


__all__ = [
    "build_image",
    "image_history",
    "inspect_image",
    "list_images",
    "prune_images",
    "pull_image",
    "push_image",
    "remove_image",
    "search_images",
    "tag_image",
]


@click.command()
@click.option("--all", "-a", is_flag=True, help="Show all images (default: hide intermediate images)")
@click.option("--filter", "-f", multiple=True, help="Filter images (e.g., reference=nginx)")
@click.option("--no-trunc", is_flag=True, help="Don't truncate output")
@click.pass_context
def list_images(
    ctx: click.Context,
    all: bool,
    filter: tuple[str, ...],
    no_trunc: bool,
) -> None:
    """List images."""
    config: CLIConfig = ctx.obj.config
    cmd = ImageCommand(config)

    # Parse filters
    filters: dict[str, Any] | None = None
    if filter:
        filters = {}
        for f in filter:
            if "=" in f:
                key, value = f.split("=", 1)
                if key not in filters:
                    filters[key] = []
                filters[key].append(value)

    cmd.run(cmd._list_images_impl(all, filters, no_trunc))


@click.command()
@click.argument("image", required=True)
@click.option("--platform", help="Set platform (e.g., linux/amd64)")
@click.option("--quiet", "-q", is_flag=True, help="Suppress output")
@click.pass_context
def pull_image(
    ctx: click.Context,
    image: str,
    platform: str | None,
    quiet: bool,
) -> None:
    """Pull an image from a registry."""
    config: CLIConfig = ctx.obj.config
    cmd = ImageCommand(config)
    cmd.run(cmd._pull_image_impl(image, platform, quiet))


@click.command()
@click.argument("image", required=True)
@click.option("--quiet", "-q", is_flag=True, help="Suppress output")
@click.pass_context
def push_image(
    ctx: click.Context,
    image: str,
    quiet: bool,
) -> None:
    """Push an image to a registry."""
    config: CLIConfig = ctx.obj.config
    cmd = ImageCommand(config)
    cmd.run(cmd._push_image_impl(image, quiet))


@click.command()
@click.argument("source", required=True)
@click.argument("target", required=True)
@click.pass_context
def tag_image(
    ctx: click.Context,
    source: str,
    target: str,
) -> None:
    """Tag an image."""
    config: CLIConfig = ctx.obj.config
    cmd = ImageCommand(config)
    cmd.run(cmd._tag_image_impl(source, target))


@click.command()
@click.argument("images", required=True, nargs=-1)
@click.option("--force", "-f", is_flag=True, help="Force removal")
@click.option("--no-prune", is_flag=True, help="Do not delete untagged parents")
@click.pass_context
def remove_image(
    ctx: click.Context,
    images: tuple[str, ...],
    force: bool,
    no_prune: bool,
) -> None:
    """Remove one or more images."""
    config: CLIConfig = ctx.obj.config
    cmd = ImageCommand(config)
    cmd.run(cmd._remove_image_impl(list(images), force, no_prune))


@click.command()
@click.argument("image", required=True)
@click.option("--size", "-s", is_flag=True, help="Display total file sizes")
@click.option(
    "--format",
    "format_template",
    help="Format output using a template (JSON/YAML/Plain)",
)
@click.pass_context
def inspect_image(
    ctx: click.Context,
    image: str,
    size: bool,
    format_template: str | None,
) -> None:
    """Inspect an image."""
    config: CLIConfig = ctx.obj.config
    cmd = ImageCommand(config)
    cmd.run(cmd._inspect_image_impl(image, size, format_template))


@click.command()
@click.argument("image", required=True)
@click.option(
    "--format",
    "format_template",
    help="Format output using a template (JSON/YAML/Plain)",
)
@click.option("--human", "-H", is_flag=True, help="Print sizes and dates in human readable format")
@click.option("--no-trunc", is_flag=True, help="Don't truncate output")
@click.option("--quiet", "-q", is_flag=True, help="Only show image IDs")
@click.pass_context
def image_history(
    ctx: click.Context,
    image: str,
    format_template: str | None,
    human: bool,
    no_trunc: bool,
    quiet: bool,
) -> None:
    """Show image history."""
    config: CLIConfig = ctx.obj.config
    cmd = ImageCommand(config)
    cmd.run(cmd._image_history_impl(image, format_template, human, no_trunc, quiet))


@click.command()
@click.argument("path", required=True, type=click.Path(exists=True, file_okay=False, dir_okay=True))
@click.option("--file", "-f", help="Name of the Dockerfile")
@click.option("--tag", "-t", multiple=True, help="Name and optionally a tag in the 'name:tag' format")
@click.option("--build-arg", multiple=True, help="Set build-time variables")
@click.option("--label", multiple=True, help="Set metadata for an image")
@click.option("--platform", help="Set platform (e.g., linux/amd64)")
@click.option("--quiet", "-q", is_flag=True, help="Suppress build output")
@click.option("--progress", type=click.Choice(["auto", "plain", "tty"]), help="Set type of progress output")
@click.pass_context
def build_image(
    ctx: click.Context,
    path: str,
    file: str | None,
    tag: tuple[str, ...],
    build_arg: tuple[str, ...],
    label: tuple[str, ...],
    platform: str | None,
    quiet: bool,
    progress: str | None,
) -> None:
    """Build an image from a Dockerfile."""
    config: CLIConfig = ctx.obj.config
    cmd = ImageCommand(config)

    # Parse build args
    build_args: dict[str, str] | None = None
    if build_arg:
        build_args = {}
        for arg in build_arg:
            if "=" in arg:
                key, value = arg.split("=", 1)
                build_args[key] = value

    # Parse labels
    labels: dict[str, str] | None = None
    if label:
        labels = parse_labels(list(label))

    # Get first tag (for now, support multiple tags later)
    tag_str = tag[0] if tag else None

    cmd.run(cmd._build_image_impl(path, file, tag_str, build_args, labels, platform, quiet, progress))


@click.command()
@click.option("--all", "-a", is_flag=True, help="Remove all unused images, not just dangling ones")
@click.option("--filter", multiple=True, help="Filter images (e.g., until=24h)")
@click.option("--force", "-f", is_flag=True, help="Do not prompt for confirmation")
@click.pass_context
def prune_images(
    ctx: click.Context,
    all: bool,
    filter: tuple[str, ...],
    force: bool,
) -> None:
    """Remove unused images."""
    config: CLIConfig = ctx.obj.config
    cmd = ImageCommand(config)

    # Parse filters
    filters: dict[str, Any] | None = None
    if filter:
        filters = {}
        for f in filter:
            if "=" in f:
                key, value = f.split("=", 1)
                if key not in filters:
                    filters[key] = []
                filters[key].append(value)

    cmd.run(cmd._prune_images_impl(all, filters, force))


@click.command()
@click.argument("term", required=True)
@click.option("--limit", type=int, help="Limit number of results")
@click.option(
    "--format",
    "format_template",
    help="Format output using a template (JSON/YAML/Plain)",
)
@click.pass_context
def search_images(
    ctx: click.Context,
    term: str,
    limit: int | None,
    format_template: str | None,
) -> None:
    """Search Docker Hub for images."""
    config: CLIConfig = ctx.obj.config
    cmd = ImageCommand(config)
    cmd.run(cmd._search_images_impl(term, limit, format_template))
