# Copyright 2025 BBDevs
# Licensed under the Apache License, Version 2.0

"""Image command handler - business logic only.

Author: A M (am@bbdevs.com)

Created At: 08 Nov 2025
"""

from __future__ import annotations

from typing import Any

from rich.table import Table

from dockpycli.commands.base import BaseCommand
from dockpycli.commands.image.helpers import parse_image_reference, resolve_image_id
from dockpycli.output.formatters import JSONFormatter
from dockpycli.output.image_formatters import format_image_history_table, format_image_plain, format_image_table
from dockpycore.exceptions import DockerSDKError, NotFound


__all__ = ["ImageCommand"]


class ImageCommand(BaseCommand):
    """Image command handler - business logic only."""

    async def _list_images_impl(
        self,
        all_images: bool,
        filters: dict[str, Any] | None,
        no_trunc: bool,
    ) -> None:
        """List images implementation."""
        images = await self.client.images.list(all=all_images, filters=filters)

        if self.config.output_format == "table":
            table = format_image_table(images, no_trunc=no_trunc)
            self.console.print(table)
        elif self.config.output_format == "plain":
            plain_output = format_image_plain(images)
            self.console.print(plain_output)
        else:
            # JSON/YAML format
            data = [image.to_dict() for image in images]
            self.formatter.format(data)

    async def _pull_image_impl(self, image_ref: str, platform: str | None, quiet: bool) -> None:
        """Pull image implementation."""
        # Parse image reference
        repository, tag = parse_image_reference(image_ref)

        try:
            if not quiet:
                self.console.print(f"[green]Pulling image:[/green] {repository}:{tag}")
            image = await self.client.images.pull(repository, tag=tag, quiet=quiet)
            if not quiet:
                self.console.print(f"[green]Image pulled:[/green] {image.id[:12]}")
        except DockerSDKError as e:
            self.console.print(f"[red]Error pulling image:[/red] {e}")
            raise

    async def _push_image_impl(self, image_ref: str, quiet: bool) -> None:
        """Push image implementation."""
        # Parse image reference
        repository, tag = parse_image_reference(image_ref)

        try:
            if not quiet:
                self.console.print(f"[green]Pushing image:[/green] {repository}:{tag}")
            await self.client.images.push(repository, tag=tag, quiet=quiet)
            if not quiet:
                self.console.print(f"[green]Image pushed:[/green] {repository}:{tag}")
        except DockerSDKError as e:
            self.console.print(f"[red]Error pushing image:[/red] {e}")
            raise

    async def _tag_image_impl(self, source: str, target: str) -> None:
        """Tag image implementation."""
        # Parse target reference
        repository, tag = parse_image_reference(target)

        try:
            await self.client.images.tag(source, repository, tag=tag)
            self.console.print(f"[green]Image tagged:[/green] {source} -> {repository}:{tag}")
        except DockerSDKError as e:
            self.console.print(f"[red]Error tagging image:[/red] {e}")
            raise

    async def _remove_image_impl(self, image_refs: list[str], force: bool, no_prune: bool) -> None:
        """Remove image(s) implementation."""
        for image_ref in image_refs:
            try:
                image_id = await resolve_image_id(self.client, image_ref)
                await self.client.images.remove(image_id, force=force, noprune=no_prune)
                self.console.print(f"[green]Image removed:[/green] {image_id[:12]}")
            except NotFound:
                self.console.print(f"[yellow]Image not found:[/yellow] {image_ref}")
            except DockerSDKError as e:
                self.console.print(f"[red]Error removing image {image_ref}:[/red] {e}")

    async def _inspect_image_impl(self, image_ref: str, size: bool, format_template: str | None) -> None:
        """Inspect image implementation."""
        try:
            image_id = await resolve_image_id(self.client, image_ref)
            image = await self.client.images.inspect_detailed(image_id)

            # Convert to dict for formatting
            inspect_dict = {
                "Id": image.id,
                "RepoTags": image.repo_tags,
                "RepoDigests": image.repo_digests,
                "Parent": image.parent,
                "Created": image.created,
                "Size": image.size,
                "VirtualSize": image.virtual_size,
                "Architecture": image.architecture,
                "Os": image.os,
                "DockerVersion": image.docker_version,
            }

            # Add size if requested
            if size:
                inspect_dict["SizeRw"] = image.size
                inspect_dict["SizeRootFs"] = image.virtual_size

            # If format template is provided, use it (basic implementation)
            if format_template:
                formatter = JSONFormatter(self.config)
            else:
                formatter = self.formatter

            formatter.format(inspect_dict)
        except NotFound:
            self.console.print(f"[yellow]Image not found:[/yellow] {image_ref}")

    async def _image_history_impl(
        self,
        image_ref: str,
        format_template: str | None,
        human: bool,
        no_trunc: bool,
        quiet: bool,
    ) -> None:
        """Get image history implementation."""
        try:
            image_id = await resolve_image_id(self.client, image_ref)
            history = await self.client.images.history(image_id)

            if self.config.output_format == "table" or format_template is None:
                table = format_image_history_table(history, no_trunc=no_trunc, quiet=quiet)
                self.console.print(table)
            else:
                # JSON/YAML/Plain format
                self.formatter.format(history)
        except NotFound:
            self.console.print(f"[yellow]Image not found:[/yellow] {image_ref}")

    async def _build_image_impl(
        self,
        path: str,
        file: str | None,
        tag: str | None,
        build_args: dict[str, str] | None,
        labels: dict[str, str] | None,
        platform: str | None,
        quiet: bool,
        progress: str | None,
    ) -> None:
        """Build image implementation."""
        try:
            if not quiet:
                self.console.print(f"[green]Building image from:[/green] {path}")

            # Build image with streaming
            async for line in self.client.images.build(
                path=path,
                dockerfile=file,
                tag=tag,
                buildargs=build_args,
                labels=labels,
                platform=platform,
                quiet=quiet,
            ):
                if not quiet and line:
                    self.console.print(line, end="")

            if not quiet:
                self.console.print("[green]Image built successfully[/green]")
        except DockerSDKError as e:
            self.console.print(f"[red]Error building image:[/red] {e}")
            raise

    async def _prune_images_impl(self, all_images: bool, filters: dict[str, Any] | None, force: bool) -> None:
        """Prune images implementation."""
        try:
            result = await self.client.images.prune(all=all_images, filters=filters, force=force)
            self.console.print(f"[green]Pruned images:[/green] {result.get('ImagesDeleted', 0)}")
            self.console.print(f"[green]Space reclaimed:[/green] {result.get('SpaceReclaimed', 0)} bytes")
        except DockerSDKError as e:
            self.console.print(f"[red]Error pruning images:[/red] {e}")
            raise

    async def _search_images_impl(self, term: str, limit: int | None, format_template: str | None) -> None:
        """Search images implementation."""
        try:
            results = await self.client.images.search(term, limit=limit)

            if self.config.output_format == "table" or format_template is None:
                # Create table for search results
                table = Table(
                    show_header=True,
                    header_style="bold magenta",
                    box=None,
                    padding=(0, 1),
                )
                table.add_column("NAME", style="blue")
                table.add_column("DESCRIPTION", style="white")
                table.add_column("STARS", style="yellow")
                table.add_column("OFFICIAL", style="green")
                table.add_column("AUTOMATED", style="cyan")

                for result in results:
                    name = result.get("name", "")
                    description = result.get("description", "")
                    stars = result.get("star_count", 0)
                    official = "[OK]" if result.get("is_official", False) else ""
                    automated = "[OK]" if result.get("is_automated", False) else ""

                    table.add_row(name, description, str(stars), official, automated)

                self.console.print(table)
            else:
                # JSON/YAML/Plain format
                self.formatter.format(results)
        except DockerSDKError as e:
            self.console.print(f"[red]Error searching images:[/red] {e}")
            raise
