# Copyright 2025 BBDevs
# Licensed under the Apache License, Version 2.0

"""Helper functions for image operations.

Author: A M (am@bbdevs.com)

Created At: 08 Nov 2025
"""

from __future__ import annotations

from typing import Any

from dockpycore.exceptions import NotFound


__all__ = [
    "format_image_size",
    "format_image_tags",
    "parse_image_reference",
    "resolve_image_id",
]


def parse_image_reference(image_ref: str) -> tuple[str, str]:
    """Parse image reference into repository and tag.

    Supports formats:
    - nginx:latest
    - nginx (defaults to latest)
    - myregistry.com/nginx:1.21
    - myregistry.com/nginx (defaults to latest)

    Args:
        image_ref: Image reference string

    Returns:
        Tuple of (repository, tag)

    Example:
        >>> parse_image_reference("nginx:latest")
        ("nginx", "latest")
        >>> parse_image_reference("nginx")
        ("nginx", "latest")
        >>> parse_image_reference("myregistry.com/nginx:1.21")
        ("myregistry.com/nginx", "1.21")
    """
    if ":" in image_ref:
        parts = image_ref.rsplit(":", 1)
        if len(parts) == 2:
            return (parts[0], parts[1])
    return (image_ref, "latest")


async def resolve_image_id(client: Any, image_ref: str) -> str:  # AsyncDockerClient type
    """Resolve image ID or name/tag to full image ID.

    Args:
        client: AsyncDockerClient instance
        image_ref: Image ID (full or short), name, or name:tag

    Returns:
        Full image ID

    Raises:
        NotFound: If image not found
    """
    # If it's already a full ID (64 chars), return it
    if len(image_ref) == 64:
        return image_ref

    # Try to find by name/tag or short ID
    images = await client.images.list(all=True)

    # Parse repository and tag
    repository, tag = parse_image_reference(image_ref)
    image_name_tag = f"{repository}:{tag}"

    # Check for exact name:tag match
    for image in images:
        for repo_tag in image.repo_tags:
            if repo_tag == image_name_tag:
                return image.id

        # Check for short ID match
        if image.id.startswith(image_ref):
            return image.id

        # Check for repository match (any tag)
        for repo_tag in image.repo_tags:
            if repo_tag.startswith(f"{repository}:"):
                return image.id

    raise NotFound("image", image_ref)


def format_image_size(size_bytes: int) -> str:
    """Format image size in bytes to human-readable format.

    Args:
        size_bytes: Size in bytes

    Returns:
        Formatted size string (e.g., "1.5 GB", "500 MB")

    Example:
        >>> format_image_size(1024 * 1024 * 1024)
        "1.0 GB"
    """
    from dockpycli.utils.helpers import format_size

    return format_size(size_bytes)


def format_image_tags(repo_tags: list[str]) -> str:
    """Format repository tags for display.

    Args:
        repo_tags: List of repository tags

    Returns:
        Formatted tag string (e.g., "nginx:latest, nginx:1.21")

    Example:
        >>> format_image_tags(["nginx:latest", "nginx:1.21"])
        "nginx:latest, nginx:1.21"
        >>> format_image_tags([])
        "<none>"
    """
    if not repo_tags:
        return "<none>"
    return ", ".join(repo_tags)
