# Copyright 2025 BBDevs
# Licensed under the Apache License, Version 2.0

"""System command handler - business logic only.

Author: A M (am@bbdevs.com)

Created At: 21 Nov 2025
"""

from __future__ import annotations

from dockpycli.commands.base import BaseCommand
from dockpycli.output.system_formatters import (
    format_system_df_plain,
    format_system_df_table,
    format_system_info_plain,
    format_system_info_table,
    format_system_version_plain,
    format_system_version_table,
)


__all__ = ["SystemCommand"]


class SystemCommand(BaseCommand):
    """System command handler - business logic only."""

    async def _info_impl(self) -> None:
        """Get system information implementation."""
        info = await self.client.info()

        if self.config.output_format == "table":
            table = format_system_info_table(info)
            self.console.print(table)
        elif self.config.output_format == "plain":
            plain_output = format_system_info_plain(info)
            self.console.print(plain_output)
        else:
            self.formatter.format(info)

    async def _df_impl(self) -> None:
        """Get disk usage implementation."""
        df_data = await self.client.df()

        if self.config.output_format == "table":
            table = format_system_df_table(df_data)
            self.console.print(table)
        elif self.config.output_format == "plain":
            plain_output = format_system_df_plain(df_data)
            self.console.print(plain_output)
        else:
            self.formatter.format(df_data)

    async def _version_impl(self) -> None:
        """Get version information implementation."""
        version = await self.client.version()

        if self.config.output_format == "table":
            table = format_system_version_table(version)
            self.console.print(table)
        elif self.config.output_format == "plain":
            plain_output = format_system_version_plain(version)
            self.console.print(plain_output)
        else:
            self.formatter.format(version)

    async def _ping_impl(self) -> None:
        """Test Docker daemon connection implementation."""
        is_accessible = await self.client.ping()

        if is_accessible:
            if self.config.output_format in ("json", "yaml"):
                self.formatter.format({"status": "ok", "message": "Docker daemon is accessible"})
            else:
                self.console.print("[green]OK[/green] - Docker daemon is accessible")
        elif self.config.output_format in ("json", "yaml"):
            self.formatter.format({"status": "error", "message": "Docker daemon is not accessible"})
        else:
            self.console.print("[red]ERROR[/red] - Docker daemon is not accessible")
